# FindAlembic.cmake
# -----------------

# This module defines:
#
# ALEMBIC_FOUND - Whether Alembic has been found.
#
# If ALEMBIC_FOUND is TRUE, also sets the following:
# ALEMBIC_INCLUDE_DIRS - The Alembic include directories
# ALEMBIC_LIBRARIES - List containing full path to each of the libraries needed
#   to use Alembic.
include(FindPackageHandleStandardArgs)

set(ALEMBIC_LIBRARY_SUFFIX "" CACHE STRING
    "Alembic: optional suffix for library names.")

# Legacy Alembic library names.
set(_ALEMBIC_LEGACY_COMPONENTS
    AbcGeom
    Abc
    AbcCoreFactory
    AbcCoreHDF5
    AbcCoreOgawa
    AbcCoreAbstract
    Ogawa
    Util
    AbcCollection
    AbcMaterial)

# Variables set by Alembic's dependencies (specifically, their CMake modules).
# find_package(...) for all these libraries must have been called prior to
# invoking Alembic's CMake module.
set(_ALEMBIC_REQUIRED_THIRDPARTY_VARS
    Boost_INCLUDE_DIRS
    HDF5_HL_LIBRARIES
    HDF5_C_LIBRARIES
    HDF5_C_LIBRARIES
    OPENEXR_LIBRARIES
    ZLIB_LIBRARIES)

find_path(ALEMBIC_INCLUDE_DIR
    NAMES Alembic/Abc/All.h
    HINTS "${ALEMBIC_ROOT}"
    PATH_SUFFIXES include)

set(ALEMBIC_INCLUDE_DIRS "${ALEMBIC_INCLUDE_DIR}")

# First, try using the modern "Alembic" library name.
find_library(ALEMBIC_LIBRARY
    NAMES "Alembic${ALEMBIC_LIBRARY_SUFFIX}" "Alembic"
    HINTS "${ALEMBIC_ROOT}"
    PATH_SUFFIXES lib lib64)

if (ALEMBIC_LIBRARY)
    set(_alembicRequiredVars ALEMBIC_LIBRARY)
    set(ALEMBIC_LIBRARIES "${ALEMBIC_LIBRARY}")
else ()
    # Can't find Alembic using the single-library variant; try the legacy style.
    set(_alembicRequiredVars)
    set(ALEMBIC_LIBRARIES)
    foreach (_alembic_component ${_ALEMBIC_LEGACY_COMPONENTS})
        string(TOUPPER "${_alembic_component}" _alembic_uppercase_component)
        find_library("ALEMBIC_${_alembic_uppercase_component}_LIBRARY"
            NAMES
                "Alembic${_alembic_component}${ALEMBIC_LIBRARY_SUFFIX}"
                "Alembic${_alembic_component}"
            HINTS "${ALEMBIC_ROOT}"
            PATH_SUFFIXES lib lib64)
        list(APPEND _alembicRequiredVars
             "ALEMBIC_${_alembic_uppercase_component}_LIBRARY")
         list(APPEND ALEMBIC_LIBRARIES
             "${ALEMBIC_${_alembic_uppercase_component}_LIBRARY}")
    endforeach ()
endif ()

find_package_handle_standard_args(Alembic DEFAULT_MSG
    ALEMBIC_INCLUDE_DIR
    ${_alembicRequiredVars}
    ${_ALEMBIC_REQUIRED_THIRDPARTY_VARS}
)

if (ALEMBIC_FOUND)
    if (ALEMBIC_LIBRARY AND NOT TARGET Alembic::Alembic)
        add_library("Alembic::Alembic" UNKNOWN IMPORTED)
        set_property(TARGET "Alembic::Alembic" PROPERTY
            IMPORTED_LOCATION
                "${ALEMBIC_LIBRARY}")
        set_property(TARGET "Alembic::Alembic" APPEND PROPERTY
            INTERFACE_INCLUDE_DIRECTORIES
                ${ALEMBIC_INCLUDE_DIRS}
                ${Boost_INCLUDE_DIRS})
        set_property(TARGET "Alembic::Alembic" APPEND PROPERTY
            INTERFACE_LINK_LIBRARIES
                ${HDF5_HL_LIBRARIES}
                ${HDF5_C_LIBRARIES}
                ${OPENEXR_LIBRARIES}
                ${ZLIB_LIBRARIES})

    elseif (NOT TARGET Alembic::Alembic)
        # Legacy Alembic libraries. Add IMPORTED targets for each of the
        # components, and a convenience "Alembic::Alembic" INTERFACE target
        # that amalgamates them.
        add_library(Alembic::Alembic INTERFACE IMPORTED)
        foreach (_alembic_component ${_ALEMBIC_LEGACY_COMPONENTS})
            string(TOUPPER "${_alembic_component}" _alembic_uppercase_component)

            add_library ("Alembic::${_alembic_component}" UNKNOWN IMPORTED)
            set_property(TARGET "Alembic::${_alembic_component}" PROPERTY
                IMPORTED_LOCATION
                    "${ALEMBIC_${_alembic_uppercase_component}_LIBRARY}")
            set_property(TARGET "Alembic::${_alembic_component}" APPEND PROPERTY
                INTERFACE_INCLUDE_DIRECTORIES
                    ${ALEMBIC_INCLUDE_DIRS}
                    ${Boost_INCLUDE_DIRS})
            set_property(TARGET "Alembic::${_alembic_component}" APPEND PROPERTY
                INTERFACE_LINK_LIBRARIES
                    ${HDF5_HL_LIBRARIES}
                    ${HDF5_C_LIBRARIES}
                    ${OPENEXR_LIBRARIES}
                    ${ZLIB_LIBRARIES})

            set_property(TARGET Alembic::Alembic APPEND PROPERTY
                INTERFACE_LINK_LIBRARIES "Alembic::${_alembic_component}")
        endforeach ()
    endif ()
endif ()
