# Copyright (c) 2017 The Foundry Visionmongers Ltd. All Rights Reserved.


import hashlib
import logging

from PyQt5 import (
    QtCore,
    QtGui,
    QtWidgets,
)

import PyFnAttribute as FnAttribute
from Nodes3DAPI import ScenegraphManager
from UI4.App.KeyboardShortcutManager import BindActions

log = logging.getLogger("ExampleViewer.ManipulatorMenu")

class NoManipulatorAction(QtWidgets.QAction):
    """
    A QAction that deactivates all the manipulators.
    """
    def __init__(self, parent, viewports):
        QtWidgets.QAction.__init__(self, 'No Manipulators', parent)

        self.__viewports = viewports

        self.setObjectName("noManipulatorAction")
        self.setShortcut("Esc")

        self.triggered.connect(self.__on_triggered)

    def __on_triggered(self):
        """
        Slot that is called when the action has been triggered.
        """
        for viewport in self.__viewports:
            viewport.deactivateAllManipulators(True)
            viewport.setDirty(True)


class OrientationAction(QtWidgets.QAction):
    """
    A QAction that sets the manipulator orientation on the Viewport.
    """
    def __init__(self, text, parent, viewports, value, checked=False):
        QtWidgets.QAction.__init__(self, text, parent)

        self.__viewports = viewports
        self.__value = FnAttribute.IntAttribute(value)
        self.setCheckable(True)
        self.setObjectName(text + "OrientationAction")

        # See if this should be checked by checking the ManipulatorOrientation
        # on one of the Viewports
        checkedAttr = self.__viewports[0].getOptionByName("ManipulatorOrientation")
        if checkedAttr:
            if checkedAttr.getValue() == value:
                self.setChecked(True)
        elif value == 0:
            self.setChecked(True)

        self.triggered.connect(self.__on_triggered)

    def __on_triggered(self):
        """
        Slot that is called when the action has been triggered.
        """
        for viewport in self.__viewports:
            viewport.setOptionByName("ManipulatorOrientation", self.__value)
            viewport.setDirty(True)

class OrientationActionGroup(QtWidgets.QActionGroup):
    """
    A QActionGroup with actions that select the manipulator orientation.
    """
    def __init__(self, parent, viewports):
        QtWidgets.QActionGroup.__init__(self, parent, exclusive=True)

        self.__viewports = viewports
        self.setObjectName("OrientationActionMenu")

        self.__objectAction = OrientationAction("Object", self, self.__viewports, 0, True)
        self.__worldAction = OrientationAction("World", self, self.__viewports, 1)
        self.__viewAction = OrientationAction("View", self, self.__viewports, 2)

        self.addAction(self.__objectAction)
        self.addAction(self.__worldAction)
        self.addAction(self.__viewAction)

class ManipulatorMenu(QtWidgets.QMenu):
    """
    A QMenu that controlls the active Manipulators on a Viewport.
    """
    # Indicates what shortcuts have been already registered.
    __registeredShortcuts = set()

    @classmethod
    def TriggerManipulatorAction(cls, tab, shortcut):
        """
        Triggers the manipulator action with the given shorcut for the given
        tab.

        @type tab: C{ExampleViewerTab.ExampleViewerTab}
        @type shortcut: C{str}
        @param tab: The tab for which to trigger the manipulator action.
        @param shortcut: The keyboard shortcut to trigger.
        """
        menu = tab.getManipulatorMenu()
        menu.triggerManipulatorActionByShortcut(shortcut)

    def __init__(self, name, parent, viewerDelegate, viewportWidgets):
        QtWidgets.QMenu.__init__(self, name, parent)

        self.__tab = parent
        self.__viewerDelegate = viewerDelegate
        self.__viewportWidgets = viewportWidgets
        self.__viewports = [vw.getViewport() for vw in viewportWidgets]

        # Table that maps keyboard shorcuts and QActions for this menu.
        self.__shortcutTable = {}

        # Get all the viewports for this viewer delegate, required by the
        # Manipulator Orientation Actions
        allViewports = []
        for i in range(viewerDelegate.getNumberOfViewports()):
            allViewports.append(viewerDelegate.getViewportByIndex(i))

        # No manipulator action
        noManipulatorAction = NoManipulatorAction(self, self.__viewports)
        self.addAction(noManipulatorAction)
        noManipulatorAction.triggered.connect(self.__uncheckManipulators)
        self.__registerManipulatorAction(noManipulatorAction, "Esc")

        self.addSeparator()

        # Initializes the Manipulator Orientation Actions
        orientationActionGroup = OrientationActionGroup(
            self, allViewports)
        for action in orientationActionGroup.actions():
            self.addAction(action)

        # Manipulator actions. These actions are added or removed
        # based on the selected items in the scene.
        self._manipulatorActions = []

        # Initialize the manipulators as of the current selection.
        self.__addCompatibleManipulators()

        # Bind the keyboard actions to the tab instance
        BindActions(self.__tab.__class__.getShortcutsContextName(),
                    self.__tab,
                    self)

    def __registerManipulatorAction(self, action, shortcut):
        # Link the shortcut with the given action.
        self.__shortcutTable[shortcut] = action

        # If the shortcut has not been registered with the tab's class,
        # register it now.
        if shortcut not in ManipulatorMenu.__registeredShortcuts:
            ManipulatorMenu.__registeredShortcuts.add(shortcut)

            actionID = hashlib.md5(bytes(action.text(), 'UTF-8')).hexdigest()
            actionName = "Manipulator Shortcut '" + shortcut + "'"

            def _shortcutCallback(tab, shortcut=shortcut):
                ManipulatorMenu.TriggerManipulatorAction(tab, shortcut)

            self.__tab.__class__.registerKeyboardShortcut(
                actionID, actionName, shortcut, _shortcutCallback)

    def __isManipulatorActive(self, pluginName):
        """
        Returns True if the Manipulator with this plugin name has been
        activated in any Viewport.
        """
        for viewport in self.__viewports:
            for i in range(viewport.getNumberOfActiveManipulators()):
                if viewport.getActiveManipulatorPluginName(i) == pluginName:
                    return True
        return False

    def __addCompatibleManipulators(self):
        """
        Adds compatible manipulators and their shortcuts to the menu.
        """
        # Clear current manipulator actions
        for action in self._manipulatorActions:
            self.removeAction(action)

        selectedLocations = ScenegraphManager.getActiveScenegraph().getSelectedLocations()
        manipsAttr = self.__viewerDelegate.getCompatibleManipulatorsInfo(
            selectedLocations)

        # manipNames = ['Translate','Rotate','Scale']
        actionGroups = {}

        # Assemble all registered manipulators into groups
        for i in range(manipsAttr.getNumberOfChildren()):
            manipulator = manipsAttr.getChildByIndex(i)
            pluginName = manipsAttr.getChildName(i)
            manipulatorName = manipulator.getChildByName("name").getValue(pluginName, False)
            shortcutKey = manipulator.getChildByName("shortcut").getValue("", False)
            group = manipulator.getChildByName("group").getValue("", False)
            technology = manipulator.getChildByName("technology").getValue("", False)

            # Create an action to select the manipulator of the current class
            actionText = "%s\t%s" % (manipulatorName, shortcutKey)
            action = QtWidgets.QAction(actionText, self)
            action.setObjectName(pluginName.replace(" ", "") + "Action")
            action.setCheckable(True)
            action.setData(pluginName)
            action.setChecked(self.__isManipulatorActive(pluginName))
            action.setEnabled(True)
            self._manipulatorActions.append(action)

            action.triggered.connect(self.__on_action_triggered)

            if group not in actionGroups:
                actionGroups[group] = []

            actionGroups[group].append(action)

            # Register the action with the keyboard shortcut.
            self.__registerManipulatorAction(action, shortcutKey)

        # Build menu from action groups
        for group in actionGroups:
            action = QtWidgets.QAction(group, self)
            action.setSeparator(True)
            self._manipulatorActions.append(action)
            self.addAction(action)
            self.addActions(actionGroups[group])

    def __uncheckManipulators(self):
        """
        Unchecks all the manipulators.
        """
        for action in self._manipulatorActions:
            action.setChecked(False)

    def __on_action_triggered(self):
        """
        Slot that is called when an action for selecting a specific manipulator
        has been triggered.
        """
        if self.sender() is not None:
            if self.sender().data() is not None:
                manipulatorName = str(self.sender().data())
                if self.sender().isChecked():
                    selectedLocations = ScenegraphManager.getActiveScenegraph().getSelectedLocations()

                    for viewport in self.__viewports:
                        viewport.activateManipulator(
                            manipulatorName, selectedLocations, True)
                        viewport.setDirty(True)
                else:
                    for viewport in self.__viewports:
                        viewport.deactivateManipulator(
                            manipulatorName, True)
                        viewport.setDirty(True)
            else:
                log.error("The action that called "
                          "__on_action_triggered() does not have any "
                          "data attached to it.")
        else:
            log.error("__on_action_triggered() is expected to be "
                      "called from an action in the Manipulators menu.")

    def _on_selection_changed(self):
        """
        Manipulator menu updates when the selection changes in the scene.
        """
        self.__addCompatibleManipulators()

    def updateManipulators(self):
        """
        Updates the manipulators.
        """
        selectedLocations = \
            ScenegraphManager.getActiveScenegraph().getSelectedLocations()

        for action in self.__shortcutTable.values():
            # If the manipulator is not active, skip.
            if not action.isChecked():
                continue

            # If it is active, re-activate the manipulator in every viewport.
            manipulatorName = str(action.data())
            for viewport in self.__viewports:
                viewport.deactivateManipulator(manipulatorName, False)
                viewport.activateManipulator(manipulatorName,
                                             selectedLocations, False)
                viewport.setDirty(True)

    def triggerManipulatorActionByShortcut(self, shortcut):
        """
        Triggers the manipulator action given its keyboard shortcut.
        @type shortcut: c{str}
        @param shortcut: Keyboard shortcut to trigger.
        """
        action = self.__shortcutTable.get(shortcut, None)
        if action is not None:
            action.trigger()
