# Copyright (c) 2016 The Foundry Visionmongers Ltd. All Rights Reserved.


import logging

from PyQt5 import (
    QtCore,
    QtWidgets,
)

import PluginSystemAPI
import UI4.Widgets
import ViewerAPI_cmodule
from Katana import (
    FnAttribute,
    FnGeolibServices,
    RenderManager,
    UI4,
    ViewerAPI,
)
from UI4.Widgets import (
    LiveRenderViewerCameraButton,
    ViewerWorkingSetButton,
)
from .ManipulatorMenu import ManipulatorMenu

log = logging.getLogger("ExampleViewer")
RenderGlobals = RenderManager.RenderGlobals
_sceneGraphCameraName = "__location"  # Camera name for the scene graph camera

class ExampleViewerTab(UI4.Tabs.BaseViewerTab):
    """
    Class implementing a tab for hosting an example Viewer widget.
    """

    def __init__(self, parent, flags=0):
        """
        Initializes an instance of the class.

        @type parent: C{QtWidgets.QWidget} or C{None}
        @type flags: C{QtCore.Qt.WindowFlags}
        @param parent: The parent widget to own the new instance. Passed
            verbatim to the base class initializer.
        @param flags: The window flags to use in case no parent is given.
            Passed verbatim to the base class initializer.
        """
        UI4.Tabs.BaseViewerTab._checkGLVersion(4, 2)
        UI4.Tabs.BaseViewerTab.__init__(self, parent, flags)

        self.__selectedManipulators = []
        # The desired layers to be added to each viewport
        self.__initialLayers = ["CameraControlLayer",
                                "GridLayer",
                                "ExampleSceneLayer",
                                "SelectionLayer",
                                "SnappingLayer",
                                "GLManipulatorLayer",
                                "SnappingHUDLayer",
                                "HUDLayer",
                                "CameraGateLayer",
                                "AnnotationLayer",
                               ]

        # The terminal Op. It will be defined after L{applyTerminalOp} gets
        # called.
        self.__terminalOp = None
        self.__cameraName = "persp"

        # Holds the viewport names as well as their UI elements such as the
        # camera picker buttons or the active camera status icon.
        self.__viewportNamesAndButtons = {}

        # Instantiate a Viewer Delegate and the left and right Viewport widgets
        self.__viewerDelegate = self.addViewerDelegate("ExampleViewerDelegate")

        # Add the VDCs.
        self.__viewerDelegate.addComponent('AnnotationLayerVDC',
                                           'AnnotationLayerVDC')

        wireframeSelectionComponent = self.__viewerDelegate.addComponent(
            'WireframeSelectionComponent', 'WireframeSelectionComponent')
        wireframeSelectionComponent.setOptionByName(
            'Enabled', FnAttribute.IntAttribute(1))

        self.__viewportWidgetL = self.addViewport("ExampleViewport",
                                                  "left",
                                                  self.__viewerDelegate,
                                                  layers=self.__initialLayers)
        self.__viewportWidgetR = self.addViewport("ExampleViewport",
                                                  "right",
                                                  self.__viewerDelegate,
                                                  layers=self.__initialLayers)

        self.__viewports = [self.__viewportWidgetL, self.__viewportWidgetR]

        # A map of viewport names and the pending built-in cameras that will
        # be set in the next loop event.
        self.__pendingBuiltInCameraRequests = {}

        # Default background color
        backgroundColor = [0.0358, 0.0358, 0.0358, 1]
        backgroundColorAttr = FnAttribute.FloatAttribute(backgroundColor)

        for viewportWidget in  self.__viewports:
            # Set the background color
            viewportWidget.setOptionByName("Viewport.BackgroundColor",
                                           backgroundColorAttr)
            gridLayer = viewportWidget.getLayer('GridLayer')
            if gridLayer:
                gridLayer.setOptionByName("Viewport.BackgroundColor",
                                          backgroundColorAttr)

        # Set up the viewport container
        self.setLayout(QtWidgets.QVBoxLayout())
        viewportFrame = QtWidgets.QSplitter(self)
        viewportFrame.setOrientation(QtCore.Qt.Horizontal)
        viewportFrame.setSizePolicy(QtWidgets.QSizePolicy.Expanding,
                                    QtWidgets.QSizePolicy.Expanding)

        # Create a pane for each viewport with viewport-specific controls
        # e.g. for camera selection
        leftPane = self.createViewportPane(self.__viewportWidgetL)
        viewportFrame.addWidget(leftPane)

        rightPane = self.createViewportPane(self.__viewportWidgetR)
        viewportFrame.addWidget(rightPane)

        # Create a status bar
        self.__statusBar = QtWidgets.QStatusBar()
        self.__statusBar.setObjectName('statusBar')
        self.__statusBar.setSizeGripEnabled(False)
        self.__statusBar.showMessage('Ready.')

        self.__statusBarMessage = ""

        # Create the Manipulators menu.
        self.__manipulatorsMenu = ManipulatorMenu("Manipulators",
                                                  self, self.__viewerDelegate,
                                                  self.__viewports)
        self.__manipulatorsMenu.setObjectName("ManipulatorMenu")

        # Other menu
        self.__otherMenu = QtWidgets.QMenu("Other", self)
        self.snappingAction = self.__otherMenu.addAction("Snapping")
        self.snappingAction.setCheckable(True)
        self.snappingAction.setChecked(False)
        self.snappingAction.toggled.connect(self.on_snappingAction_toggled)

        self.__layersMenu = QtWidgets.QMenu("Layers", self)
        self.__layersMenu.setObjectName("LayersMenu")
        self.__layersMenu.aboutToShow.connect(self.on_layersMenu_aboutToShow)

        menuBar = QtWidgets.QMenuBar(self)
        menuBar.addMenu(self.__manipulatorsMenu)
        menuBar.addMenu(self.__layersMenu)
        menuBar.addMenu(self.__otherMenu)

        # Compose Main Layout
        self.layout().setMenuBar(menuBar)
        self.layout().addWidget(viewportFrame)
        self.layout().addWidget(self.__statusBar)

    def _cleanup(self):
        """
        Clears all the maps and dictionaries that may have cached a reference
        to the viewer delegate and the viewports.
        """
        del self.__manipulatorsMenu
        del self.__selectedManipulators[:]
        del self.__terminalOp
        for cpb, _, _ in self.__viewportNamesAndButtons.values():
            if cpb.getPopupWindow().isVisible():
                cpb.getPopupWindow().close()
        self.__viewportNamesAndButtons.clear()
        del self.__viewerDelegate
        del self.__viewportWidgetL
        del self.__viewportWidgetR
        del self.__viewports[:]

    def thaw(self):
        """
        Overridden to update manipulators when the tab is thawed.
        """
        UI4.Tabs.BaseViewerTab.thaw(self)

        # Update manipulators, which depend on the lock state of the node graph
        # that may have potentially changed while the tab was frozen.
        self.__manipulatorsMenu.updateManipulators()

    def applyTerminalOps(self, txn, op, viewerDelegate):
        '''
        This adds an Op that sets an attribute called 'viewerType' with value
        'ExampleViewer' on /root. This demonstrates how to set terminal Ops for
        a custom viewer.

        @type txn: C{PyFnGeolib.GeolibRuntime.Transaction}
        @type op: C{PyFnGeolib.GeolibRuntime.Op}
        @type viewerDelegate: C{ViewerAPI.ViewerDelegate}
        @rtype: C{PyFnGeolib.GeolibRuntime.Op}
        @param txn: A Geolib3 transaction.
        @param op: The last Op in the current Op-tree.
        @param viewerDelegate: The viewer delegate.
        @return: The last appended Op, or the given Op if no Ops are appended.
        '''
        terminalOp = txn.createOp()
        argsBuilder = FnGeolibServices.OpArgsBuilders.AttributeSet()
        argsBuilder.setCEL(['/root'])
        argsBuilder.setAttr('viewer.name',
            FnAttribute.StringAttribute('ExampleViewer'))
        argsBuilder.build()
        txn.setOpArgs(terminalOp, 'AttributeSet', argsBuilder.build())
        txn.setOpInputs(terminalOp, [op])

        return terminalOp

    def on_layersMenu_aboutToShow(self):
        """
        Populates the Layers menu, allowing the arbitrary insertion and removal
        of layers.
        """
        self.__layersMenu.clear()

        addLayerMenu = QtWidgets.QMenu("Add Layer", self)
        addLayerMenu.setObjectName("AddLayerMenu")

        # List registered layer plug-ins
        pluginVersion = ViewerAPI_cmodule.kViewportLayerPluginVersion
        pluginTypes = PluginSystemAPI.PluginCache.GetPluginNames(pluginVersion)
        for layerName in pluginTypes['ViewportLayerPlugin']:
            action = QtWidgets.QAction(layerName, addLayerMenu)
            action.setObjectName("Add%sAction" % layerName.replace(" ", ""))
            action.setData(layerName)
            addLayerMenu.addAction(action)
            # Clicking one of these actions will add the layer
            action.triggered.connect(self.on_addLayerAction_triggered)
        self.__layersMenu.addMenu(addLayerMenu)

        action = QtWidgets.QAction("Remove Layers", self.__layersMenu)
        action.setObjectName("RemoveLayerAction")
        action.setSeparator(True)
        self.__layersMenu.addAction(action)

        # List the currently added layers
        layerCount = self.__viewportWidgetL.getNumberOfLayers()
        for index in range(layerCount):
            layerName = self.__viewportWidgetL.getLayerName(index)
            action = QtWidgets.QAction("%s) %s" % (index, layerName), self.__layersMenu)
            action.setObjectName(layerName.replace(" ", "") + "Action")
            action.setData(index)
            # Clicking one of these actions will remove this layer
            action.triggered.connect(self.on_removeLayerAction_triggered)
            self.__layersMenu.addAction(action)

    def on_snappingAction_toggled(self):
        """
        Slot to enable/disable Snapping Layer
        """
        for viewport in self.__viewports:
            snappingLayer = viewport.getLayer('SnappingLayer')
            if snappingLayer is None:
                continue

            snappingLayer.setOptionByName(
                'SnappingLayer.Enabled',
                FnAttribute.IntAttribute(self.snappingAction.isChecked()))

    def on_removeLayerAction_triggered(self):
        """
        Slot to remove a layer from viewports.
        """
        if self.sender() is not None:
            if self.sender().data() is not None:
                index = int(self.sender().data())
                for viewport in self.__viewports:
                    viewport.removeLayerByIndex(index)
            else:
                log.error("The action that called "
                          "on_removeLayerAction_triggered() does not have any "
                          "data attached to it.")
        else:
            log.error("on_removeLayerAction_triggered() is expected to be "
                      "called from an action in the Layers menu.")

    def on_addLayerAction_triggered(self):
        """
        Slot to remove a layer from viewports.
        """
        if self.sender() is not None:
            if self.sender().data() is not None:
                layerType = str(self.sender().data())

                if self.__viewportWidgetL.getLayer(layerType) is not None:
                    log.warning("Attempting to add layer '%s' to viewports, "
                                "but it has already been added." % layerType)
                    return

                for viewport in self.__viewports:
                    viewport.addLayer(layerType, layerType)
            else:
                log.error("The action that called "
                          "on_removeLayerAction_triggered() does not have any "
                          "data attached to it.")
        else:
            log.error("on_removeLayerAction_triggered() is expected to be "
                      "called from an action in the Layers menu.")

    def on_cameraPicker_aboutToShow(self):
        """
        Slot that is called when the popup for the camera picker button is
        about to be opened.

        Built-in cameras will be populated the first time it is displayed.
        """
        cameraPickerButton = self.sender()

        # Early out if built-in types have already been defined.
        if cameraPickerButton.getBuiltIns():
            return

        viewportName = cameraPickerButton.property("viewportName")
        if hasattr(viewportName, "toString"):
            viewportName = viewportName.toString()

        viewportWidget = self.getViewportWidget(str(viewportName),
                                                self.__viewerDelegate)

        for index in range(viewportWidget.getNumberOfCameras()):
            camName = viewportWidget.getCameraName(index)
            # We do not want to add the meta camera as a built-in camera,
            # which can happen if we add the meta camera before the camera
            # list has been displayed for the first time.
            if camName == _sceneGraphCameraName:
                continue
            cameraPickerButton.addBuiltin(camName)

        # Manually populate now because the internal on_aboutToShow()
        # function gets called before this one.
        cameraPickerButton.populate()

    def createViewportPane(self, viewport):
        """
        Creates a viewport widget and a camera selection control.
        """
        # Add the camera picker button in the center
        cameraPickerButton = UI4.Widgets.CameraPickerButton(
            self, showIcon=True, showBuiltIns=True, abbreviateNames=False)
        cameraPickerButton.setProperty("viewportName", viewport.getName())
        cameraPickerButton.setDisplayMode(
            UI4.Widgets.CameraPickerButton.DISPLAY_CAMERAS
            | UI4.Widgets.CameraPickerButton.DISPLAY_LIGHTS)
        cameraPickerButton.setButtonType("textToolbar")
        cameraPickerButton.setText("persp")

        viewportName = viewport.getName()

        cameraPickerButton.aboutToShow.connect(
            self.on_cameraPicker_aboutToShow)
        cameraPickerButton.lookThrough.connect(
            lambda path: self.setCamera(viewportName, path))

        # Create the status icon for the active camera.
        lookThroughWarningIcon = QtWidgets.QLabel(self)
        lookThroughWarningIcon.setToolTip("Location does not exist")
        lookThroughWarningIcon.setPixmap(
            UI4.Util.IconManager.GetPixmap("Icons/Scenegraph/warning16.png"))
        lookThroughWarningIcon.setFixedSize(
            UI4.Util.IconManager.GetSize("Icons/Scenegraph/warning16.png"))
        lookThroughWarningIcon.hide()

        # Create the status icon for the active camera.
        lookThroughCookingIcon = UI4.Widgets.SimpleStopWidget(self, small=True)
        lookThroughCookingIcon.setToolTip("Waiting for location")
        lookThroughCookingIcon.setFixedHeight(16)
        lookThroughCookingIcon.setClickable(False)
        lookThroughCookingIcon.setActive(False)
        lookThroughCookingIcon.setVisible(False)

        # Bind the viewport name and the UI elements into the map for future
        # access.
        self.__viewportNamesAndButtons[viewport.getName()] = \
            (cameraPickerButton,
             lookThroughWarningIcon,
             lookThroughCookingIcon)

        # Create a viewer visibility button
        viewerVisibilityButton = ViewerWorkingSetButton(self)
        liveRenderFromCameraButton = LiveRenderViewerCameraButton(self, viewport)

        # Horizontal layout with the camera picker button and the status icon.
        hboxLayout = QtWidgets.QHBoxLayout()
        hboxLayout.addWidget(cameraPickerButton)
        hboxLayout.addWidget(lookThroughWarningIcon)
        hboxLayout.addWidget(lookThroughCookingIcon)
        hboxLayout.addStretch()
        hboxLayout.addWidget(liveRenderFromCameraButton)
        hboxLayout.addWidget(viewerVisibilityButton)

        # Layout for the pane.
        layout = QtWidgets.QVBoxLayout()
        layout.addWidget(viewport)
        layout.addLayout(hboxLayout)

        # Create the actual pane and set the layout.
        vbox = QtWidgets.QFrame(self)
        vbox.setLayout(layout)
        return vbox

    def setCamera(self, viewport, nameOrPath):
        """
        Tells the Viewport which camera to draw from.

        @type viewport: C{int}, C{str}, C{ViewportWidget} or
            C{ViewerAPI.Viewport}
        @type nameOrPath: C{str}
        @param viewport: The viewport whose camera location should be set. This
            can be the index of the viewport, the name, the Viewport instance,
            or the ViewportWidget instance.
        @param nameOrPath: The name of the built-in camera or the path to the
            camera location from which the given viewport should view the
            scene.
        @raise TypeError: If the type of the second argument is not C{str}.
        @raise ValueError: If the given viewport cannot be resolved as a valid
            viewport.
        """
        if not isinstance(nameOrPath, str):
            raise TypeError("Invalid argument 'nameOrPath' (expected str)")
        nameOrPath = str(nameOrPath)

        # Resolve the ViewportWidget instance from the viewport argument.
        viewportWidget = self.getViewportWidget(viewport,
                                                self.__viewerDelegate)

        if viewportWidget is None:
            raise ValueError("Unknown 'viewport': " + str(viewport))

        viewport = viewportWidget.getViewport()

        # Is this a location backed camera?
        if nameOrPath and nameOrPath[0] == '/':
            # We do not wish to allow interactions until the new look-through
            # location is ready.
            camera = viewport.getActiveCamera()
            if camera:
                camera.disableInteraction(True)

            # We're going to set the location as the look-through location.
            # When the location is ready, we'll receive a callback, and only
            # then we'll reset the active camera.
            self.__viewerDelegate.setLookThroughLocation(viewport, nameOrPath)
            return

        # Otherwise, this is considered a built-in camera, which can be set
        # immediately.
        self.__setBuiltInCamera(viewportWidget, nameOrPath)

    def __setBuiltInCamera(self, viewportWidget, cameraName):
        """
        Tells the Viewport which built-in camera to draw from.

        @type viewportWidget: C{ViewportWidget}
        @type cameraName: C{str}
        @param viewportWidget: The viewport whose camera location should be
            set.
        @param cameraName: The name of the built-in camera from which the given
            viewport should view the scene.
        """
        viewport = viewportWidget.getViewport()

        camera = viewportWidget.getCameraByName(cameraName)
        if not camera:
            log.warning("No camera found in Viewport '%s': %s",
                        viewportWidget.getViewport().getName(),
                        cameraName)
            return

        # Make sure that interactions are allowed.
        camera.disableInteraction(False)

        # Set this camera as the active camera
        viewportWidget.setActiveCamera(camera)
        camera.setViewportDimensions(viewportWidget.width(),
                                     viewportWidget.height())
        viewportWidget.update()
        viewportWidget.setDirty(True)

        # Update the camera picker button.
        cameraPickerButton, _, _ = \
            self.__viewportNamesAndButtons[viewportWidget.getName()]
        cameraPickerButton.setText(cameraName)

        # Finally, clear the look-through location from the Viewer Delegate.
        self.__viewerDelegate.setLookThroughLocation(viewport, "")

    def saveViewports(self):
        """
        Saves the contents of the viewports to images on disk. Hard-wired to
        write to "/tmp/viewport_left.png" and "/tmp/viewport_right.png".
        """
        self.__viewportWidgetL.writeToFile("/tmp/viewport_left.png")
        if self.__viewportWidgetR:
            self.__viewportWidgetR.writeToFile("/tmp/viewport_right.png")

    def _on_node_setLocked(self, eventType, eventID, node, locked):
        """
        Overridden to update the manipulators when the lock state of a node
        changes (manipulators depend on the lock state of the node they are
        targeting).
        """
        UI4.Tabs.BaseViewerTab._on_node_setLocked(self, eventType, eventID,
                                                  node, locked)
        self.__manipulatorsMenu.updateManipulators()

    def _on_scenegraphManager_selectionChanged(self, eventType, eventID, **kwargs):
        """
        Handles C{scenegraphManager_selectionChanged} events by dirtying
        attached viewports.
        """
        UI4.Tabs.BaseViewerTab._on_scenegraphManager_selectionChanged(self,
                                                                      eventType,
                                                                      eventID,
                                                                      **kwargs)
        self.__manipulatorsMenu._on_selection_changed()

    def _onUpdateEvent(self):
        """
        This extends the BaseViewerTab implementation.
        """
        UI4.Tabs.BaseViewerTab._onUpdateEvent(self)

        # Update the cooking icon, so that it spins.
        for _, _, cookingIcon in self.__viewportNamesAndButtons.values():
            if cookingIcon.getActive():
                cookingIcon.advanceTick()

        # Check if there was a pending request to change to a built-in camera.
        for viewportName, cameraName in \
                self.__pendingBuiltInCameraRequests.items():
            viewportWidget = self.getViewportWidget(
                viewportName, self.__viewerDelegate)
            if viewportWidget:
                self.__setBuiltInCamera(viewportWidget, cameraName)

        self.__pendingBuiltInCameraRequests = {}

        # Check if there is a status update to display
        statusOptionId = ViewerAPI.GenerateOptionId("ExampleViewerStatus")
        statusOptionAttr = self.__viewerDelegate.getOption(statusOptionId)
        if statusOptionAttr:
            status = statusOptionAttr.getValue("")
            self.__setStatusBarMessage(status)

    def _on_lookThroughLocation_changeRequested(self, viewportName, location):
        """
        Overridden from base class to update the picker and the status icon.
        """
        UI4.Tabs.BaseViewerTab._on_lookThroughLocation_changeRequested(
            self, viewportName, location)

        _, warningIcon, cookingIcon = \
            self.__viewportNamesAndButtons[viewportName]
        warningIcon.hide()
        cookingIcon.setActive(True)
        cookingIcon.setVisible(True)

    def _on_lookThroughLocation_doesNotExist(self, viewportName, location):
        """
        Overridden from base class to update the picker and the status icon.
        """
        UI4.Tabs.BaseViewerTab._on_lookThroughLocation_doesNotExist(
            self, viewportName, location)

        _, warningIcon, cookingIcon = \
            self.__viewportNamesAndButtons[viewportName]
        warningIcon.show()
        cookingIcon.setActive(False)
        cookingIcon.setVisible(False)

        # Mark the viewport dirty, so that things like the CameraGateLayer
        # can be re-drawn.
        viewportWidget = self.getViewportWidget(viewportName,
                                                self.__viewerDelegate)
        viewportWidget.setDirty(True)

        # Pop to "persp". However, we cannot do this during the callback, we
        # need to defer it.
        self.__pendingBuiltInCameraRequests[viewportName] = "persp"

    def _on_lookThroughLocation_changed(self, viewportName, location, attrs):
        """
        Overridden from base class to update the picker and the status icon.
        """
        UI4.Tabs.BaseViewerTab._on_lookThroughLocation_changed(
            self, viewportName, location, attrs)

        _, warningIcon, cookingIcon = \
            self.__viewportNamesAndButtons[viewportName]
        warningIcon.hide()
        cookingIcon.setActive(False)
        cookingIcon.setVisible(False)

        # Get the viewport and the active camera.
        viewport = self.__viewerDelegate.getViewport(viewportName)
        camera = viewport.getActiveCamera()

        # Make sure that interactions are allowed.
        camera.disableInteraction(False)

        # Ensure live rendering knows about the new camera
        self.updateLiveRenderCamera(location, camera)

    def _on_lookThroughLocation_cooked(self, viewportName, location, attrs):
        """
        Overridden from base class to update the camera, in case its attributes
        have changed.
        """
        UI4.Tabs.BaseViewerTab._on_lookThroughLocation_cooked(
            self, viewportName, location, attrs)

        # Setup the camera with the new location and attributes.
        viewportWidget = self.getViewportWidget(viewportName,
                                                self.__viewerDelegate)
        self.__setSceneGraphCamera(viewportWidget, location, attrs)

    def _on_viewportView_frozenStateChanged(self, viewportName, isViewFrozen):
        """
        Overridden from base class to update the camera when it has been
        thawed.
        """
        UI4.Tabs.BaseViewerTab._on_viewportView_frozenStateChanged(
            self, viewportName, isViewFrozen)

        # Only handle when it has been thawed.
        if isViewFrozen:
            return

        viewportWidget = self.getViewportWidget(viewportName,
                                                self.__viewerDelegate)

        # If this is a scene graph camera, setup the camera with its current
        # attributes.
        camera = viewportWidget.getActiveCamera()
        location = camera.getLocationPath() if camera else ""
        if location:
            attrs = self.__viewerDelegate.getAttributes(
                location, ViewerAPI.kFnViewerDelegateLookThrough)
            self.__setSceneGraphCamera(viewportWidget, location, attrs)

    def __setSceneGraphCamera(self, viewportWidget, location, attrs):
        """
        Sets the camera up given the viewport, the scene graph location and its
        attributes.

        It resets the camera geometry, dimensions, location path, sets the
        camera active, marks the viewport dirty, recreates the camera if the
        projection has changed, etc.

        @type viewportWidget: C{ViewportWidget}
        @type location: C{str}
        @type attrs: C{FnAttribute.GroupAttribute}
        @param viewportWidget: The viewport whose camera location should be
            set.
        @param location: This is the look-through location for the camera.
        @param attrs: Attributes for the camera that includes the xform, center
            of interest, near/fov planes, etc.
        """
        # attrs cannot be None
        assert attrs is not None

        # Do not modify the camera if the viewport view is frozen.
        isViewportViewFrozen = viewportWidget.getViewport().isViewFrozen()
        if isViewportViewFrozen:
            return

        # Get the expected camera type for the current projection.
        cameraTypeID = ViewerAPI.kFnViewportCameraTypePerspective
        projectionAttr = attrs.getChildByName("geometry.projection")
        if projectionAttr and projectionAttr.getValue("") == "orthographic":
            cameraTypeID = ViewerAPI.kFnViewportCameraTypeOrthographic

        # Create the camera if it doesn't exist, or if camera type is not the
        # one that is required for the current projection.
        cameraName = _sceneGraphCameraName
        camera = viewportWidget.getCameraByName(cameraName)
        if not camera or camera.getCameraTypeID() != cameraTypeID:
            pluginName = "PerspectiveCamera"
            if cameraTypeID == ViewerAPI.kFnViewportCameraTypeOrthographic:
                pluginName = "OrthographicCamera"

            # Add (or replace) the camera with the correct plug-in name.
            camera = viewportWidget.addCamera(pluginName, cameraName)

        # Setup the camera.
        camera.setLocationPath(location)
        camera.setup(attrs)
        camera.setViewportDimensions(viewportWidget.width(),
                                     viewportWidget.height())
        viewportWidget.setActiveCamera(camera)
        viewportWidget.update()
        viewportWidget.setDirty(True)

        # Update the camera picker button.
        cameraPickerButton, _, _ = \
            self.__viewportNamesAndButtons[viewportWidget.getName()]
        cameraPickerButton.setText(location)

    def __setStatusBarMessage(self, statusBarMessage):
        """
        Sets the message in the status bar.

        @type statusBarMessage: C{str}
        @param statusBarMessage: The message to display in the status bar.
        """
        if statusBarMessage != self.__statusBarMessage:
            self.__statusBarMessage = statusBarMessage
            self.__statusBar.showMessage(statusBarMessage)

    def getManipulatorMenu(self):
        """
        @rtype: C{ManipulatorMenu}
        @return: Returns the manipulator menu.
        """
        return self.__manipulatorsMenu
