// Copyright (c) 2016 The Foundry Visionmongers Ltd. All Rights Reserved.
#include "translate/FnGLTranslateScreenPlaneHandle.h"
#include "translate/FnGLTranslateManipulator.h"
#include "FnMeshGeneration.h"
#include "FnManipulatorsConfig.h"
#include <FnViewer/plugin/FnViewportCamera.h>
#include <FnViewer/utils/FnImathHelpers.h>
#include <FnViewer/utils/FnDrawingHelpers.h>
#include <FnViewer/plugin/FnOptionIdGenerator.h>
#include <FnViewer/plugin/FnViewportLayer.h>

#include <algorithm>
#include <cmath>
#include <vector>

#include <GL/glew.h>
#include <GL/glu.h>
// Adding a possibility of ignoring the Qt dependecy when building
// ViewerPlugins separately
#ifdef BUILD_WITH_QT
#include <QtCore/QSize>
#include <QtGui/QIcon>
#include <QtGui/QImage>
#include <QtGui/QPainter>
#include <QtSvg/QSvgRenderer>
#endif

#include <ImathMatrix.h>
#include <ImathMatrixAlgo.h>

using Foundry::Katana::ViewerAPI::Vec4f;
using Foundry::Katana::ViewerAPI::ViewportCameraWrapperPtr;

using Foundry::Katana::ViewerUtils::GLStateRestore;
using Foundry::Katana::ViewerUtils::toImathMatrix44d;
using Foundry::Katana::ViewerUtils::toImathV3d;
using Foundry::Katana::ViewerUtils::toVec3d;

using namespace Foundry::Katana::ViewerUtils;
using OptID = Foundry::Katana::ViewerAPI::OptionIdGenerator;

FnLogSetup("Viewer.GLTranslateScreenPlaneHandle");

namespace
{
GLuint loadTexture(const std::string& textureSubPath, const float scale)
{
#ifdef BUILD_WITH_QT
    const char* const katanaRootCStr = getenv("KATANA_ROOT");
    if (katanaRootCStr == nullptr)
    {
        FnLogError("Couldn't get KATANA_ROOT environment variable.");
        return 0;
    }

    const std::string fullPath = std::string(katanaRootCStr) +
                                 "/bin/python/UI4/Resources/Icons/" +
                                 textureSubPath;

    QSvgRenderer renderer(QString::fromStdString(fullPath));
    if (!renderer.isValid())
    {
        return 0u;
    }

    QImage image(QSize(48, 48) * scale, QImage::Format_RGBA8888);
    image.fill(0x00000000);

    // Get QPainter that paints to the image
    QPainter painter(&image);
    renderer.render(&painter);

    GLuint textureId;
    glGenTextures(1, &textureId);
    glActiveTexture(GL_TEXTURE0);
    glBindTexture(GL_TEXTURE_2D, textureId);
    glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_S, GL_REPEAT);
    glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_T, GL_REPEAT);
    glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_LINEAR);
    glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_LINEAR);
    glPixelStorei(GL_UNPACK_ALIGNMENT, 1);
    glTexImage2D(GL_TEXTURE_2D, 0, GL_RGBA, image.width(),
                 image.height(), 0, GL_RGBA, GL_UNSIGNED_BYTE,
                 image.bits());
    glGenerateMipmap(GL_TEXTURE_2D);
    glBindTexture(GL_TEXTURE_2D, 0);
    return textureId;
#else
    return 0u;
#endif
}
}  // namespace

GLTranslateScreenPlaneHandle::GLTranslateScreenPlaneHandle()
    : GLTransformManipulatorHandle(
          GLTranslateManipulator::kAlwaysAtLocationOrigin),
      m_orientation(Foundry::Katana::ViewerUtils::WORLD)
{
    m_handlePriority = 1;
}

GLTranslateScreenPlaneHandle::~GLTranslateScreenPlaneHandle()
{
    m_snappingTextures.cleanup();
}

void GLTranslateScreenPlaneHandle::setup()
{
    const float scale =
        isPlacedOnCenterOfInterest() ? FnManipulators::TranslateCoiScale : 1.0f;

    FnAttribute::DoubleAttribute globalScale =
        getManipulator()->getOption(OpId::GenerateId("Manipulator.Scale"));

    loadTextures(scale * static_cast<float>(globalScale.getValue()));

    generateSquare(Vec3f(0.f, 0.f, 0.f),
                   FnManipulators::TranslateSquareLength * scale, m_mesh);
}

void GLTranslateScreenPlaneHandle::draw()
{
    // Change the orientation before calling the base definition. This will make
    // the mesh to be oriented towards the camera.
    m_orientation = Foundry::Katana::ViewerUtils::VIEW;
    GLTransformManipulatorHandle::draw();
    m_orientation = Foundry::Katana::ViewerUtils::WORLD;

    // When dragging, we want to ignore the Z buffer and render on top.
    GLStateRestore glStateRestore(Foundry::Katana::ViewerUtils::DepthBuffer);
    glDepthFunc(GL_ALWAYS);

    if (isSnapping())
    {
        drawSnappingHandle();
    }
    else
    {
        drawHandle();
    }
}

void GLTranslateScreenPlaneHandle::pickerDraw(int64_t pickerId)
{
    // Change the orientation before calling the base definition. This will make
    // the mesh to be oriented towards the camera.
    m_orientation = Foundry::Katana::ViewerUtils::VIEW;
    GLTransformManipulatorHandle::pickerDraw(pickerId);
    m_orientation = Foundry::Katana::ViewerUtils::WORLD;

    // When dragging, we want to ignore the Z buffer and render on top.
    GLStateRestore glStateRestore(Foundry::Katana::ViewerUtils::DepthBuffer);
    glDepthFunc(GL_ALWAYS);

    // Use the shader and draw
    usePickingShader(getXform(), pickerId, m_handlePriority);
    m_mesh.draw(false, false);
}

Foundry::Katana::ViewerUtils::Orientation
GLTranslateScreenPlaneHandle::getOrientation()
{
    return m_orientation;
}

bool GLTranslateScreenPlaneHandle::getDraggingPlane(Vec3d& origin, Vec3d& normal)
{
    if (!isDragging())
    {
        // We want to cache the manipulator position
        // to minimise rounding errors.
        m_manipOrigin = toVec3d(getOrigin());
    }
    // Get the manipulator and the camera
    GLTransformManipulator* manip = getGLTransformManipulator();
    ViewportCameraWrapperPtr camera = manip->getViewport()->getActiveCamera();

    origin = m_manipOrigin;
    normal = camera->getDirection();

    return true;
}

void GLTranslateScreenPlaneHandle::drag(const Vec3d& initialPointOnPlane,
    const Vec3d& previousPointOnPlane,
    const Vec3d& currentPointOnPlane,
    const Vec2i& initialMousePosition,
    const Vec2i& previousMousePosition,
    const Vec2i& currentMousePosition,
    bool isFinal)
{
    SnappingData snappingData = pickSnappingTarget(currentMousePosition);

    IMATH_NAMESPACE::V3d delta;
    if (snappingData.snap)
    {
        Vec3d origin, normal;
        getDraggingPlane(origin, normal);
        delta = toImathV3d(snappingData.point) - toImathV3d(origin);
    }
    else
    {
        delta =
            toImathV3d(currentPointOnPlane) - toImathV3d(initialPointOnPlane);
    }

    if (getTransformMode() == FnKat::ViewerUtils::kCenterOfInterest)
    {
        GLTransformManipulator* const manip = getGLTransformManipulator();
        manip->applyCoiTranslationToAllLocations(
            delta, delta.length(), getOrientation(), isFinal);

        // Notice that the object itself is not translated in this mode.
    }
    else if (getTransformMode() == FnKat::ViewerUtils::kAroundCenterOfInterest)
    {
        // Get the dragging along the axis in manipulator space.
        IMATH_NAMESPACE::M44d translationXform;
        translationXform.setTranslation(delta);

        // Set the translation values
        applyXformToAllLocations(translationXform, isFinal);

        // Now that the objects are in their final position, we can apply the
        // rest of the manipulation (i.e. the orientation).
        GLTransformManipulator* const manip = getGLTransformManipulator();
        manip->applyTranslationAroundCoiToAllLocations(isFinal);
    }
    else
    {
        // The handle was snapped with orientation.
        if (isSnapWithOrientation(snappingData))
        {
            const IMATH_NAMESPACE::V3d orientation(snappingData.normal.x,
                                                   snappingData.normal.y,
                                                   snappingData.normal.z);
            const IMATH_NAMESPACE::V3d fromDir =
                getGLTransformManipulator()->getAxisNormalOrientation();
            const IMATH_NAMESPACE::V3d fromUpDir =
                getGLTransformManipulator()->getAxisUpOrientation();
            const IMATH_NAMESPACE::M44d rotationMatrix =
                GLTransformManipulator::rotationMatrixWithUpDir(
                    fromDir, orientation, fromUpDir);
            setAllLocationsOrientation(rotationMatrix, isFinal);
        }
        else
        {
            restoreAllLocationsXform();
        }

        // Get the dragging along the axis in manipulator space.
        IMATH_NAMESPACE::M44d translationXform;
        translationXform.setTranslation(delta);

        // Set the translation values
        applyXformToAllLocations(translationXform, isFinal);
    }
}

void GLTranslateScreenPlaneHandle::startDrag(const Vec3d& initialPointOnPlane,
    const Vec2i& initialMousePosition)
{
    GLTransformManipulatorHandle::startDrag(initialPointOnPlane,
        initialMousePosition);
}

void GLTranslateScreenPlaneHandle::endDrag()
{
    GLTransformManipulatorHandle::endDrag();
}

void GLTranslateScreenPlaneHandle::applyXformToLocation(
    const std::string& locationPath,
    const IMATH_NAMESPACE::M44d& xform,
    bool isFinal)
{
    GLTransformManipulator* manipulator = getGLTransformManipulator();

    const IMATH_NAMESPACE::M44d initialXform =
        manipulator->getInitialTranslateXformFromLocation(locationPath);

    // Apply the manipulation transform to the current translate value
    IMATH_NAMESPACE::M44d translateMatrix = initialXform * xform;

    // Extract the resulting translation values
    IMATH_NAMESPACE::V3d translate = translateMatrix.translation();

    // Set the translation back in the node graph
    manipulator->setValue(locationPath, "xform.interactive.translate",
        FnAttribute::DoubleAttribute(&translate[0], 3, 3), isFinal);
}


void GLTranslateScreenPlaneHandle::setOption(
    OpId::value_type optionId,
    FnAttribute::Attribute attr)
{
    GLTransformManipulatorHandle::setOption(optionId, attr);

    static const OpId::value_type s_globalScaleId =
        OpId::GenerateId("Manipulator.Scale");

    if (s_globalScaleId == optionId)
    {
        // For safety, all the multipliers will be ranged in [0.01, 10.0].
        double value = FnAttribute::DoubleAttribute(attr).getValue(1.0, false);
        value = std::max(0.01, std::min(10.0, value));
        getViewport()->makeGLContextCurrent();
        m_snappingTextures.cleanup();
        loadTextures(static_cast<float>(value));
        getViewport()->setDirty(true);
    }
}

void GLTranslateScreenPlaneHandle::calculateAndSetLocalXform(
    const std::string& locationPath)
{
    if (GLTranslateManipulator::kAlwaysAtLocationOrigin)
    {
        GLTransformManipulatorHandle::calculateAndSetLocalXform(locationPath);
    }
    else
    {
        IMATH_NAMESPACE::M44d orientation =
            getGLTransformManipulator()->getOrientationXform(locationPath,
                                                             getOrientation());
        IMATH_NAMESPACE::M44d localXform =
            orientation * calculateXform(locationPath, true);
        setLocalXform(toMatrix44d(localXform));
    }
}

void GLTranslateScreenPlaneHandle::loadTextures(const float scale)
{
    m_snappingTextures.defaultTexture =
        ::loadTexture("Snapping/snappingModeHandleVisualClue.svg", scale);
    m_snappingTextures.hoverTexture = ::loadTexture(
        "Snapping/snappingModeHandleVisualClue_hoverOver.svg", scale);
    m_snappingTextures.draggingTexture =
        ::loadTexture("Snapping/snappingModeHandleVisualClue_drag.svg", scale);
}

bool GLTranslateScreenPlaneHandle::isSnapping()
{
    ViewportWrapperPtr viewport = getViewport();
    ViewportLayerWrapperPtr snappingLayer = viewport->getLayer("SnappingLayer");

    static const OptID::value_type kSnappingLayerEnabled =
        OptID::GenerateId("SnappingLayer.Enabled");
    if (snappingLayer == nullptr)
    {
        FnLogWarn("'SnappingLayer' not available.");
        return false;
    }
    const FnAttribute::IntAttribute snappingLayerEnabled =
        snappingLayer->getOption(kSnappingLayerEnabled);
    return snappingLayerEnabled.getValue() == 1;
}


void GLTranslateScreenPlaneHandle::drawSnappingHandle()
{
    const GLuint textureId = getTexture();
    if (textureId == 0)
    {
        // Fallback to drawing regular handle
        drawHandle();
        return;
    }

    glActiveTexture(GL_TEXTURE0);
    glBindTexture(GL_TEXTURE_2D, textureId);
    useSnappingShader(getXform());
    const GLboolean sRGBFramebufferWasEnabled =
        glIsEnabled(GL_FRAMEBUFFER_SRGB);
    if (sRGBFramebufferWasEnabled == GL_TRUE)
    {
        glDisable(GL_FRAMEBUFFER_SRGB);
    }
    m_mesh.draw(true, false);
    if (sRGBFramebufferWasEnabled == GL_TRUE)
    {
        glEnable(GL_FRAMEBUFFER_SRGB);
    }
    glBindTexture(GL_TEXTURE_2D, 0);
}

void GLTranslateScreenPlaneHandle::drawHandle()
{
    const Vec4f color = isPlacedOnCenterOfInterest()
                            ? FnManipulators::HandleAlternativeColorCentre
                            : FnManipulators::HandleColorCentre;

    useDrawingShader(getXform(), getDisplayColor(color), true);
    m_mesh.draw(true, false);
}

GLuint GLTranslateScreenPlaneHandle::getTexture()
{
    GLuint textureId = 0;
    if (isDragging())
    {
        textureId = m_snappingTextures.draggingTexture;
    }
    else if (isHovered() || isActive())
    {
        textureId = m_snappingTextures.hoverTexture;
    }
    else
    {
        textureId = m_snappingTextures.defaultTexture;
    }
    return textureId;
}
