// Copyright (c) 2018 The Foundry Visionmongers Ltd. All Rights Reserved.
#include "coi/FnGLCoiManipulator.h"

#include "coi/FnGLCoiHandle.h"
#include "rotate/FnGLRotateAxisHandle.h"
#include "rotate/FnGLRotateBallHandle.h"
#include "translate/FnGLTranslateAxisHandle.h"
#include "translate/FnGLTranslatePlaneHandle.h"
#include "translate/FnGLTranslateScreenPlaneHandle.h"


using Foundry::Katana::ViewerUtils::GLTransformManipulatorHandle;

// --- GLCoiManipulator --------------------

GLCoiManipulator::GLCoiManipulator()
    : m_coiHandle(nullptr)
{
}

GLCoiManipulator::~GLCoiManipulator() {}

void GLCoiManipulator::draw()
{
    m_coiHandle->draw();
    for (GLTransformManipulatorHandle* const handle : m_rotateAroundCoiHandles)
    {
        handle->draw();
    }

    // Clear the depth buffer, so that the following handles are drawn on top.
    glClear(GL_DEPTH_BUFFER_BIT);

    for (GLTransformManipulatorHandle* const handle :
         m_translateAroundCoiHandles)
    {
        handle->draw();
    }
    for (GLTransformManipulatorHandle* const handle : m_translateCoiHandles)
    {
        handle->draw();
    }
}

void GLCoiManipulator::pickerDraw(int64_t pickerID)
{
    m_coiHandle->pickerDraw(pickerID++);
    for (GLTransformManipulatorHandle* const handle : m_rotateAroundCoiHandles)
    {
        handle->pickerDraw(pickerID++);
    }

    // Clear the depth buffer, so that the following handles are drawn on top.
    glClear(GL_DEPTH_BUFFER_BIT);

    for (GLTransformManipulatorHandle* const handle :
         m_translateAroundCoiHandles)
    {
        handle->pickerDraw(pickerID++);
    }
    for (GLTransformManipulatorHandle* const handle : m_translateCoiHandles)
    {
        handle->pickerDraw(pickerID++);
    }
}

void GLCoiManipulator::setup()
{
    using FnKat::ViewerUtils::TransformMode;

    // COI handle. This is what renders the lines and crosses to join objects
    // and their center of interests.
    Foundry::Katana::ViewerAPI::ManipulatorHandleWrapperPtr coiHandleWrapper =
        Manipulator::addManipulatorHandle("GLCoiHandle", "CoiHandle");
    m_coiHandle = coiHandleWrapper->getPluginInstance<GLCoiHandle>();

    // Rotate around COI manipulators.
    {
        // Central Ball handle
        addManipulatorHandle<GLRotateBallHandle>(
            "GLRotateBallHandle",
            "RotateBall",
            true,
            kAroundCenterOfInterest,
            m_rotateAroundCoiHandles)->setup();

        // X axis handle
        addManipulatorHandle<GLRotateAxisHandle>(
            "GLRotateAxisHandle", "RotateAxisX", true, kAroundCenterOfInterest,
            m_rotateAroundCoiHandles)
            ->setup(IMATH_NAMESPACE::V3d(1.0, 0.0, 0.0));

        // Y axis handle
        addManipulatorHandle<GLRotateAxisHandle>(
            "GLRotateAxisHandle", "RotateAxisY", true, kAroundCenterOfInterest,
            m_rotateAroundCoiHandles)
            ->setup(IMATH_NAMESPACE::V3d(0.0, 1.0, 0.0));

        // Z axis handle
        addManipulatorHandle<GLRotateAxisHandle>(
            "GLRotateAxisHandle", "RotateAxisZ", true, kAroundCenterOfInterest,
            m_rotateAroundCoiHandles)
            ->setup(IMATH_NAMESPACE::V3d(0.0, 0.0, 1.0));

        // View handle
        addManipulatorHandle<GLRotateAxisHandle>(
            "GLRotateAxisHandle",
            "RotateAxisV",
            true,
            kAroundCenterOfInterest,
            m_rotateAroundCoiHandles)->setupViewHandle();
    }

    // Translate around COI manipulators.
    {
        // X axis handle
        addManipulatorHandle<GLTranslateAxisHandle>(
            "GLTranslateAxisHandle", "TranslateAxisX", false,
            kAroundCenterOfInterest, m_translateAroundCoiHandles)
            ->setup(IMATH_NAMESPACE::V3d(1.0, 0.0, 0.0));

        // Y axis handle
        addManipulatorHandle<GLTranslateAxisHandle>(
            "GLTranslateAxisHandle", "TranslateAxisY", false,
            kAroundCenterOfInterest, m_translateAroundCoiHandles)
            ->setup(IMATH_NAMESPACE::V3d(0.0, 1.0, 0.0));

        // Z axis handle
        addManipulatorHandle<GLTranslateAxisHandle>(
            "GLTranslateAxisHandle", "TranslateAxisZ", false,
            kAroundCenterOfInterest, m_translateAroundCoiHandles)
            ->setup(IMATH_NAMESPACE::V3d(0.0, 0.0, 1.0));

        // Free handle
        addManipulatorHandle<GLTranslateScreenPlaneHandle>(
            "GLTranslateScreenPlaneHandle",
            "TranslateScreenPlane",
            false,
            kAroundCenterOfInterest,
            m_translateAroundCoiHandles)->setup();

        // YZ plane handle
        addManipulatorHandle<GLTranslatePlaneHandle>(
            "GLTranslatePlaneHandle", "TranslatePlaneYZ", false,
            kAroundCenterOfInterest, m_translateAroundCoiHandles)
            ->setup(IMATH_NAMESPACE::V3d(1.0, 0.0, 0.0));

        // XZ plane handle
        addManipulatorHandle<GLTranslatePlaneHandle>(
            "GLTranslatePlaneHandle", "TranslatePlaneXZ", false,
            kAroundCenterOfInterest, m_translateAroundCoiHandles)
            ->setup(IMATH_NAMESPACE::V3d(0.0, 1.0, 0.0));

        // XY plane handle
        addManipulatorHandle<GLTranslatePlaneHandle>(
            "GLTranslatePlaneHandle", "TranslatePlaneXY", false,
            kAroundCenterOfInterest, m_translateAroundCoiHandles)
            ->setup(IMATH_NAMESPACE::V3d(0.0, 0.0, 1.0));
    }

    // Translate COI manipulators.
    {
        // X axis handle
        addManipulatorHandle<GLTranslateAxisHandle>(
            "GLTranslateAxisHandle", "TranslateCoiAxisX", true,
            kCenterOfInterest, m_translateCoiHandles)
            ->setup(IMATH_NAMESPACE::V3d(1.0, 0.0, 0.0));

        // Y axis handle
        addManipulatorHandle<GLTranslateAxisHandle>(
            "GLTranslateAxisHandle", "TranslateCoiAxisY", true,
            kCenterOfInterest, m_translateCoiHandles)
            ->setup(IMATH_NAMESPACE::V3d(0.0, 1.0, 0.0));

        // Z axis handle
        addManipulatorHandle<GLTranslateAxisHandle>(
            "GLTranslateAxisHandle", "TranslateCoiAxisZ", true,
            kCenterOfInterest, m_translateCoiHandles)
            ->setup(IMATH_NAMESPACE::V3d(0.0, 0.0, 1.0));

        // Free handle
        addManipulatorHandle<GLTranslateScreenPlaneHandle>(
            "GLTranslateScreenPlaneHandle",
            "TranslateCoiScreenPlane",
            true,
            kCenterOfInterest,
            m_translateCoiHandles)->setup();

        // YZ plane handle
        addManipulatorHandle<GLTranslatePlaneHandle>(
            "GLTranslatePlaneHandle", "TranslateCoiPlaneYZ", true,
            kCenterOfInterest, m_translateCoiHandles)
            ->setup(IMATH_NAMESPACE::V3d(1.0, 0.0, 0.0));

        // XZ plane handle
        addManipulatorHandle<GLTranslatePlaneHandle>(
            "GLTranslatePlaneHandle", "TranslateCoiPlaneXZ", true,
            kCenterOfInterest, m_translateCoiHandles)
            ->setup(IMATH_NAMESPACE::V3d(0.0, 1.0, 0.0));

        // XY plane handle
        addManipulatorHandle<GLTranslatePlaneHandle>(
            "GLTranslatePlaneHandle", "TranslateCoiPlaneXY", true,
            kCenterOfInterest, m_translateCoiHandles)
            ->setup(IMATH_NAMESPACE::V3d(0.0, 0.0, 1.0));
    }
}
