// Copyright (c) 2018 The Foundry Visionmongers Ltd. All Rights Reserved.

#include "ExampleSpotlightLocator.h"

#include <FnViewer/plugin/FnGLStateHelper.h>
#include <fstream>
#include <vector>
//-----------------------------------------------------------------------------
using namespace FnKat::ViewerUtils;

ExampleSpotlightLocatorVDC::ExampleSpotlightLocatorVDC()
: m_overrideStandardLocators(true)
{
}

bool ExampleSpotlightLocatorVDC::locationEvent(
    const Foundry::Katana::ViewerAPI::ViewerLocationEvent& event,
    bool locationHandled)
{
    bool handled = FnBaseLocatorVDC::locationEvent(event, locationHandled);
    if (m_overrideStandardLocators)
    {
        return handled;
    }

    return false;
}

void ExampleSpotlightLocatorVDC::setOption(
        FnKat::ViewerAPI::OptionIdGenerator::value_type optionId,
        FnAttribute::Attribute attr)
{
    FnBaseLocatorVDC::setOption(optionId, attr);

    static const FnKat::ViewerAPI::OptionIdGenerator::value_type
        s_overrideHash = FnKat::ViewerAPI::OptionIdGenerator::GenerateId(
            "overrideStandardLocators");

    if (optionId == s_overrideHash)
    {
        FnAttribute::IntAttribute overrideAttr(attr);
        m_overrideStandardLocators = 0 != overrideAttr.getValue(1, false);
    }
}


ExampleSpotlightLocatorViewportLayer::ExampleSpotlightLocatorViewportLayer()
    : m_enlargedLocators(false)
{
}

void ExampleSpotlightLocatorViewportLayer::draw()
{
        // GLStateRestore prevents us from polluting the GL context
    FnKat::ViewerUtils::GLStateRestore glStateRestore(
        FnKat::ViewerUtils::Polygon | FnKat::ViewerUtils::Lighting |
        FnKat::ViewerUtils::Line);
    if (m_enlargedLocators)
    {
        glLineWidth(3);
    }
    FnBaseLocatorViewportLayer::draw();
}

/// Sets a generic option.
void ExampleSpotlightLocatorViewportLayer::setOption(
    FnKat::ViewerAPI::OptionIdGenerator::value_type optionId,
    FnAttribute::Attribute attr)
{
    static const FnKat::ViewerAPI::OptionIdGenerator::value_type
        s_enlargedLocatorsHash =
            FnKat::ViewerAPI::OptionIdGenerator::GenerateId(
                "enlarged_locators");

    if (optionId == s_enlargedLocatorsHash)
    {
        FnAttribute::IntAttribute enlargedLocatorsAttr(attr);
        m_enlargedLocators = 0 != enlargedLocatorsAttr.getValue(1, false);

        // Dirty this viewport
        getViewport()->setDirty(true);
    }
    else
    {
        FnBaseLocatorViewportLayer::setOption(optionId, attr);
    }
}

//-----------------------------------------------------------------------------

bool ExampleSpotlightLocator::matches(
    const FnKat::ViewerAPI::ViewerLocationEvent& event)
{
    FnAttribute::StringAttribute lightTypeAttr =
        event.attributes.getChildByName("viewer.lightType");

    return lightTypeAttr.getValue("", false) == "spot";
}

bool ExampleSpotlightLocator::overridesBaseGeometry(
    const FnKat::ViewerAPI::ViewerLocationEvent& event)
{
    return true;
}

/// Gets the bounds of the given location.
/*static*/ FnAttribute::DoubleAttribute ExampleSpotlightLocator::getBounds(
    const FnKat::ViewerAPI::ViewerLocationEvent& event)
{
    const double bound[6] = {-2.5, 2.5, -2.5, 2.5, -5, 0};
    static const FnAttribute::DoubleAttribute s_boundAttr(bound, 6, 2);
    return s_boundAttr;
}

/// Computes the extent of the given location.
/*static*/ FnAttribute::DoubleAttribute ExampleSpotlightLocator::computeExtent(
    const FnKat::ViewerAPI::ViewerLocationEvent& event)
{
    return ExampleSpotlightLocator::getBounds(event);
}

void ExampleSpotlightLocator::draw(const std::string& locationPath)
{
    // GLStateRestore prevents us from polluting the GL context
    FnKat::ViewerUtils::GLStateRestore glStateRestore(
        FnKat::ViewerUtils::Polygon | FnKat::ViewerUtils::Lighting |
        FnKat::ViewerUtils::Line);

    glDisable(GL_LIGHTING);
    glPolygonMode(GL_FRONT_AND_BACK, GL_LINE);

    auto shaderProgram = getDefaultShaderProgram();
    assert(shaderProgram);
    if (isLocationSelected(locationPath))
    {
        // Selected objects draw in white
        shaderProgram->setUniform("Color", 1.0f, 1.0f, 1.0f, 1.0f);
    }
    else
    {
        // See if the draw color has been set by viewer object settings
        FnAttribute::FloatAttribute colorAttr =
            getAttribute(locationPath, "viewer.default.drawOptions.color");
        if (colorAttr.isValid())
        {
            assert(colorAttr.getNumberOfValues() == 3);

            auto color = colorAttr.getNearestSample(0.0f);
            shaderProgram->setUniform("Color", color[0], color[1], color[2],
                1.0f);
        }
        else
        {
            // Draw light in yellow
            shaderProgram->setUniform("Color", 1.0f, 1.0f, 0.0f, 1.0f);
        }
    }

    drawLight(locationPath);
}

void ExampleSpotlightLocator::pickerDraw(
    const std::string& locationPath)
{
    FnKat::ViewerUtils::GLStateRestore glStateRestore(
        FnKat::ViewerUtils::Polygon | FnKat::ViewerUtils::Lighting |
        FnKat::ViewerUtils::Line);
    glDisable(GL_LIGHTING);
    glPolygonMode(GL_FRONT_AND_BACK, GL_LINE);

    /* Easier picking. */
    glLineWidth(7);

    drawLight(locationPath);
}

void ExampleSpotlightLocator::generateCone(float radius,
                                           float height,
                                           int segments,
                                           std::vector<float>& vertices,
                                           std::vector<unsigned int>& indices)
{
    const float angleDelta = ((2.f * static_cast<float>(M_PI))
                              / static_cast<float>(segments));

    // Cone base
    for (int i = 0; i < segments; ++i)
    {
        float theta = angleDelta * static_cast<float>(i);
        float x = std::cos(theta);
        float y = std::sin(theta);

        FnKat::ViewerAPI::Vec3f vert(x * radius, y * radius, -height);

        vertices.push_back(vert.x);
        vertices.push_back(vert.y);
        vertices.push_back(vert.z);
        indices.push_back(i);
        indices.push_back((i == segments - 1) ? 0 : i + 1);
    }
    int originIndex = static_cast<int>(vertices.size()) / 3;
    vertices.push_back(0.0f);
    vertices.push_back(0.0f);
    vertices.push_back(0.0f);

    int topIndex = static_cast<int>(vertices.size()) / 3;
    vertices.push_back(0.0f);
    vertices.push_back(radius);
    vertices.push_back(-height);

    int bottomIndex = static_cast<int>(vertices.size()) / 3;
    vertices.push_back(0.0f);
    vertices.push_back(-radius);
    vertices.push_back(-height);

    int leftIndex = static_cast<int>(vertices.size()) / 3;
    vertices.push_back(-radius);
    vertices.push_back(0.0f);
    vertices.push_back(-height);

    int rightIndex = static_cast<int>(vertices.size()) / 3;
    vertices.push_back(radius);
    vertices.push_back(0.0f);
    vertices.push_back(-height);

    // Add lines from cone base to origin
    indices.push_back(originIndex);
    indices.push_back(topIndex);

    indices.push_back(originIndex);
    indices.push_back(bottomIndex);

    indices.push_back(originIndex);
    indices.push_back(leftIndex);

    indices.push_back(originIndex);
    indices.push_back(rightIndex);
}

void ExampleSpotlightLocator::setup()
{
    const float radius = 2.5f;
    const float height = 5.0f;
    const int segments = 32;

    generateCone(radius, height, segments, m_vertices, m_indices);

    glGenVertexArrays(1, &m_vao);
    glBindVertexArray(m_vao);

    glGenBuffers(1, &m_vbo);

    // Fill the vertex buffer
    glBindBuffer(GL_ARRAY_BUFFER, m_vbo);
    glBufferData(GL_ARRAY_BUFFER, m_vertices.size() * sizeof(GLfloat),
        m_vertices.data(), GL_STATIC_DRAW);

    glGenBuffers(1, &m_ebo);
    // Fill the index buffer
    glBindBuffer(GL_ELEMENT_ARRAY_BUFFER, m_ebo);
    glBufferData(GL_ELEMENT_ARRAY_BUFFER,
        m_indices.size() * sizeof(unsigned int), &m_indices[0],
        GL_STATIC_DRAW);

    glBindBuffer(GL_ARRAY_BUFFER, m_vbo);
    glVertexAttribPointer(0, 3, GL_FLOAT, GL_FALSE, 0, 0);
    glEnableVertexAttribArray(0);  // VertexPosition

                                   // Unbind buffers
    glBindVertexArray(0);
    glBindBuffer(GL_ELEMENT_ARRAY_BUFFER, 0);
    glBindBuffer(GL_ARRAY_BUFFER, 0);
}

void ExampleSpotlightLocator::cleanup()
{
    glDeleteBuffers(1, &m_vbo);
    glDeleteVertexArrays(1, &m_vao);
}

void ExampleSpotlightLocator::drawLight(const std::string& locationPath)
{
    glBindVertexArray(m_vao);
    glDrawElements(GL_LINES, static_cast<GLsizei>(m_indices.size()),
                   GL_UNSIGNED_INT, 0);
    glBindVertexArray(0);
}
