# Copyright (c) 2018 The Foundry Visionmongers Ltd. All Rights Reserved.
"""
Module defining the L{ExampleSpotlightLocatorVPE} class.
"""

import weakref

import NodegraphAPI
from Katana import (
    FnAttribute,
    FnGeolibServices,
)
from PluginAPI.BaseViewerPluginExtension import BaseViewerPluginExtension


class ExampleSpotlightLocatorVPE(BaseViewerPluginExtension):
    """
    This class is responsible for automatically adding the
    C{'ExampleSpotlightLocatorVDC'} viewer delegate component to a viewer
    delegate, and also for adding the C{'ExampleSpotlightLocatorViewportLayer'}
    to viewports in order to draw spotlight locations.

    This class demonstrates several behaviours:
    - How to register a custom Locator plug-in.
    - Controlling how locators are drawn via messages to the ViewportLayer.
    - How to override the drawing of locations by other Viewer Delegate Components.
    - How to pass changes to locators via the scene graph.
    - How to customize a viewer tab when it is created.
    """

    def __init__(self):
        """
        Initializes an instance of the class.
        """
        BaseViewerPluginExtension.__init__(self)

        self.__vdcName = 'ExampleSpotlightLocatorVDC'
        self.__viewportLayerName = 'ExampleSpotlightLocatorViewportLayer'

        self.__tab = None
        self.__drawExampleSpotlightLocatorsAction = None
        self.__overrideStandardLocatorsAction = None
        self.__enlargeLocatorsAction = None
        self.__drawSpotlightsInCyanAction = None
        self.__viewerDelegateRef = None
        self.__viewerDelegateComponent = None
        self.__viewportWidgets = []

    def onTabCreated(self, tab):
        """
        Callback triggered when an instance of a C{BaseViewerTab} has been
        created.

        @type tab: C{BaseViewerTab}
        @param tab: The viewer tab that was created.
        """
        menuBar = tab.getMenuBar()
        if menuBar is None:
            return

        for action in menuBar.actions():
            if str(action.text()) != 'View':
                continue

            viewMenu = action.menu()
            if not viewMenu:
                continue

            # Add a separator with a title to separate our custom menu items
            # from built-in menu items
            viewMenu.addSeparator().setText(
                'ExampleSpotlightLocatorVPE')

            # Create a menu item for toggling whether to draw geometry for
            # spotlights
            self.__drawExampleSpotlightLocatorsAction = viewMenu.addAction(
                'Draw Example Spotlight Locators')
            self.__drawExampleSpotlightLocatorsAction.setObjectName(
                'drawExampleSpotlightLocatorsAction')
            self.__drawExampleSpotlightLocatorsAction.setCheckable(True)
            self.__drawExampleSpotlightLocatorsAction.setChecked(False)
            self.__drawExampleSpotlightLocatorsAction.toggled[bool].connect(
                self.__on_drawExampleSpotlightLocatorsAction_toggled)

            # Create a menu item for toggling whether to Example spotlights
            # should override the default spotlight locators
            self.__overrideStandardLocatorsAction = viewMenu.addAction(
                'Override Standard Locators')
            self.__overrideStandardLocatorsAction.setObjectName(
                'overrideStandardLocatorsAction')
            self.__overrideStandardLocatorsAction.setCheckable(True)
            self.__overrideStandardLocatorsAction.setChecked(True)
            self.__overrideStandardLocatorsAction.setEnabled(False)
            self.__overrideStandardLocatorsAction.toggled[bool].connect(
                self.__on_overrideStandardLocatorsAction_toggled)

            # Create a menu item for toggling whether to Example spotlights
            # should override the default spotlight locators
            self.__enlargeLocatorsAction = viewMenu.addAction(
                'Enlarge Example Spotlights')
            self.__enlargeLocatorsAction.setObjectName(
                'enlargeExampleSpotlights')
            self.__enlargeLocatorsAction.setCheckable(True)
            self.__enlargeLocatorsAction.setChecked(False)
            self.__enlargeLocatorsAction.setEnabled(False)
            self.__enlargeLocatorsAction.toggled[bool].connect(
                self.__on_enlargeLocatorsAction_toggled)

            # Create a menu item for toggling whether to draw geometry
            # for spotlights in a cyan color, as set up by a custom
            # terminal Op in onApplyTerminalOps() below
            self.__drawSpotlightsInCyanAction = viewMenu.addAction(
                'Draw Spotlights in Cyan')
            self.__drawSpotlightsInCyanAction.setObjectName(
                'drawSpotlightsInCyanAction')
            self.__drawSpotlightsInCyanAction.setCheckable(True)
            self.__drawSpotlightsInCyanAction.toggled[bool].connect(
                self.__on_drawSpotlightsInCyanAction_toggled)

            # Store the tab for use in the slot functions below
            self.__tab = weakref.proxy(tab)

            break

    def onDelegateCreated(self, viewerDelegate, pluginName):
        """
        Adds the C{'ExampleSpotlightLocatorVDC'} viewer delegate component to
        the given viewer delegate.

        @type viewerDelegate: C{ViewerDelegate}
        @type pluginName: C{str}
        @param viewerDelegate: The C{ViewerDelegate} that was created.
        @param pluginName: The registered plug-in name of the
            C{ViewerDelegate}.
        """
        # pylint: disable=unused-argument
        self.__viewerDelegateRef = weakref.ref(viewerDelegate)

        if self.__drawExampleSpotlightEnabled():
            # Add the viewer delegate component if necessary
            self.__viewerDelegateComponent = \
                viewerDelegate.addComponent('ExampleSpotlightLocatorVDC',
                                            self.__vdcName)

    def onViewportCreated(self, viewportWidget, pluginName, viewportName):
        """
        Adds the C{'ExampleSpotlightLocatorViewportLayer'} viewport layer to
        the given viewport widget.

        @type viewportWidget: C{ViewportWidget}
        @type pluginName: C{str}
        @type viewportName: C{str}
        @param viewportWidget: The C{ViewportWidget} that was created.
        @param pluginName: The registered plug-in name of the
            C{ViewportWidget}.
        @param viewportName: The easily identifiable name of the viewport
            which can be used to look-up the viewport in the C{ViewerDelegate}.
        """
        # pylint: disable=unused-argument
        def removeViewport(viewportRef):
            self.__viewportWidgets.remove(viewportRef)

        self.__viewportWidgets.append(
            weakref.ref(viewportWidget, removeViewport))

        # Add the viewport layer if necessary
        if self.__drawExampleSpotlightEnabled():
            layer = viewportWidget.insertLayer(
                'ExampleSpotlightLocatorViewportLayer',
                self.__viewportLayerName, 0)
            layer.setOptionByName('vdc_name',
                                  FnAttribute.StringAttribute(self.__vdcName))

            enlargedLocators = self.__enlargeLocatorsAction.isChecked()
            layer.setOptionByName('enlarged_locators',
                                  FnAttribute.IntAttribute(enlargedLocators))

    def onApplyTerminalOps(self, txn, inputOp, viewerDelegate):
        """
        Callback triggered when the chain of terminal Ops is being created for
        the given viewer delegate, for example after setting the view flag on a
        node.

        @type txn: C{FnGeolib.GeolibRuntime.Transaction}
        @type inputOp: C{FnGeolib.GeolibRuntimeOp} or C{None}
        @type viewerDelegate: C{ViewerAPI.ViewerDelegate}
        @rtype: C{FnGeolib.GeolibRuntimeOp}
        @param txn: The current Geolib3 tansaction.
        @param inputOp: The last Op in the current Op chain.
        @param viewerDelegate: The viewer delegate for which terminal Ops are
            applied.
        @return: The terminal Op to use in the Op chain for the viewer. Should
            be the given input Op, if no changes to the Op chain are required.
        """
        # pylint: disable=unused-argument

        outputOp = inputOp

        if self.__drawExampleSpotlightEnabled():
            # Add a terminal op that sets the name of the VDC as an attribute
            # on the affected locations. This is only really to cause a
            # re-cook of the locations when the VDC is added or removed,
            # which allows other VDCs to handle the location

            overrideLocators = self.__overrideStandardLocatorsAction.isChecked()
            argsBuilder = FnGeolibServices.OpArgsBuilders.AttributeSet()
            argsBuilder.setCEL([
                '/root/world//*{attr("viewer.lightType") == "spot" }'])
            argsBuilder.setAttr('viewer.vdc.%s.overrideStandardLocators'
                                % self.__vdcName,
                                FnAttribute.IntAttribute(overrideLocators))
            vdcInfoOp = txn.createOp()
            txn.setOpArgs(vdcInfoOp, 'AttributeSet', argsBuilder.build())
            txn.setOpInputs(vdcInfoOp, [outputOp])
            outputOp = vdcInfoOp

        if (self.__drawSpotlightsInCyanAction is not None
                and self.__drawSpotlightsInCyanAction.isChecked()):
            # Add a terminal op to override the color of spot lights
            argsBuilder = FnGeolibServices.OpArgsBuilders.AttributeSet()
            argsBuilder.setCEL([
                '/root/world//*{attr("viewer.lightType") == "spot" '
                'and not hasattr("viewer.default.drawOptions.color")}'])
            argsBuilder.setAttr('viewer.default.drawOptions.color',
                                FnAttribute.FloatAttribute([0, 1, 1]))
            attributeSetOp = txn.createOp()
            txn.setOpArgs(attributeSetOp, 'AttributeSet', argsBuilder.build())
            txn.setOpInputs(attributeSetOp, [outputOp])

            outputOp = attributeSetOp

        return outputOp

    # Private Slots -----------------------------------------------------------

    def __drawExampleSpotlightEnabled(self):
        """
        Returns True if the 'Draw Example Spotlight Locators' menu item is
        checked, otherwise False.
        """
        return (self.__drawExampleSpotlightLocatorsAction is not None
                and self.__drawExampleSpotlightLocatorsAction.isChecked())

    def __reapplyTerminalOps(self):
        """
        Force the onApplyTerminalOps() to be called again, by re-setting the
        current view node.
        """
        if self.__tab is None or self.__viewerDelegateRef is None:
            return

        delegate = self.__viewerDelegateRef()
        if delegate is None:
            return

        # Set the view node
        viewNode = NodegraphAPI.GetViewNode()
        self.__tab.setViewedNode(viewNode, delegate)

    def __on_drawExampleSpotlightLocatorsAction_toggled(self, state):
        """
        Slot that is called when the action for adding the example spotlight
        locators to a viewer tab  has been toggled.

        @type state: C{bool}
        @param state: Flag that states whether the menu item is turned on or
            off.
        """
        if self.__viewerDelegateRef is None:
            return

        delegate = self.__viewerDelegateRef()
        if delegate is None:
            return

        # Update the enabled state of menu actions
        self.__overrideStandardLocatorsAction.setEnabled(state)
        self.__enlargeLocatorsAction.setEnabled(state)

        if state:
            # Adds the required viewer delegate component and viewport layers
            # for drawing of the example spotlight locator
            self.__viewerDelegateComponent = \
                delegate.addComponent('ExampleSpotlightLocatorVDC',
                                      self.__vdcName)

            # Ensure the override state is correct
            overrideLocators = self.__overrideStandardLocatorsAction.isChecked()
            enlargedLocators = self.__enlargeLocatorsAction.isChecked()
            self.__viewerDelegateComponent.setOptionByName(
                'overrideStandardLocators',
                FnAttribute.IntAttribute(overrideLocators))

            # Add the layer for each viewport
            for viewportRef in self.__viewportWidgets:
                viewport = viewportRef()
                if viewport is None:
                    continue

                layer = viewport.insertLayer(
                    'ExampleSpotlightLocatorViewportLayer',
                    self.__viewportLayerName, 0)
                layer.setOptionByName('vdc_name',
                    FnAttribute.StringAttribute(self.__vdcName))
                layer.setOptionByName('enlarged_locators',
                    FnAttribute.IntAttribute(enlargedLocators))
        else:
            # Removes the viewport layers and viewer delegate component
            for viewportRef in self.__viewportWidgets:
                viewport = viewportRef()
                if viewport is None:
                    continue

                viewport.removeLayer(self.__viewportLayerName)
            delegate.removeComponent(self.__vdcName)

        self.__reapplyTerminalOps()

    def __on_drawSpotlightsInCyanAction_toggled(self, state):
        """
        Slot that is called when the action for drawing spotlights in a viewer
        tab in a cyan color has been toggled.

        @type state: C{bool}
        @param state: Flag that states whether the menu item is turned on or
            off.
        """
        # pylint: disable=unused-argument
        self.__reapplyTerminalOps()

    def __on_overrideStandardLocatorsAction_toggled(self, state):
        """
        Slot that is called when the action for overriding the standard
        Spotlight Locators with the Example Spotlight Locator has been toggled.

        @type state: C{bool}
        @param state: Flag that states whether the menu item is turned on or
            off.
        """
        if self.__viewerDelegateComponent is None:
            return

        self.__viewerDelegateComponent.setOptionByName(
            'overrideStandardLocators',
            FnAttribute.IntAttribute(state))

        self.__reapplyTerminalOps()

    def __on_enlargeLocatorsAction_toggled(self, state):
        """
        Informs the viewports whether they should draw the Example
        Spotlight Locator with an enlarged line width.
        """
        for viewportRef in self.__viewportWidgets:
            viewport = viewportRef()
            if viewport is None:
                continue

            layer = viewport.getLayer(self.__viewportLayerName)
            if layer:
                layer.setOptionByName('enlarged_locators',
                                      FnAttribute.IntAttribute(state))


PluginRegistry = [
    ('ViewerPluginExtension', 1, 'ExampleSpotlightLocatorVPE',
     ExampleSpotlightLocatorVPE),
]
