// Copyright (c) 2018 The Foundry Visionmongers Ltd. All Rights Reserved.

#ifndef _EXAMPLESPOTLIGHTLOCATOR_H_
#define _EXAMPLESPOTLIGHTLOCATOR_H_

#include <string>
#include <vector>

#include <FnViewer/utils/FnBaseLocator.h>

/**
 * Class implementing a ViewerDelegateComponent which inherrits most of its
 * behaviour from FnBaseLocatorVDC. The only difference is that this class
 * can accept an IntAttribute via setOption() which determines whether a
 * locationEvent() should mark an event as handled when it successfully
 * matches a locator to the event. This works like a global switch the
 * to allow locators to either be instead of the default implementations
 * or in addition to them.
 */
class ExampleSpotlightLocatorVDC : public FnKat::ViewerUtils::FnBaseLocatorVDC
{
public:
    virtual ~ExampleSpotlightLocatorVDC() {}

    static void flush() {}

    static FnKat::ViewerAPI::ViewerDelegateComponent* create()
    {
        return new ExampleSpotlightLocatorVDC();
    }

    /// Handles viewer location events and maintains the SceneNode tree.
    bool locationEvent(
        const Foundry::Katana::ViewerAPI::ViewerLocationEvent& event,
        bool locationHandled) override;

    /// Sets generic options
    void setOption(
        FnKat::ViewerAPI::OptionIdGenerator::value_type optionId,
        FnAttribute::Attribute attr) override;

private:
    ExampleSpotlightLocatorVDC();

    bool m_overrideStandardLocators;
};

/**
 * Class implementing a ViewportLayer derives most of its behavior from
 * FnBaseLocatorViewportLayer. The difference here is that it can accept
 * an additional IntAttribute option via setOption() which allows it to
 * increase the line thickness of its Locator plug-ins.
 */
class ExampleSpotlightLocatorViewportLayer :
    public FnKat::ViewerUtils::FnBaseLocatorViewportLayer
{
public:
    virtual ~ExampleSpotlightLocatorViewportLayer() {}

    static FnKat::ViewerAPI::ViewportLayer* create()
    {
        return new ExampleSpotlightLocatorViewportLayer();
    }

    /// Draws the viewport layer's contents.
    virtual void draw() override;

    /// Sets a generic option.
    virtual void setOption(
        FnKat::ViewerAPI::OptionIdGenerator::value_type optionId,
        FnAttribute::Attribute attr) override;

private:
    ExampleSpotlightLocatorViewportLayer();

    bool m_enlargedLocators;
};


/// Example of a class implementing a locator plug-in that draws a cone at
/// matching light locations.
class ExampleSpotlightLocator : public FnKat::ViewerUtils::FnBaseLocator
{
public:
    ExampleSpotlightLocator() : m_vao(0), m_ebo(0), m_vbo(0) {}
    virtual ~ExampleSpotlightLocator() {}

    static FnKat::ViewerUtils::FnBaseLocator* create()
    {
        return new ExampleSpotlightLocator();
    }

    /// Determines whether this locator plug-in should be used to draw the
    /// location described in the given viewer location event.
    static bool matches(const FnKat::ViewerAPI::ViewerLocationEvent& event);

    /// Determines whether this locator plug-in overrides the standard geometry
    /// representation for matched locations.
    static bool overridesBaseGeometry(
        const FnKat::ViewerAPI::ViewerLocationEvent& event);

    /// Gets the bounds of the given location.
    static FnAttribute::DoubleAttribute getBounds(
        const FnKat::ViewerAPI::ViewerLocationEvent& event);

    /// Computes the extent of the given location.
    static FnAttribute::DoubleAttribute computeExtent(
        const FnKat::ViewerAPI::ViewerLocationEvent& event);

    /// Initializes the locator plug-in's resources.
    virtual void setup();
    /// Cleans up the locator plug-in's resources.
    virtual void cleanup();

    /// Draws the object that's represented by the scene graph location with
    /// the given path using the given shader program.
    virtual void draw(const std::string& locationPath);

    /// Draws the object that's represented by the scene graph location with
    /// the given path using the given shader program as part of a picking
    /// pass.
    virtual void pickerDraw(
        const std::string& locationPath);

    /// Returns the vertex and index data for a cone.
    void generateCone(float radius,
                      float height,
                      int segments,
                      std::vector<float>& vertices,
                      std::vector<unsigned int>& indices);

private:
    /// Helper function that draws a light geometry.
    void drawLight(const std::string& locationPath);

    GLuint m_vao;

    /// The Element Buffer Object containing indices to vertices.
    GLuint m_ebo;

    /// The Vertex Buffer Object containing vertices.
    GLuint m_vbo;

    std::vector<float> m_vertices;
    std::vector<unsigned int> m_indices;
};

#endif
