# Copyright (c) 2017 The Foundry Visionmongers Ltd. All Rights Reserved.


from Katana import (
    FnAttribute,
    FnGeolibServices,
)
from PluginAPI.BaseViewerPluginExtension import BaseViewerPluginExtension

class BallAppender(BaseViewerPluginExtension):
    """
    This class is responsible for automatically adding the 'BallComponent' 
    ViewerDelegateComponent to a viewer delegate, and also for adding the
    C{BallLayer} to viewports in order to draw ball locations.

    It additionally adds a terminal Op which provides a default radius for
    ball locations, but in reality, this is best left as a default value
    within the C{BallComponent} plug-in.
    """

    def onDelegateCreated(self, viewerDelegate, pluginName):
        """
        Adds the C{BallComponent} VDC to the new viewer delegate.

        @type viewerDelegate: C{ViewerDelegate}
        @type pluginName: C{str}
        @param viewerDelegate: The C{ViewerDelegate} that was created.
        @param pluginName: The registered plug-in name of the C{ViewerDelegate}.
        """
        viewerDelegate.addComponent('BallComponent', 'BallComponent')

    def onViewportCreated(self, viewportWidget, pluginName, viewportName):
        """
        Adds the C{BallLayer} viewport layer to the new viewport.

        @type viewportWidget: C{ViewportWidget}
        @type pluginName: C{str}
        @type viewportName: C{str}
        @param viewportWidget: The C{ViewportWidget} that was created.
        @param pluginName: The registered plug-in name of the C{ViewportWidget}.
        @param viewportName: The easily identifiable name of the viewport
            which can be used to look-up the viewport in the C{ViewerDelegate}.
        """
        BaseViewerPluginExtension.onViewportCreated(self, viewportWidget, pluginName, viewportName)
        viewportWidget.insertLayer('BallLayer', 'BallLayer', 4)

    def onApplyTerminalOps(self, txn, inputOp, viewerDelegate):
        """
        Callback triggered when the chain of terminal Ops is being created for
        the given viewer delegate, for example after setting the view flag on a
        node.

        Appends an AttributeSet Op to the Op chain that currently ends on the
        given input Op, to set the B{geometry.radius} attribute on any B{ball}
        locations that do not already have a radius set.

        @type txn: C{FnGeolib.GeolibRuntime.Transaction}
        @type inputOp: C{FnGeolib.GeolibRuntimeOp} or C{None}
        @type viewerDelegate: C{ViewerAPI.ViewerDelegate}
        @rtype: C{FnGeolib.GeolibRuntimeOp}
        @param txn: The current Geolib3 tansaction.
        @param inputOp: The last Op in the current Op chain.
        @param viewerDelegate: The viewer delegate for which terminal Ops are
            applied.
        @return: The terminal Op to use in the Op chain for the viewer. Should
            be the given input Op, if no changes to the Op chain are required.
        """
        terminalOp = txn.createOp()
        argsBuilder = FnGeolibServices.OpArgsBuilders.AttributeSet()
        argsBuilder.setCEL([
            '/root/world//*{attr("type") == "ball" '
            'and not hasattr("geometry.radius")}'])
        argsBuilder.setAttr('geometry.radius',
            FnAttribute.DoubleAttribute(5.0))
        txn.setOpArgs(terminalOp, 'AttributeSet', argsBuilder.build())
        txn.setOpInputs(terminalOp, [inputOp])

        return terminalOp


PluginRegistry = [
    ("ViewerPluginExtension", 1, "BallAppender", BallAppender),
]
