import NodegraphAPI
from Katana import LayeredMenuAPI

def PopulateCallback(layeredMenu, tab=None):
    """
    Callback for the layered menu, which adds entries to the given
    C{layeredMenu} based on the primitive types available to the
    PrimitiveCreate node.

    @type layeredMenu: L{LayeredMenuAPI.LayeredMenu}
    @type tab: C{NodeGraphTab.NodegraphPanel.NodegraphPanel}
    @param layeredMenu: The layered menu to add entries to.
    @param tab: The B{Node Graph} tab where the entry was chosen.
    """
    _ = tab

    # Nodes3DAPI module is imported explicitly to access PrimitiveCreate.
    import Nodes3DAPI
    # Obtain a list of names of available primitives from the PrimitiveCreate
    # Node type's parameter hints
    primitiveHints = Nodes3DAPI.PrimitiveCreate._ExtraHints
    primitiveTypeHints = primitiveHints.get('PrimitiveCreate.type',{})
    primitiveTypes = primitiveTypeHints.get('options')
    if primitiveTypes is None:
        return
    primitiveTypes = primitiveTypes.split('|')
    # Iterate over the primitve types and add a menu entry for each of
    # them to the given layered menu, using a green-ish color
    for primitiveType in primitiveTypes:
        layeredMenu.addEntry(primitiveType, text=primitiveType,
                             color=(0.3, 0.7, 0.3))

def ActionCallback(value, tab=None):
    """
    Callback for the layered menu, which creates a PrimitiveCreate node and
    sets its B{type} parameter to the given C{value}, which is the name of
    a primitive as set for the menu entry in L{PopulateCallback()}.

    @type value: C{str}
    @type tab: C{NodeGraphTab.NodegraphPanel.NodegraphPanel}
    @rtype: C{object}
    @param value: An arbitrary object that the menu entry that was chosen
        represents. In our case here, this is the name of a primitive as
        passed to the L{LayeredMenuAPI.LayeredMenu.addEntry()} function in
        L{PopulateCallback()}.
    @param tab: The B{Node Graph} tab where the entry was chosen.
    @return: An arbitrary object. In our case here, we return the created
        PrimitiveCreate node, which is then placed in the B{Node Graph} tab
        because it is a L{NodegraphAPI.Node} instance.
    """
    _ = tab

    # Create the node, set its type, and set the name.
    node = NodegraphAPI.CreateNode('PrimitiveCreate')
    node.getParameter('type').setValue(value, 0)
    node.setName(value)
    node.getParameter('name').setValue('/root/world/geo/' + node.getName(), 0)
    return node


# Create and register a layered menu using the above callbacks
layeredMenu = LayeredMenuAPI.LayeredMenu(PopulateCallback, ActionCallback,
                                         'P', alwaysPopulate=False,
                                         onlyMatchWordStart=False)
LayeredMenuAPI.RegisterLayeredMenu(layeredMenu, 'Primitives')

