# Copyright (c) 2015 The Foundry Visionmongers Ltd. All Rights Reserved.
"""
Example script that registers a layered menu for the B{Node Graph} tab, which
shows the names of available PRMan shaders and creates a PrmanShadingNode node
with the chosen shader set on it when one of the menu entries is chosen.
"""

import logging

import NodegraphAPI
from Katana import (
    LayeredMenuAPI,
    RenderingAPI,
)
from RenderingAPI import RenderPlugins


log = logging.getLogger('CustomLayeredMenuExample')

def PopulateCallback(layeredMenu, tab):
    """
    Callback for the layered menu, which adds entries to the given
    C{layeredMenu} based on the available PRMan shaders.

    @type layeredMenu: L{LayeredMenuAPI.LayeredMenu}
    @type tab: C{NodeGraphTab.NodegraphPanel.NodegraphPanel}
    @param layeredMenu: The layered menu to add entries to.
    @param tab: The B{Node Graph} tab where the entry was chosen.
    """
    _ = tab

    # Obtain a list of names of available PRMan shaders from the PRMan renderer
    # info plug-in
    rendererInfoPlugin = RenderPlugins.GetInfoPlugin('prman')
    shaderType = RenderingAPI.RendererInfo.kRendererObjectTypeShader
    shaderNames = rendererInfoPlugin.getRendererObjectNames(shaderType)

    # Iterate over the names of shaders and add a menu entry for each of them
    # to the given layered menu, using a light blue color
    for shaderName in shaderNames:
        layeredMenu.addEntry(shaderName, text=shaderName,
                             color=(0.35, 0.57, 1.0))

def ActionCallback(value, tab):
    """
    Callback for the layered menu, which creates a PrmanShadingNode node and
    sets its B{nodeType} parameter to the given C{value}, which is the name of
    a PRMan shader as set for the menu entry in L{PopulateCallback()}.

    @type value: C{str}
    @type tab: C{NodeGraphTab.NodegraphPanel.NodegraphPanel}
    @rtype: C{object}
    @param value: An arbitrary object that the menu entry that was chosen
        represents. In our case here, this is the name of a PRMan shader as
        passed to the L{LayeredMenuAPI.LayeredMenu.addEntry()} function in
        L{PopulateCallback()}.
    @param tab: The B{Node Graph} tab where the entry was chosen.
    @return: An arbitrary object. In our case here, we return the created
        PrmanShadingNode node, which is then placed in the B{Node Graph} tab
        because it is a L{NodegraphAPI.Node} instance.
    """
    _ = tab

    # Create the node, set its shader, and set the name with the shader name
    node = NodegraphAPI.CreateNode('PrmanShadingNode')
    node.getParameter('nodeType').setValue(value, 0)
    node.setName(value)
    node.getParameter('name').setValue(node.getName(), 0)
    return node


# Create and register a layered menu using the above callbacks

rendererInfoPlugin = RenderPlugins.GetInfoPlugin('prman')
if rendererInfoPlugin is not None:
    layeredMenu = LayeredMenuAPI.LayeredMenu(PopulateCallback, ActionCallback,
                                             'Alt+P', alwaysPopulate=False,
                                             onlyMatchWordStart=False)
    layeredMenu.setAssociatedRenderer("prman")
    LayeredMenuAPI.RegisterLayeredMenu(layeredMenu, 'PrmanShaders')
else:
    log.warning('The "PrmanShaders" layered menu [Alt+P] is not going to be '
                'available in the Node Graph tab, as the "prman" renderer '
                'plug-in is not available.')

