# Copyright (c) 2012 The Foundry Visionmongers Ltd. All Rights Reserved.


from PyQt5 import (
    QtCore,
    QtWidgets,
)

import QT4Widgets
import Utils
from Katana import (
    UI4,
    NodegraphAPI,
)
from . import ScriptActions as SA

# This editor shows three different types of UI:
#  1) A standard FormWidget is used for the 'location' parameter.
#     It automatically tracks changes to the parameter.
#  2) A custom tree view shows the list of ponies. We are responsible for
#     rebuilding this when the internal network changes.
#  3) A FormWidget structure is used to expose the 'transform' parameters of
#     the selected pony's internal node. We track the selection, but the
#     FormWidgets take care of syncing with the internal parameters.

#//////////////////////////////////////////////////////////////////////////////
#///
class PonyStackEditor(QtWidgets.QWidget):
    def __init__(self, parent, node):
        node.upgrade()
        
        self.__node = node
        
        QtWidgets.QWidget.__init__(self, parent)
        QtWidgets.QVBoxLayout(self)
        
        self.__frozen = True
        self.__updateTreeOnIdle = False
        self.__selectedPonyPolicy = None
        self.__preselectName = None

        # location FormWidget
        locationPolicy = UI4.FormMaster.CreateParameterPolicy(
            None, self.__node.getParameter('location'))
        factory = UI4.FormMaster.ParameterWidgetFactory
        w = factory.buildWidget(self, locationPolicy)
        self.layout().addWidget(w)

        # toolbar
        self.__toolbarLayout = QtWidgets.QHBoxLayout()
        self.layout().setSpacing(0)
        self.layout().addItem(self.__toolbarLayout)

        self.__addButton = UI4.Widgets.ToolbarButton(
            'Add Pony', self,
            UI4.Util.IconManager.GetPixmap('Icons/plus16.png'),
            rolloverPixmap = UI4.Util.IconManager.GetPixmap('Icons/plusHilite16.png'))
        self.__addButton.clicked.connect(self.__addButtonClicked)
        self.__toolbarLayout.addWidget(self.__addButton)
        self.__toolbarLayout.addStretch()

        # tree widget
        self.__treeWidget = QT4Widgets.SortableTreeWidget(self)
        self.__treeWidget.setHeaderLabels(['Name'])
        self.__treeWidget.setSelectionMode(QtWidgets.QTreeWidget.SingleSelection)
        self.__treeWidget.setAllColumnsShowFocus(True)
        self.__treeWidget.setRootIsDecorated(False)
        self.__treeWidget.header().setSectionResizeMode(QtWidgets.QHeaderView.ResizeToContents)
        self.__treeWidget.header().setSectionsClickable(False)

        self.__treeWidget.itemSelectionChanged.connect(self.__treeWidgetSelectionChanged)
        self.__treeWidget.keyPressEventSignal.connect(
            self.__treeWidgetKeyPressCallback)

        self.layout().addWidget(self.__treeWidget)
        self.layout().addWidget(
            UI4.Widgets.VBoxLayoutResizer(self.__treeWidget, 120))
        self.layout().addSpacing(1)

        self.__parameterWidgetLayout = QtWidgets.QVBoxLayout()
        self.layout().addLayout(self.__parameterWidgetLayout)

        self.__updateTreeContents()

        self.layout().addStretch()
        
    
    # We thaw/freeze the UI when it is shown/hidden.  This means that we aren't
    # wasting CPU cycles by listening and responding to events when the editor
    # is not active.
    def showEvent(self, event):
        QtWidgets.QWidget.showEvent(self, event)
        if self.__frozen:
            self.__frozen = False
            self._thaw()
    
    def hideEvent(self, event):
        QtWidgets.QWidget.hideEvent(self, event)
        if not self.__frozen:
            self.__frozen = True
            self._freeze()
    
    def _thaw(self):
        self.__setupEventHandlers(True)
    
    def _freeze(self):
        self.__setupEventHandlers(False)

    def __setupEventHandlers(self, enabled):
        Utils.EventModule.RegisterEventHandler(self.__idle_callback,
            'event_idle', enabled=enabled)

        Utils.EventModule.RegisterCollapsedHandler(self.__updateCB,
            'port_connect', enabled=enabled)
        Utils.EventModule.RegisterCollapsedHandler(self.__updateCB,
            'port_disconnect', enabled=enabled)
        Utils.EventModule.RegisterCollapsedHandler(self.__updateCB,
            'parameter_finalizeValue', enabled=enabled)

    def __updateCB(self, args):
        if self.__updateTreeOnIdle:
            return

        for arg in args:
            if arg[0] in ('port_connect', 'port_disconnect'):
                for nodeNameKey in 'nodeNameA', 'nodeNameB':
                    nodeName = arg[2][nodeNameKey]
                    node = NodegraphAPI.GetNode(nodeName)
                    if node is not None and node.getParent() == self.__node:
                        self.__updateTreeOnIdle = True
                        return
            
            if arg[0] in ('parameter_finalizeValue'):
                node = arg[2].get('node')
                param = arg[2].get('param')
                if node.getParent() == self.__node and param == node.getParameter('name'):
                    self.__updateTreeOnIdle = True
                    return


    
    def __idle_callback(self, *args, **kwargs):
        if self.__updateTreeOnIdle:
            self.__updateTreeContents()
            self.__updateTreeOnIdle = False

    def __addButtonClicked(self):
        name, index = self.__node.addPony('pony')
        self.__preselectName = name

    def __updateTreeContents(self):
        scrollToPreselect = False
        scrollToItem = None
        vScrollbar = self.__treeWidget.verticalScrollBar()
        if vScrollbar and vScrollbar.isVisible():
            yPos = vScrollbar.value()
        else:
            vScrollbar = None

        try:
            self.__rebuilding = True
            node = self.__node
            
            selectedTable, openTable = self.__treeWidget.getExpandedAndSelectionTables()
            
            if self.__preselectName:
                scrollToPreselect = True
                selectedNames = set([self.__preselectName])
                self.__preselectName = None
            else:
                selectedNames = set([str(x.text(0))
                                     for x in selectedTable.values()])

            self.__treeWidget.clear()

            ponyNames = node.getPonyNames()
            for ponyName in ponyNames:
                item = QT4Widgets.SortableTreeWidgetItem(
                    self.__treeWidget, ponyName, data=None)
                selected = ponyName in selectedNames
                item.setSelected(selected)
                if scrollToPreselect and selected:
                    scrollToItem = item

        finally:
            if scrollToItem:
                self.__treeWidget.scrollToItem(scrollToItem)
            else:
                if vScrollbar:
                    vScrollbar.setValue(yPos)
            
            self.__rebuilding = False

        self.__treeWidgetSelectionChanged()

    def __treeWidgetSelectionChanged(self):
        if self.__rebuilding: return

        selection = self.__treeWidget.selectedItems()
        selectedPonyNode = None
        if selection:
            index = self.__treeWidget.indexOfTopLevelItem(selection[0])
            selectedPonyNode = SA.GetPonyNode(self.__node, index)

        selectedPonyParam = None
        if selectedPonyNode:
            selectedPonyParam = selectedPonyNode.getParameter('transform')

        if self.__selectedPonyPolicy and \
           self.__selectedPonyPolicy.getParameter() == selectedPonyParam:
            return

        self.__selectedPonyPolicy = None

        # Clear the layout of parameter widgets
        while self.__parameterWidgetLayout.count():
            layoutItem = self.__parameterWidgetLayout.takeAt(0)
            if isinstance(layoutItem, QtWidgets.QWidgetItem):
                widget = layoutItem.widget()
                widget.deleteLater()

        if selectedPonyParam:
            self.__selectedPonyPolicy = UI4.FormMaster.CreateParameterPolicy(None,
                selectedPonyParam)
            factory = UI4.FormMaster.ParameterWidgetFactory
            parameterWidget = factory.buildWidget(self,
                                                  self.__selectedPonyPolicy)
            self.__parameterWidgetLayout.addWidget(parameterWidget)


    def __treeWidgetKeyPressCallback(self, event):
        if event.isAccepted(): return
        
        if event.key() == QtCore.Qt.Key_Delete:
            event.accept()

            selection = self.__treeWidget.selectedItems()
            if selection:
                index = self.__treeWidget.indexOfTopLevelItem(selection[0])
                self.__node.deletePony(index)


# The following code shows how to register a custom NodeActionDelegate for
# the PonyStack node.
# NodeActionDelegates are subclasses of BaseNodeActionDelegate and allow to
# add QActions to the node's context menu and wrench menu

from UI4.FormMaster.NodeActionDelegate import BaseNodeActionDelegate

class PonyStackActionDelegate(BaseNodeActionDelegate.BaseNodeActionDelegate):
    class _AddPony(QtWidgets.QAction):
        def __init__(self, parent, node):
            QtWidgets.QAction.__init__(self, "Add Pony", parent)
            self.__node = node
            if node:
                self.triggered.connect(self.__triggered)
        def __triggered(self, checked):
            self.__node.addPony('pony')

    def addToWrenchMenu(self, menu, node, hints=None):
        menu.addAction(self._AddPony(menu, node))

    def addToContextMenu(self, menu, node):
        menu.addAction(self._AddPony(menu, node))


UI4.FormMaster.NodeActionDelegate.RegisterActionDelegate("PonyStack", PonyStackActionDelegate())
