# Copyright (c) 2020 The Foundry Visionmongers Ltd. All Rights Reserved.
"""
Module containing an implementation of L{FarmAPI.BaseFarmPlugin} that provides
a bridge to I{OpenCue}, an open source render management system.
"""


import getpass
import logging
import os.path
import shlex
import shutil
import signal
import sys

from Callbacks import Callbacks
from Katana import (
    Configuration,
    FarmAPI,
    Nodes2DAPI,
    NodegraphAPI
)

from UI4.KatanaPrefs import Prefs, PrefNames

from RenderingAPI import RenderPlugins

log = logging.getLogger(__name__)


# Import PyCue and PyOutline.
#
# Import Mechanism
# Note: the instructions below provide a mechanism for adding additional
# third-party Python packages to Katana's PYTHONPATH. Other techniques work
# so please replace this section with whatever is used by your facility.
#
# 1. Create a separate Python virtualenv that uses Katana's builtin Python
#    interpreter as its 'base'
#
#    virtualenv venv -p $KATANA_ROOT/bin/pythonX.Y/bin/python
#    source venv/bin/activate
#
# 2. Install PyCue and PyOutline into the virtualenv. If your facility has
#    made source changes to PyCue or PyOutline, install these from your
#    internal Python package management repository. Otherwise, follow the
#    instructions here:
#    https://www.opencue.io/docs/getting-started/installing-pycue-and-pyoutline
#
# 3. Set the environment variable OPENCUE_INSTALL_DIR to the site-packages
#    folder in the virtualenv created in step 1 (usually
#    venv/lib/pythonX.Y/site-packages)

def ConfigureOpenCuePaths():
    """
    Adds the path specified by the environment variable OPENCUE_INSTALL_DIR
    to the list of site packages used by Katana.
    """
    import site
    openCueSitePackage = os.getenv('OPENCUE_INSTALL_DIR')
    if not openCueSitePackage:
        log.error('OPENCUE_INSTALL_DIR not specified')
        return

    site.addsitedir(openCueSitePackage)


ConfigureOpenCuePaths()

# pylint: disable=wrong-import-position
import opencue
from opencue.wrappers.job import Job as OpenCueJob

from outline import Outline
from outline.cuerun import OutlineLauncher
from outline.modules.shell import Shell

# Increasing counter to differentiate jobs
_RENDER_SEQ_NUM = 0


def GetJobNameSuffix():
    """
    @rtype: C{str}
    @return: A suffix appended to the job name prior to submission to the cue.
        This suffix should ensure that each job name submitted to the queue
        is unique to allow for batch submission/wedging scripts to submit many
        jobs from the same project file.
    """
    # pylint: disable=global-statement
    global _RENDER_SEQ_NUM
    sequence = _RENDER_SEQ_NUM
    _RENDER_SEQ_NUM += 1

    processID = os.getpid()
    return '{pid}_{seq}'.format(pid=processID, seq=sequence)


def GetJobName():
    """
    @rtype: C{str}
    @return: A name for the Job that will be submitted to the cue. Customize
        this function to return a job name in the format expected by your
        facility.
    """
    # Include the project name
    projectName = os.path.basename(NodegraphAPI.GetProjectFile()) or 'Untitled'

    # Include the artist's user name
    artistLogin = getpass.getuser()

    return '{project}_{artist}_{suffix}'.format(
        project=projectName,
        artist=artistLogin,
        suffix=GetJobNameSuffix()
    )


def GetRemoteKatanaRoot():
    """
    @rtype: C{str}
    @return: Path the Katana installation on the render nodes.
    """
    # In this example we assume Katana is installed in the same location
    # across the facility (such as on a network mounted drive)
    return Configuration.get('KATANA_ROOT')


def GetRemoteRenderbootPath():
    """
    @rtype: C{str}
    @return: Path to renderboot on the remote render nodes. This allows users
        to customise the renderboot that is run on the render node if the
        default is not used.
    """
    return os.path.join(GetRemoteKatanaRoot(), 'bin', 'renderboot')


def GetRemotePluginDsoPath(rendererName):
    """
    @rtype: C{str}
    @return: Path to the renderer plugin on the remote render nodes.
    """
    # In this example we assume Katana is installed in the same location
    # across the facility (such as on a network mounted drive)
    return RenderPlugins.GetRendererPluginDir(rendererName)


# Class Definitions -----------------------------------------------------------

class OpenCueFarmPlugin(FarmAPI.BaseFarmPlugin):
    """
    Implementation of L{FarmAPI.BaseFarmPlugin} that enables Katana to send
    remote render jobs to an I{OpenCue} deployment.
    """

    # Mapping of OpenCue job state to FarmAPI job state.
    # The OpenCue job states are described here, and suggest POSTED is a state
    # just before PENDING.
    # https://github.com/AcademySoftwareFoundation/OpenCue/blob/f347994faac0be9daa5381880993adb45d935a06/proto/job.proto
    kJobStateMap = {
        OpenCueJob.JobState.PENDING: FarmAPI.Job.State.kRunning,
        OpenCueJob.JobState.FINISHED: FarmAPI.Job.State.kCompleted,
        OpenCueJob.JobState.STARTUP: FarmAPI.Job.State.kRunning,
        OpenCueJob.JobState.SHUTDOWN: FarmAPI.Job.State.kRunning,
        OpenCueJob.JobState.POSTED: FarmAPI.Job.State.kWaiting
    }

    kStoppedStates = (
        OpenCueJob.JobState.FINISHED,
    )

    kRunningStates = (
        OpenCueJob.JobState.PENDING,
        OpenCueJob.JobState.STARTUP,
        OpenCueJob.JobState.SHUTDOWN
    )

    # Name of the OpenCue service that corresponds to Katana.
    kOpenCueServiceName = 'katana'

    # Name of the user to run the render jobs as on the render nodes.
    #
    # Note: In this example we assume all render jobs on the farm are run as
    # the OS user kOpenCueRenderUser. However, this could be changed to the
    # artist's OS login if your facility has set up network logins on your
    # farm machines.
    kOpenCueRenderUser = 'pixelmaker'

    # The OpenCue Database schema by default has a 4000 character limit on
    # commands, so we check that we're not sending commands longer than that
    # here.
    kOpenCueMaxCommandLength = 4000

    # Initializer -------------------------------------------------------------

    def __init__(self):
        """
        Initializes an instance of the class.
        """
        self.__cueFsRoot = os.environ['CUE_FS_ROOT']
        self.__cueFacility = os.environ['KATANA_CUE_DEFAULT_FACILITY']

        # If true, while OpenCueFarmPlugin is monitoring currently active jobs,
        # if it detects a frame has started to fail or has died it will
        # set the 'eatDeadFrames' property on the job to true, kill the job
        # and return a kFailed job state to Katana.
        self.__eatDeadFrames = os.getenv(
            'KATANA_CUE_EAT_DEAD_FRAMES', True)

    # Instance Methods --------------------------------------------------------

    def getRemoteInstallationMetadata(self):
        """
        @rtype: L{BaseFarmPlugin.InstallationMetadata}
        @return: An instance of C{InstallationMetadata} containing the details
            of the remote installation of Katana.
        """
        # Build a mapping with names of registered render plug-ins as keys, and
        # absolute paths to the folders containing the renderers` DSOs as
        # values, e.g.
        # {
        #     'dl': ('/usr/local/3delight-2.1.4/Linux-x86_64/'
        #            '3DelightForKatana/Libs'),
        #     'prman': ('/opt/pixar/RenderManForKatana-23.3-katana3.5/plugins/'
        #               'Resources/PRMan23/Libs'),
        #     'customRenderer': '/path/to/customRenderer/Libs'
        # }
        renderPluginDsoPaths = {}
        for rendererName in RenderPlugins.GetRendererPluginNames(
                includeViewer=False, includeInternal=True):
            renderPluginDsoPaths[rendererName] = \
                GetRemotePluginDsoPath(rendererName)

        return FarmAPI.BaseFarmPlugin.InstallationMetadata(
            GetRemoteKatanaRoot(),
            renderPluginDsoPaths,
            GetRemoteRenderbootPath()
        )

    def getEnvironmentForJob(self, renderContextName):
        """
        @type renderContextName: C{str}
        @rtype: C{dict} of C{str}
        @param renderContextName: The render context for which to return an
            environment.
        @return: Dictionary of environment variables to be available to the
            process environment during the render job.
        """
        # pylint: disable=unused-argument
        result = {}

        # Use this function to supply any additional environment variables
        # that should be applied to the remote render environment.
        #
        # In this example we check for the presence of an environment variable
        # (locally) that points to the license server used by the render
        # farm.
        farmLicenseServer = os.getenv('FOUNDRY_FARM_LICENSE_SERVER')
        if farmLicenseServer:
            result['foundry_LICENSE'] = farmLicenseServer
        return result

    def _projectDirectoryForContext(self, renderContextName):
        """
        Produces the path to a unique directory for the specified
        C{renderContextName}.

        @type renderContextName: C{str}
        @rtype: C{str}
        @param renderContextName: A globally unique identifier for the render
            job for which to create a temporary directory.
        @return: Full path to a unique directory for the given render context.
        """
        userLogin = getpass.getuser()
        return os.path.join(self.__cueFsRoot, 'jobs', userLogin,
                            renderContextName)

    def createTemporaryDirectory(self, renderContextName):
        """
        Creates a temporary directory accessible by the render nodes to be used
        by the C{renderboot} process.

        Typically an NFS share or other object store would be used.

        @type renderContextName: C{str}
        @rtype: C{str}
        @param renderContextName: The render context for which to create the
            temporary directory.
        @return: The file system location path of the temporary directory that
            was created.
        """
        renderDirectory = self._projectDirectoryForContext(renderContextName)
        os.makedirs(renderDirectory)
        return renderDirectory

    def copyRenderFiles(self, renderFiles, renderContextName):
        """
        Copies the list of files to a location accessible by the render nodes.

        Typically the files will be copied to a subfolder of the temporary
        directory created by L{createTempDirectory()}.

        @type renderFiles: C{list} of C{str}
        @type renderContextName: C{str}
        @rtype: C{map} of C{str} to C{str}
        @param renderFiles: A list of names of files to be copied.
        @param renderContextName: A globally unique ID for the render job for
            which to copy the files.
        @return: A map containing each of the given filenames and the absolute
            path of its respective copy.
        """
        _projectDirectory = self._projectDirectoryForContext(renderContextName)

        result = {}
        for renderFile in renderFiles:
            fileName = os.path.basename(renderFile)
            remoteFile = os.path.join(_projectDirectory, fileName)
            shutil.copy(renderFile, remoteFile)
            result[renderFile] = remoteFile
        return result

    def _extractRendererName(self, commandLine):
        """
        Extracts the renderer plugin name that will be used when the specified
        command line is run.

        @type commandLine: C{str}
        @rtype: C{str} or C{None}
        @param commandLine: The command line that should be parsed.
        @return: The renderer name if one is present in the command line.
        """
        arguments = shlex.split(commandLine)
        for i in range(0, len(arguments)):
            if arguments[i] == '-renderer':
                return arguments[i + 1]
        return None

    def _getTerminationSignalForCommand(self, command):
        # Windows doesn't have the concept of signal termination.
        if sys.platform.startswith('win'):
            return signal.SIGTERM

        terminationSignal = signal.SIGTERM

        # Each renderer may use a different signal to terminate and shutdown
        # cleanly so use the associated environment variable determine which
        # to use.
        renderer = self._extractRendererName(command)
        if not renderer:
            return terminationSignal

        return Nodes2DAPI.GetRenderTerminationSignal(renderer)

    def _getShowAndShot(self):
        """
        Derives the current show and shot for the current Katana session.

        The method will check the following variables defined in the project
        and user environment,

            1. Graph state variables - examine the global Graph State Variables
               B{show} and B{shot}.
            2. Environment variables - examine the environment variables
               C{SHOW} and C{SHOT}.
            3. Return defaults.

        @rtype: 2-tuple of C{str}
        @return: The current show and shot.
        """
        show = 'default_show'
        shot = 'default_shot'

        # Check the graph state variables
        variablesParam = NodegraphAPI.GetRootNode().getParameter('variables')
        if variablesParam:
            showParam = variablesParam.getChild('show')
            shotParam = variablesParam.getChild('shot')
            if showParam and shotParam:
                currentTime = NodegraphAPI.GetCurrentTime()
                show = showParam.getChild('value').getValue(currentTime)
                shot = shotParam.getChild('value').getValue(currentTime)
                return show, shot

        if os.environ['SHOW'] and os.environ['SHOT']:
            show = os.environ['SHOW']
            shot = os.environ['SHOT']

        return show, shot

    def _getFrameSpec(self, jobRequest):
        """
        @type jobRequest: L{FarmAPI.FarmPluginManager.JobRequest}
        @rtype: C{str}
        @param jobRequest: Contains details of the job that should be submitted
            to the farm, such as the commands to be run, frame range, process
            environment, etc.
        @return: The specification for the frames that should be rendered in a
            format that can be understood by OpenCue.
        """
        frames = []
        for frame in jobRequest.frames:
            frames.append(str(int(frame)))
        frameSpecification = ','.join(frames)
        return frameSpecification

    def _prepareRenderCommand(self, command, environment):
        """
        @rtype: C{str}
        @return: Command line that should be run on remotely.
        @raise ValueError: If the generated command is too long to be accepted
            by OpenCue.
        """
        envWrapper = ['env']
        for envVar in environment:
            envWrapper.append(envVar)
        envWrapper.append(command)

        preparedCommand = ' '.join(envWrapper)
        if len(preparedCommand) > OpenCueFarmPlugin.kOpenCueMaxCommandLength:
            errorMessage = 'Cannot submit command to OpenCue as its length ' \
                           '({}) is greater than the maximum allowed ' \
                           'limit ({}). Command ({})'.format(
                               len(preparedCommand),
                               OpenCueFarmPlugin.kOpenCueMaxCommandLength,
                               preparedCommand)
            raise ValueError(errorMessage)

        return preparedCommand

    def submitJob(self, jobRequest):
        """
        Creates and submits a job to the render farm.

        This method will be called to allow the farm plug-in to create a
        description of the render job (using the render farm's own description
        language). This job should then be submitted to the queue management
        server.

        The commands that should be run are pre-processed based on the
        information provided by L{BaseFarmPlugin.InstallationMetadata}.

        @type jobRequest: L{FarmAPI.FarmPluginManager.JobRequest}
        @rtype: C{str}
        @param jobRequest: Contains details of the job that should be submitted
            to the farm, such as the commands to be run, frame range, process
            environment, etc.
        @return: A unique identifier for the Job on the farm.
        @raise Exception: Re-raise any exceptions thrown during the
            job submission process.
        """
        # Submit the job using the PyOutline API
        jobName = GetJobName()

        # Create the render script
        show, shot = self._getShowAndShot()
        outline = Outline(jobName,
                          shot=shot,
                          show=show,
                          user=OpenCueFarmPlugin.kOpenCueRenderUser)

        # Prepare the environment variables
        environment = []
        for key, value in jobRequest.environment.items():
            environment.append('{}={}'.format(key, value))

        # Format the requested frames in a format expected by Outline/OpenCue
        frameSpecification = self._getFrameSpec(jobRequest)

        # Build up the layers for this job based on the commands Katana has
        # asked us to run to render this job.
        previousLayer = None
        for command in jobRequest.commands:
            # Run with required environment variables
            command = self._prepareRenderCommand(command, environment)

            layer = Shell('Katana',
                          command=command.split(),
                          threads=Prefs[PrefNames.RENDERING_THREADS_3D],
                          range=frameSpecification,
                          threadable=True)

            layer.set_service(OpenCueFarmPlugin.kOpenCueServiceName)
            if previousLayer:
                layer.depend_on(previousLayer)

            outline.add_layer(layer)

            previousLayer = layer

        outline.set_facility(self.__cueFacility)

        # Submit the job
        try:
            launcher = OutlineLauncher(outline,
                                       env=environment)
            result = launcher.launch(use_pycuerun=False)
            job = result[0]
            return job.id()
        except Exception as exception:
            # Show the user an error message
            if Configuration.get('KATANA_UI_MODE'):
                from PyQt5 import QtWidgets
                errorMessage = ('An error occurred submitting the job to the '
                                'render farm.\n\n' + str(exception))
                QtWidgets.QMessageBox.critical(None,
                                               'Error Submitting Job',
                                               errorMessage)
            else:
                log.exception('Could not submit render job')
            raise

    def _getJobs(self, **options):
        """
        @rtype: C{list} of C{opencue.wrapper.job.Job}
        @return: A list of jobs matching the specified options (see
            C{opencue.wrapper.api.getJobs()} for a description of supported
            filters).
        """
        return opencue.api.getJobs(**options)

    def _getLogfileFromOpenCueJob(self, openCueJob):
        """
        @type openCueJob: L{opencue.wrapper.job.Job}
        @rtype: C{str} or None if the file cannot be found
        @param openCueJob: The OpenCue Job whose render log filename to return.
        @return: Path to the render log for the specified OpenCue Job.
        """
        logDirectory = openCueJob.logDir()
        if not os.path.exists(logDirectory):
            return None

        for fileName in os.listdir(logDirectory):
            if fileName.endswith('.rqlog'):
                return os.path.join(logDirectory, fileName)

        return None

    def getRenderLogFilename(self, jobId):
        """
        Returns the filename of the render log for the specified C{jobId}, or
        C{None} if it cannot be accessed from this host.

        @type jobId: C{str}
        @rtype: C{str} or C{None}
        @param jobId: The unique identifier of the job whose render log
            filename to return.
        @return: File path to render log or None.
        @raise JobNotFoundException: If the specified job does not exist on
            the render farm.
        """
        # Query for just our job.
        jobs = self._getJobs(id=[jobId], include_finished=True)
        if not jobs:
            raise FarmAPI.JobNotFoundException(jobId,
                                               'Retrieve render log filename')
        return self._getLogfileFromOpenCueJob(jobs[0])

    def getJobState(self, jobId):
        """
        Returns the state of the job specified by C{jobId}.

        @type jobId: C{str}
        @rtype: C{str}
        @param jobId: The unique identifier of the job whose state to return.
        @return: The state of the job with the given ID as a value from the
            L{Job.State} enumeration.
        @raise JobNotFoundException: If the specified job does not exist on
            the render farm.
        """
        # Query for just our job.
        jobs = self._getJobs(id=[jobId], include_finished=True)
        if not jobs:
            raise FarmAPI.JobNotFoundException(jobId, 'Retrieve job state')

        job = jobs[0]

        # Inspect the job to see if it is currently in a failing state, if so,
        # update the details here.
        if self.__eatDeadFrames:
            if job.deadFrames() > 0:
                job.setAutoEating(True)
                job.kill()
                return FarmAPI.Job.State.kFailed

        return OpenCueFarmPlugin.kJobStateMap[job.state()]

    def getJobs(self, filters):
        """
        @type filters: C{list}
        @rtype: C{list}
        @param filters: The filters to apply for obtaining the list of jobs.
        @return: A list of Jobs based on the specified filter criteria.
        """
        # Convert filters to match our internal representation
        opencueFilters = {}
        for jobFilter in filters:
            key, value = jobFilter.split('=', 2)
            if key == 'jobId':
                # OpenCue search requires this is supplied as a list
                opencueFilters['id'] = [value]
            else:
                opencueFilters[key] = value

        jobs = self._getJobs(**opencueFilters)

        # Convert our job representation to Katana's representation.
        result = []
        for job in jobs:
            openCueJobState = OpenCueFarmPlugin.kJobStateMap[job.state()]
            logFile = self._getLogfileFromOpenCueJob(job)

            result.append(FarmAPI.Job(job.id(),
                                      job.name(),
                                      job.startTime(),
                                      job.stopTime(),
                                      openCueJobState,
                                      logFile))
        return result

    def startJob(self, jobId):
        """
        Starts the Job specified by the job ID, if it is not already running.

        @type jobId: C{str}
        @param jobId: The ID of the job to start.
        @raise JobNotFoundException: If the requested job cannot be found.
        """
        log.info('Starting job "%s"...', jobId)
        jobs = self._getJobs(id=[jobId], include_finished=True)
        if not jobs:
            raise FarmAPI.JobNotFoundException(jobId, 'Start job')

        # Only start the Job if it's not currently in a running state
        job = jobs[0]
        if job.state() in OpenCueFarmPlugin.kRunningStates:
            log.info('Job "%s" is already running.', job.name())
        else:
            job.resume()

    def stopJob(self, jobId):
        """
        Stops the Job specified by the job ID, if it is not already stopped.

        @type jobId: C{str}
        @param jobId: The ID of the job to stop.
        @raise JobNotFoundException: If the requested job cannot be found.
        """
        log.info('Stopping job "%s"...', jobId)

        jobs = self._getJobs(id=[jobId], include_finished=True)
        if not jobs:
            raise FarmAPI.JobNotFoundException(jobId, 'Stop job')

        # Only stop the Job if it's not currently in a stoppped state
        job = jobs[0]
        if job.state in OpenCueFarmPlugin.kStoppedStates:
            log.info('Job "%s" has already been stopped.', job.name())
        else:
            job.kill()


def ValidateOpenCueConfiguration(objectHash):
    """
    Checks that the necessary OpenCue environment variables have been set.
    If missing environment variables are detected, an error message will be
    logged. Otherwise OpenCue will be set as the default FarmPlugin.

    @type objectHash: C{int} or C{None}
    @param objectHash: The hash of an object that was passed when the callback
        was called. Is C{None} for C{onStartupComplete} callbacks.
    """
    # pylint: disable=unused-argument
    requiredEnvironmentVariables = ('CUE_FS_ROOT',
                                    'CUEBOT_HOSTS',
                                    'OL_CONFIG',
                                    'KATANA_CUE_DEFAULT_FACILITY')
    missingEnvironmentVariables = [variable for variable in
                                   requiredEnvironmentVariables
                                   if variable not in os.environ]

    if missingEnvironmentVariables:
        log.error("OpenCueFarmPlugin will not function correctly as the "
                  "environment variables %s have not been set.",
                  missingEnvironmentVariables)
        return

    FarmAPI.FarmPluginManager.SetDefaultFarm('OpenCue')


if Configuration.get('KATANA_UI_MODE'):
    Callbacks.addCallback(
        Callbacks.Type.onStartupComplete, ValidateOpenCueConfiguration)

PluginRegistry = [("FarmPlugin", 2.0, "OpenCue", OpenCueFarmPlugin)]
