# Copyright (c) 2020 The Foundry Visionmongers Ltd. All Rights Reserved.
"""
Module defining the L{KatanaQueueTab} class which implements the B{Katana
Queue} tab.

Depends on the C{kq} Python package.
"""


import logging
import os
import textwrap

from PyQt5 import (
    QtCore,
    QtGui,
    QtWidgets,
)

import KatanaResources
import Utils

from Katana import (
    FarmAPI,
    UI4,
)
from UI4.Tabs import BaseTab
from UI4.Widgets import IconLabelFrame

import kq.Util.LoggingConfig
from kq.Client.KatanaQueue import GetSharedKatanaQueueWrapper
from kq.Util.Timing import GetCurrentMilliseconds


log = logging.getLogger('KatanaQueue')

_startTimeMs = GetCurrentMilliseconds()


# Module Variables ------------------------------------------------------------

kColorByJobState = {
    FarmAPI.Job.State.kWaiting: QtGui.QColor.fromRgbF(0.204, 0.275, 0.408),
    FarmAPI.Job.State.kRunning: QtGui.QColor.fromRgbF(0.36, 0.25, 0.38),
    FarmAPI.Job.State.kCompleted: QtGui.QColor.fromRgbF(0.20, 0.36, 0.10),
    FarmAPI.Job.State.kCancelled: QtGui.QColor.fromRgbF(0.35, 0.30, 0.20),
    FarmAPI.Job.State.kFailed: QtGui.QColor.fromRgbF(0.45, 0.25, 0.25),
}
kDisabledColor = QtGui.QColor(255, 155, 0)


# Class Definitions -----------------------------------------------------------

class StateItemDelegate(QtWidgets.QStyledItemDelegate):

    # Initializer -------------------------------------------------------------

    def __init__(self, *args):
        """
        Initializes an instance of the class.
        """
        super(StateItemDelegate, self).__init__(*args)
        self.missingJobsRows = set()
        self.queueSortFilterProxyModel = None
        self.missingJobColor = QtGui.QColor.fromRgbF(0.3, 0.3, 0.3)

    # QStyledItemDelegate Instance Functions ----------------------------------

    def initStyleOption(self, option, index):
        """
        Initializes the given style option with the values of the item with the
        given model index.

        @type option: C{QtWidgets.QStyleOptionViewItem}
        @type index: C{QtCore.QModelIndex}
        @param option: The style option to fill with values of the item with
            the given model index.
        @param index: The model index of the item whose values to use for
            initializing the given style option.
        """
        QtWidgets.QStyledItemDelegate.initStyleOption(self, option, index)

        jobState = option.text
        mappedIndex = self.queueSortFilterProxyModel.mapToSource(index)
        if mappedIndex.row() in self.missingJobsRows:
            color = self.missingJobColor
        else:
            color = kColorByJobState.get(jobState)
        if color is not None:
            option.backgroundBrush.setStyle(QtCore.Qt.SolidPattern)
            option.backgroundBrush.setColor(color)

        # Check if the given style option indicates that the item is selected
        if (option.state & QtWidgets.QStyle.State_Selected
                == QtWidgets.QStyle.State_Selected):
            # Adjust the given style option's state, so that we can override
            # drawing of selected items
            option.state ^= QtWidgets.QStyle.State_Selected

            option.palette.setColor(QtGui.QPalette.Text,
                                    option.palette.highlightedText().color())

class KatanaQueueTab(BaseTab):
    """
    Class implementing a simple GUI interface to the Katana Queue.
    """
    # pylint: disable=too-many-instance-attributes

    # Class Variables ---------------------------------------------------------

    kFarmPluginName = "Katana Queue"
    kStartJobs1ActionID = '870d407a33003e888dc09527f4fd84e7'
    kStartJobs2ActionID = '81fac7329db0389f7e264ebcfac57233'
    kStopJobsActionID = '8228ced74c03c4a859846c988fe4b2c8'

    # Static Functions --------------------------------------------------------

    @staticmethod
    def registerKeyboardShortcuts():
        """
        Registers action callbacks for this tab using functions provided by
        L{BaseTab}.
        """
        KatanaQueueTab.setShortcutsContextName('Katana Queue Tab')

        KatanaQueueTab.registerKeyboardShortcut(
            KatanaQueueTab.kStartJobs1ActionID, 'Start Jobs 1', 'Return',
            KatanaQueueTab.startJobsActionCallback)
        KatanaQueueTab.registerKeyboardShortcut(
            KatanaQueueTab.kStartJobs2ActionID, 'Start Jobs 2', 'Enter',
            KatanaQueueTab.startJobsActionCallback)
        KatanaQueueTab.registerKeyboardShortcut(
            KatanaQueueTab.kStopJobsActionID, 'Stop Jobs', '',
            KatanaQueueTab.stopJobsActionCallback)

    # Initializer -------------------------------------------------------------

    def __init__(self, parent=None, flags=0):
        """
        Initializes an instance of the tab.

        @type parent: C{QtWidgets.QWidget} or C{None}
        @type flags: C{QtCore.Qt.WindowFlags}
        @param parent: The parent widget to own the new instance. Passed
            verbatim to the base class initializer.
        @param flags: The window flags to use in case no parent is given.
            Passed verbatim to the base class initializer.
        """
        BaseTab.__init__(self, parent, flags)

        # Set a default object name for the instance based on the class name
        self.setObjectName(self.__class__.__name__[0].lower()
                           + self.__class__.__name__[1:])

        if not kq.IsDisabled():
            self.queueClientWrapper = GetSharedKatanaQueueWrapper()
        else:
            self.queueClientWrapper = None

        # Create a data model for the queue
        headerLabels = [
            'Name',
            'State',
            'Submitted',
            'Completed',
            'Log File',
        ]
        self.queueModel = QtGui.QStandardItemModel(self)
        self.queueModel.setObjectName('queueModel')
        self.queueModel.setHorizontalHeaderLabels(headerLabels)
        self.queueSortFilterProxyModel = QtCore.QSortFilterProxyModel(self)
        self.queueSortFilterProxyModel.setObjectName(
            'queueSortFilterProxyModel')
        self.queueSortFilterProxyModel.setFilterKeyColumn(
            headerLabels.index('State'))
        self.queueSortFilterProxyModel.setSourceModel(self.queueModel)
        self.queueSelectionModel = QtCore.QItemSelectionModel(
            self.queueSortFilterProxyModel, self)
        self.queueSelectionModel.setObjectName('queueSelectionModel')

        self.jobToRowMapping = {}
        self.missingJobsRows = set()

        self.katanaQueueStatusFrame = None
        self.queueTableView = None
        self.startJobsButton = None
        self.stopJobsButton = None

        # Log Viewer
        self.logViewer = None
        self.selectedRenderLogFilename = None
        self.furthestRead = -1

        self.__setupUI()

        self.lastPolled = GetCurrentMilliseconds()

        # Only register the event handler if Katana Queue is enabled
        if not kq.IsDisabled():
            Utils.EventModule.RegisterEventHandler(self.__on_event_idle,
                                                   'event_idle')

        QtCore.QMetaObject.connectSlotsByName(self)

    # Action Callbacks --------------------------------------------------------

    def startJobsActionCallback(self):
        """
        Action callback for the tab that is called when the associated keyboard
        shortcut has been pressed.

        Simulates a click of the B{Start} button, so that jobs that correspond
        to the selected items in the table view are restarted.
        """
        self.startJobsButton.click()

    def stopJobsActionCallback(self):
        """
        Action callback for the tab that is called when the associated keyboard
        shortcut has been pressed.

        Simulates a click of the B{Stop} button, so that jobs that correspond
        to the selected items in the table view are stopped.
        """
        self.stopJobsButton.click()

    # Private Instance Functions ----------------------------------------------

    def __setupUI(self):
        """
        Creates the tab's widgets and layouts.
        """
        # pylint: disable=too-many-statements

        # Create a frame to show the Katana Queue status.
        self.katanaQueueStatusFrame = QtWidgets.QFrame()
        self.katanaQueueStatusFrame.setObjectName('katanaQueueStatusFrame')
        icon = KatanaResources.GetResourceFile('Icons/warningBlack24.png')
        iconLabelFrame = IconLabelFrame(QtGui.QPixmap(icon),
                                        'Katana Queue is not responding.',
                                        margin=4,
                                        parent=self.katanaQueueStatusFrame)
        iconLabelFrame.setStyleSheet('QLabel { color: black; }')
        self.katanaQueueStatusFrame.setStyleSheet(
            'QFrame { background-color: #e08915; }')
        self.restartKatanaQueueButton = QtWidgets.QPushButton(
            'Restart Katana Queue', self.katanaQueueStatusFrame)
        self.restartKatanaQueueButton.setObjectName('restartKatanaQueueButton')
        katanaQueueStatusLayout = QtWidgets.QHBoxLayout()
        katanaQueueStatusLayout.addWidget(iconLabelFrame, 10)
        katanaQueueStatusLayout.addWidget(self.restartKatanaQueueButton)
        katanaQueueStatusLayout.addSpacing(8)
        self.katanaQueueStatusFrame.setLayout(katanaQueueStatusLayout)
        self.katanaQueueStatusFrame.hide()
        if not kq.IsDisabled():
            self.__updateKatanaQueueStatusFrame()

        # Set up widgets to show above the Katana Queue table view
        jobsLabel = QtWidgets.QLabel('Jobs')
        self.__jobStateFilterToolBar = QtWidgets.QToolBar(self)
        self.__jobStateFilterToolBar.setObjectName('jobStateFilterToolBar')
        self.__jobStateFilterToolBar.setProperty('horizontalIcons', True)
        smallIconSize = self.style().pixelMetric(
            QtWidgets.QStyle.PM_SmallIconSize, widget=self)
        for jobState in FarmAPI.Job.State.kAllStates:
            action = self.__jobStateFilterToolBar.addAction(jobState)
            action.setCheckable(True)
            action.setChecked(True)
            action.setToolTip('Click to hide or show jobs that are in the %s '
                              'state.' % jobState)
            toolButton = self.__jobStateFilterToolBar.widgetForAction(action)
            toolButton.setMinimumHeight(smallIconSize + 2)
            toolButton.setStyleSheet(textwrap.dedent('''
                QToolButton {
                    background-color: palette(shadow);
                    padding-left: 2px;
                    padding-right: 2px;
                }
                QToolButton:open {
                    color: palette(text);
                    background-color: %s;
                }
                QToolButton:open:hover {
                    color: palette(highlighted-text);
                }
                ''') % kColorByJobState[jobState].name())

        if kq.IsDisabled():
            warningIconLabelFrame = UI4.Widgets.IconLabelFrame(
                QtGui.QPixmap(
                    KatanaResources.GetIconPath('yellowWarning16.png')),
                '<span style="color: %s;">KQ is disabled</span>'
                % kDisabledColor.name(), parent=self)
            warningIconLabelFrame.setObjectName('warningIconLabelFrame')
        else:
            numberOfAgentsLabel = QtWidgets.QLabel(
                'using <b>%s</b> local agents' % kq.GetNumberOfLocalAgents())
            numberOfAgentsLabel.setObjectName('numberOfAgentsLabel')
            numberOfAgentsLabel.setEnabled(False)

        # Set up a table view to display the jobs on the Katana Queue
        self.queueTableView = QtWidgets.QTableView(self)
        self.queueTableView.setObjectName('queueTableView')
        self.queueTableView.setAlternatingRowColors(True)
        self.queueTableView.setEditTriggers(
            QtWidgets.QAbstractItemView.NoEditTriggers)
        stateItemDelegate = StateItemDelegate()
        stateItemDelegate.missingJobsRows = self.missingJobsRows
        stateItemDelegate.queueSortFilterProxyModel = \
            self.queueSortFilterProxyModel
        self.queueTableView.setItemDelegateForColumn(1, stateItemDelegate)
        self.queueTableView.setSelectionBehavior(
            QtWidgets.QAbstractItemView.SelectRows)
        self.queueTableView.setSelectionMode(
            QtWidgets.QAbstractItemView.ExtendedSelection)
        self.queueTableView.setSortingEnabled(True)
        self.queueTableView.setWordWrap(False)

        # Set up the table view's headers, both horizontal and vertical
        horizontalHeaderView = self.queueTableView.horizontalHeader()
        horizontalHeaderView.setObjectName('horizontalHeaderView')
        horizontalHeaderView.setHighlightSections(False)
        horizontalHeaderView.setStretchLastSection(True)
        horizontalHeaderView.setDefaultAlignment(QtCore.Qt.AlignLeft
                                                 | QtCore.Qt.AlignVCenter)
        horizontalHeaderView.setStyleSheet(textwrap.dedent("""
           QHeaderView::section:hover {
               color: palette(highlighted-text);
               background-color: %s;
           }
           QHeaderView::down-arrow {
               image: url(%s);
               width: 16px;
               height: 16px;
           }
           QHeaderView::up-arrow {
               image: url(%s);
               width: 16px;
               height: 16px;
           }
           """) % (self.palette().button().color().lighter(140).name(),
                   QtCore.QDir.fromNativeSeparators(
                       KatanaResources.GetResourceFile('Icons/menuArrow16.png')),
                   QtCore.QDir.fromNativeSeparators(
                       KatanaResources.GetResourceFile(
                           'Icons/menuArrowUp16.png')),
                  ))
        verticalHeaderView = self.queueTableView.verticalHeader()
        verticalHeaderView.setHighlightSections(False)
        verticalHeaderView.setDefaultSectionSize(
            self.style().pixelMetric(QtWidgets.QStyle.PM_LargeIconSize))
        verticalHeaderView.setSectionResizeMode(QtWidgets.QHeaderView.Fixed)

        # Set up the table view's underlying models
        self.queueTableView.setModel(self.queueSortFilterProxyModel)
        self.queueTableView.setSelectionModel(self.queueSelectionModel)

        # Get keyboard shortcuts associated with start and stop buttons
        from UI4.App.KeyboardShortcutManager import GetShortcutForKeyEvent
        startJobsShortcuts = [
            GetShortcutForKeyEvent(actionID)
            for actionID in (KatanaQueueTab.kStartJobs1ActionID,
                             KatanaQueueTab.kStartJobs2ActionID)]
        stopJobsShortcut = GetShortcutForKeyEvent(
            KatanaQueueTab.kStopJobsActionID)

        # Create buttons for starting and stopping selected jobs
        self.startJobsButton = QtWidgets.QPushButton('Start')
        self.startJobsButton.setObjectName('startJobsButton')
        self.startJobsButton.setEnabled(False)
        if startJobsShortcuts:
            self.startJobsButton.setToolTip('Start Selected Jobs [%s]'
                                            % '/'.join(startJobsShortcuts))
        else:
            self.startJobsButton.setToolTip('Start Selected Jobs')
        self.stopJobsButton = QtWidgets.QPushButton('Stop')
        self.stopJobsButton.setObjectName('stopJobsButton')
        self.stopJobsButton.setEnabled(False)
        if stopJobsShortcut:
            self.stopJobsButton.setToolTip('Stop Selected Jobs [%s]'
                                           % stopJobsShortcut)
        else:
            self.stopJobsButton.setToolTip('Stop Selected Jobs')

        # Create a layout for the buttons that act on selected jobs
        buttonLayout = QtWidgets.QHBoxLayout()
        buttonLayout.addStretch()
        buttonLayout.addWidget(self.startJobsButton)
        buttonLayout.addWidget(self.stopJobsButton)

        # Set up the log viewer
        logLabel = QtWidgets.QLabel('Log')
        actionButton = UI4.Widgets.MenuButton(self, 'Action')
        actionButtonMenu = QtWidgets.QMenu(actionButton)
        actionButtonMenu.setObjectName('actionButtonMenu')
        actionButton.setMenu(actionButtonMenu)
        self.logViewer = QtWidgets.QTextEdit()
        self.logViewer.setProperty('codeFont', True)
        self.logViewer.setReadOnly(True)

        # Create a layout for widgets above the queue table view widget
        queueHeaderLayout = QtWidgets.QHBoxLayout()
        queueHeaderLayout.addWidget(jobsLabel)
        queueHeaderLayout.addStretch()
        queueHeaderLayout.addWidget(self.__jobStateFilterToolBar)
        queueHeaderLayout.addStretch()
        if kq.IsDisabled():
            queueHeaderLayout.addWidget(warningIconLabelFrame)
        else:
            queueHeaderLayout.addWidget(numberOfAgentsLabel)

        # Create a layout for widgets above the log viewer widget
        logViewerHeaderLayout = QtWidgets.QHBoxLayout()
        logViewerHeaderLayout.addWidget(logLabel)
        logViewerHeaderLayout.addStretch()
        logViewerHeaderLayout.addWidget(actionButton)

        # Create the tab's layout and lay out the widgets in it
        tabLayout = QtWidgets.QVBoxLayout()
        tabLayout.setContentsMargins(4, 4, 4, 4)
        tabLayout.addWidget(self.katanaQueueStatusFrame)
        tabLayout.addLayout(queueHeaderLayout)
        tabLayout.addWidget(self.queueTableView)
        tabLayout.addLayout(buttonLayout)
        tabLayout.addWidget(
            UI4.Widgets.VBoxLayoutResizer(self.logViewer, 200,
                                          beforeTargetWidget=True))
        tabLayout.addLayout(logViewerHeaderLayout)
        tabLayout.addWidget(self.logViewer)
        self.setLayout(tabLayout)

    # Katana Event Handlers ---------------------------------------------------

    def __on_event_idle(self, eventType, eventID):
        """
        Event handler for C{'event_idle'} events.

        Updates the queue model by obtaining a list of Jobs from the
        C{FarmAPI}, and updates other UI widgets, notably the render log view.

        @type eventType: C{str} or C{None}
        @type eventID: C{object}
        @param eventType: The name of the type of event to handle.
        @param eventID: A hashable object that can be used to further filter
            event handling.
        """
        # pylint: disable=unused-argument

        timeNow = GetCurrentMilliseconds()
        if timeNow - self.lastPolled > 500.0:
            self.__updateQueueModel()
            self.__tailLogFile()
            self.__updateJobButtons()
            self.__updateKatanaQueueStatusFrame()
            self.lastPolled = timeNow

    # Private Instance Functions (continued) ----------------------------------

    def __tailLogFile(self):
        """
        Reads the current log file (if not none) and fills the log viewer`
        """
        if not self.selectedRenderLogFilename:
            return

        # Check the file exists
        if not os.path.exists(self.selectedRenderLogFilename):
            return

        # Already read to end
        with open(self.selectedRenderLogFilename, 'r') as renderLogFile:
            # Already read to end
            currentSize = os.path.getsize(self.selectedRenderLogFilename)
            if self.furthestRead == currentSize:
                return

            if self.furthestRead > 0:
                renderLogFile.seek(self.furthestRead)

            toRead = currentSize - self.furthestRead
            fileContents = renderLogFile.read(toRead)
            self.logViewer.setPlainText(self.logViewer.toPlainText() +
                                        fileContents)
            self.furthestRead = renderLogFile.tell()

            # Scroll to end
            self.logViewer.verticalScrollBar().setValue(
                self.logViewer.verticalScrollBar().maximum())

    def __updateQueueModel(self):
        """
        Updates the model that contains the current listing of Jobs in the
        Katana Queue.

        Also updates the filter tool buttons at the top of the tab.
        """
        # pylint: disable=too-many-locals

        # Initialize a dictionary of numbers of jobs for each job state defined
        # in the FarmAPI
        numberOfJobsByState = dict(
            (jobState, 0) for jobState in FarmAPI.Job.State.kAllStates)

        # Create a code font for use in the render log filename column
        codeFont = self.font()
        codeFont.setFamily('monospace')

        # Retrieve the list of jobs that are currently managed by the farm
        jobs = FarmAPI.FarmPluginManager.GetJobs(
            [], KatanaQueueTab.kFarmPluginName)

        # Update the set of missing jobs, so they are colored in gray if they
        # go missing.
        previousMissingJobsRows = set(self.missingJobsRows)
        self.missingJobsRows.clear()
        jobIds = set([job.jobId for job in jobs])
        for jobId, jobRow in self.jobToRowMapping.items():
            if jobId not in jobIds:
                self.missingJobsRows.add(jobRow)

        # Invalidate the rows that have changed from missing to found, or
        # viceversa.
        for jobRow in self.missingJobsRows ^ previousMissingJobsRows:
            stateModelIndex = self.queueModel.index(jobRow, 1)
            currentState = self.queueModel.data(stateModelIndex)
            self.queueModel.setData(stateModelIndex, '')
            self.queueModel.setData(stateModelIndex, currentState)

        for job in jobs:
            # Do we have a row for this job?
            jobRow = self.jobToRowMapping.get(job.jobId)
            if jobRow is not None:
                # Update job state if different
                stateModelIndex = self.queueModel.index(jobRow, 1)
                oldState = self.queueModel.data(stateModelIndex)
                if oldState != job.state:
                    self.queueModel.setData(stateModelIndex, job.state)

                # Update start time if different
                index = self.queueModel.index(jobRow, 2)
                oldJobSubmittedTime = self.queueModel.data(index)
                newJobSubmittedTime = (
                    kq.Util.Timing.GetTimestamp(job.startTime)
                    if job.startTime is not None
                    else '')
                if newJobSubmittedTime != oldJobSubmittedTime:
                    self.queueModel.setData(index, newJobSubmittedTime)
                    self.queueModel.setData(index, job.startTime,
                                            QtCore.Qt.UserRole)

                # Update end time if different
                index = self.queueModel.index(jobRow, 3)
                oldJobCompletedTime = self.queueModel.data(index)
                newJobCompletedTime = (kq.Util.Timing.GetTimestamp(job.endTime)
                                       if job.endTime is not None
                                       else '')
                if newJobCompletedTime != oldJobCompletedTime:
                    self.queueModel.setData(index, newJobCompletedTime)
                    self.queueModel.setData(index, job.endTime,
                                            QtCore.Qt.UserRole)
            else:
                # Add new Job
                jobNameItem = QtGui.QStandardItem(job.name)
                jobNameItem.setData(job.jobId, QtCore.Qt.UserRole)

                jobStateItem = QtGui.QStandardItem(job.state)
                jobStateItem.setData(QtCore.Qt.AlignCenter,
                                     QtCore.Qt.TextAlignmentRole)

                jobSubmittedTimeItem = QtGui.QStandardItem(
                    kq.Util.Timing.GetTimestamp(job.startTime)
                    if job.startTime is not None
                    else '')
                jobSubmittedTimeItem.setData(QtCore.Qt.AlignCenter,
                                             QtCore.Qt.TextAlignmentRole)
                jobSubmittedTimeItem.setData(job.startTime, QtCore.Qt.UserRole)

                jobCompletedTimeItem = QtGui.QStandardItem(
                    kq.Util.Timing.GetTimestamp(job.endTime)
                    if job.endTime is not None
                    else '')
                jobCompletedTimeItem.setData(QtCore.Qt.AlignCenter,
                                             QtCore.Qt.TextAlignmentRole)
                jobCompletedTimeItem.setData(job.endTime, QtCore.Qt.UserRole)

                jobLogFileItem = QtGui.QStandardItem(job.renderLogFilename)
                jobLogFileItem.setData(codeFont, QtCore.Qt.FontRole)

                rowItems = [
                    jobNameItem,
                    jobStateItem,
                    jobSubmittedTimeItem,
                    jobCompletedTimeItem,
                    jobLogFileItem,
                ]
                self.queueModel.appendRow(rowItems)
                self.jobToRowMapping[job.jobId] = \
                    self.queueModel.rowCount(QtCore.QModelIndex()) - 1

            numberOfJobsByState[job.state] += 1

        # Update the texts of actions in the job state filter toolbar with the
        # number of jobs in each state
        for jobState, action in zip(FarmAPI.Job.State.kAllStates,
                                    self.__jobStateFilterToolBar.actions()):
            action.setText('%d %s' % (numberOfJobsByState[jobState], jobState))

    def __updateJobButtons(self):
        """
        Updates the availability of the buttons for starting and stopping
        selected jobs based on the states of selected jobs.
        """
        anyJobsThatCanBeStarted = False
        anyJobsThatCanBeStopped = False

        # Get a list of model indexes for cells in the state column for rows
        # that are selected
        stateIndexes = self.queueTableView.selectionModel().selectedRows(1)
        for stateIndex in stateIndexes:
            stateIndex = self.queueSortFilterProxyModel.mapToSource(stateIndex)
            jobRow = stateIndex.row()
            state = self.queueModel.data(stateIndex)

            if (not anyJobsThatCanBeStarted
                    and state in (FarmAPI.Job.State.kCompleted,
                                  FarmAPI.Job.State.kCancelled,
                                  FarmAPI.Job.State.kFailed)
                    and jobRow not in self.missingJobsRows):
                anyJobsThatCanBeStarted = True

            if (not anyJobsThatCanBeStopped
                    and state in (FarmAPI.Job.State.kWaiting,
                                  FarmAPI.Job.State.kRunning)
                    and jobRow not in self.missingJobsRows):
                anyJobsThatCanBeStopped = True

            # Break out of the loop once we know there are jobs that can be
            # started and stopped
            if anyJobsThatCanBeStarted and anyJobsThatCanBeStopped:
                break

        self.startJobsButton.setEnabled(anyJobsThatCanBeStarted)
        self.stopJobsButton.setEnabled(anyJobsThatCanBeStopped)

    def __updateKatanaQueueStatusFrame(self):
        """
        Checks if the Katana Queue status frame needs to be shown.
        """
        # Grace period before potentially showing the alert.
        if GetCurrentMilliseconds() - _startTimeMs < 5000:
            return

        client = self.queueClientWrapper.getClient()
        alive = client is not None and client.isAlive()
        self.katanaQueueStatusFrame.setVisible(not alive)

    # Slots -------------------------------------------------------------------

    @QtCore.pyqtSlot(QtWidgets.QAction)
    def on_jobStateFilterToolBar_actionTriggered(self, action):
        """
        Slot that is called when one of the actions in the toolbar for
        filtering the jobs in the table view of the tab by their job state.

        @type action: C{QtWidgets.QAction}
        @param action: The action that was triggered.
        """
        activeJobStates = [jobState
                           for jobState, action in zip(
                               FarmAPI.Job.State.kAllStates,
                               self.__jobStateFilterToolBar.actions())
                           if action.isChecked()]

        self.queueSortFilterProxyModel.setFilterRegExp(
            '(%s)' % '|'.join(activeJobStates))

    @QtCore.pyqtSlot(int, QtCore.Qt.SortOrder)
    def on_horizontalHeaderView_sortIndicatorChanged(self, logicalIndex,
                                                     order):
        """
        Slot that is called when a column header in the horizontal header view
        has been clicked to sort the table of jobs by a particular column.

        @type logicalIndex: C{int}
        @type order: C{QtCore.Qt.SortOrder}
        @param logicalIndex: The index of the column header that was clicked,
            ignoring any reordering of columns that might have taken place.
        @param order: C{QtCore.Qt.AscendingOrder} or
            C{QtCore.Qt.DescendingOrder}.
        """
        # pylint: disable=unused-argument

        # Sort time items using the timestamp in miliseconds that is stored as
        # user data instead of the display text, for more accurate sorting
        if logicalIndex == 2 or logicalIndex == 3:
            self.queueSortFilterProxyModel.setSortRole(QtCore.Qt.UserRole)
        else:
            self.queueSortFilterProxyModel.setSortRole(QtCore.Qt.DisplayRole)

    @QtCore.pyqtSlot(QtCore.QItemSelection, QtCore.QItemSelection)
    def on_queueSelectionModel_selectionChanged(self, selected, deselected):
        """
        Slot that is called when the selection of items in the table view of
        jobs has changed.

        @type selected: C{QtCore.QItemSelection}
        @type deselected: C{QtCore.QItemSelection}
        @param selected: The new selection of items.
        @param deselected: Items that are no longer selected.
        """
        # pylint: disable=unused-argument

        self.__updateJobButtons()

    @QtCore.pyqtSlot(QtCore.QModelIndex, QtCore.QModelIndex)
    def on_queueSelectionModel_currentRowChanged(self, current, previous):
        """
        Slot that is called when the current item changes and its row is
        different to the row of the previous current item.

        @type current: C{QtCore.QModelIndex}
        @type previous: C{QtCore.QModelIndex}
        @param current: The index of the item that is now the current item.
        @param previous: The index of the item that was previously the current
            item.
        """
        # pylint: disable=unused-argument

        # Determine which of our jobs the user selected.
        jobRow = current.row()
        if jobRow == -1:
            # This happens when all jobs are filtered out, and the table view
            # ends up being empty
            return

        nameIndex = self.queueSortFilterProxyModel.mapToSource(
            self.queueSortFilterProxyModel.index(jobRow, 0))
        jobId = str(self.queueModel.data(nameIndex, QtCore.Qt.UserRole))
        jobs = FarmAPI.FarmPluginManager.GetJobs(
            ['jobId=%s' % jobId], KatanaQueueTab.kFarmPluginName)
        if not jobs:
            return

        renderLogFilenameIndex = self.queueSortFilterProxyModel.mapToSource(
            self.queueSortFilterProxyModel.index(jobRow, 4))
        renderLogFilename = self.queueModel.data(renderLogFilenameIndex)
        if renderLogFilename == self.selectedRenderLogFilename:
            return

        self.logViewer.setPlainText('')
        self.selectedRenderLogFilename = renderLogFilename
        self.furthestRead = -1

    @QtCore.pyqtSlot()
    def on_startJobsButton_clicked(self):
        """
        Slot that is called when the B{Start} button has been clicked.
        """
        if kq.IsDisabled():
            QtWidgets.QMessageBox.critical(self, self.sender().text(),
                                           '%s is not available.'
                                           % KatanaQueueTab.kFarmPluginName)
            return

        nameIndexes = self.queueTableView.selectionModel().selectedRows()
        for nameIndex in nameIndexes:
            # Determine the ID, name, and state of the job to start from the
            # data stored in the model for the selected row
            nameIndex = self.queueSortFilterProxyModel.mapToSource(nameIndex)
            jobName = self.queueModel.data(nameIndex)
            jobId = str(self.queueModel.data(nameIndex, QtCore.Qt.UserRole))
            stateIndex = self.queueModel.index(nameIndex.row(), 1)
            state = self.queueModel.data(stateIndex)

            if state in (FarmAPI.Job.State.kCompleted,
                         FarmAPI.Job.State.kCancelled,
                         FarmAPI.Job.State.kFailed):
                log.info('Restarting job "%s" (%s)...', jobName, jobId)

                if FarmAPI.FarmPluginManager.StartJob(
                        jobId, KatanaQueueTab.kFarmPluginName):
                    self.__updateQueueModel()

    @QtCore.pyqtSlot()
    def on_stopJobsButton_clicked(self):
        """
        Slot that is called when the B{Stop} button has been clicked.
        """
        if kq.IsDisabled():
            QtWidgets.QMessageBox.critical(self, self.sender().text(),
                                           '%s is not available.'
                                           % KatanaQueueTab.kFarmPluginName)
            return

        nameIndexes = self.queueTableView.selectionModel().selectedRows()
        for nameIndex in nameIndexes:
            # Determine the ID, name, and state of the job to stop from the
            # data stored in the model for the selected row
            nameIndex = self.queueSortFilterProxyModel.mapToSource(nameIndex)
            jobName = self.queueModel.data(nameIndex)
            jobId = str(self.queueModel.data(nameIndex, QtCore.Qt.UserRole))
            stateIndex = self.queueModel.index(nameIndex.row(), 1)
            state = self.queueModel.data(stateIndex)

            if state in (FarmAPI.Job.State.kWaiting,
                         FarmAPI.Job.State.kRunning):
                log.info('Stopping job "%s" (%s)...', jobName, jobId)

                if FarmAPI.FarmPluginManager.StopJob(
                        jobId, KatanaQueueTab.kFarmPluginName):
                    self.__updateQueueModel()

    @QtCore.pyqtSlot()
    def on_actionButtonMenu_aboutToShow(self):
        """
        Slot that is called when the B{Action} button has been clicked.
        """
        actionButtonMenu = self.sender()
        actionButtonMenu.clear()

        def openJobRenderLog():
            renderLogFilename = self.selectedRenderLogFilename
            if renderLogFilename is None:
                return

            if os.path.isfile(renderLogFilename):
                UI4.Util.ExternalTools.openFileForEdit(renderLogFilename)
            else:
                QtWidgets.QMessageBox.critical(self, self.sender().text(),
                                               'File not found: "%s"'
                                               % renderLogFilename)

        def openKqLog():
            kqLogFilename = kq.Util.LoggingConfig.GetLogFilename('kq')
            if os.path.isfile(kqLogFilename):
                UI4.Util.ExternalTools.openFileForEdit(kqLogFilename)
            else:
                QtWidgets.QMessageBox.critical(self, self.sender().text(),
                                               'File not found: "%s"'
                                               % kqLogFilename)

        def openAgentLog():
            agentLogFilename = kq.Util.LoggingConfig.GetLogFilename('agent')
            if os.path.isfile(agentLogFilename):
                UI4.Util.ExternalTools.openFileForEdit(agentLogFilename)
            else:
                QtWidgets.QMessageBox.critical(self, self.sender().text(),
                                               'File not found: "%s"'
                                               % agentLogFilename)

        openJobRenderLogAction = actionButtonMenu.addAction(
            'Open Job Render Log Externally', openJobRenderLog)
        openJobRenderLogAction.setObjectName('openJobRenderLogAction')
        openJobRenderLogAction.setEnabled(
            self.selectedRenderLogFilename is not None)
        actionButtonMenu.addSeparator()
        actionButtonMenu.addAction('Open KQ Log Externally', openKqLog)
        openAgentLogExternallyAction = actionButtonMenu.addAction(
            'Open Agent Log Externally', openAgentLog)
        openAgentLogExternallyAction.setObjectName(
            'openAgentLogExternallyAction')
        openAgentLogExternallyAction.setEnabled(not kq.IsDisabled())

    @QtCore.pyqtSlot()
    def on_restartKatanaQueueButton_clicked(self):
        """
        Slot that is called when the B{Restart Katana Queue} button is clicked.
        """
        # Disable button temporarily.
        self.restartKatanaQueueButton.setEnabled(False)
        QtCore.QTimer.singleShot(10000,
                                 self.__enableRestartKatanaQueueButton)

        def restart():
            kq.StopQueue()
            kq.StartQueue()

        # Schedule restart for the next tick, to give the button a chance to
        # get disabled.
        QtCore.QTimer.singleShot(1, restart)

    def __enableRestartKatanaQueueButton(self):
        """
        Slot that is called from a single-shot timer to re-enable the B{Restart
        Katana Queue} button that was disabled previously.
        """
        self.restartKatanaQueueButton.setEnabled(True)


PluginRegistry = [("KatanaPanel", 2.0, "Katana Queue", KatanaQueueTab)]
