# Copyright (c) 2018 The Foundry Visionmongers Ltd. All Rights Reserved.

import os
import zipfile

import ConfigurationAPI_cmodule as Configuration
import LookFileBakeAPI
from LookFileBakeAPI import LookFileBakeException
from .KlfLookFileBake import KlfLookFileBakeOutputFormat

class KlfZipLookFileBakeOutputFormat(KlfLookFileBakeOutputFormat):
    """
    Class implementing a KLF ZIP archive look file bake output format.
    """

    DisplayName = "as archive"
    FileExtension = "klf"
    PassFileExtension = "klf"

    def writeSinglePass(self, passData):
        """
        @type passData: C{LookFileBakeAPI.LookFilePassData}
        @rtype: C{list} of C{str}
        @param passData: The data representing a single look file pass to be
            baked.
        @return: A list of paths to files which have been written.
        """
        # Create a temp path to write files to
        tmpDir = os.path.join(Configuration.get('KATANA_TMPDIR'),
                              'lookfileBakeDir')

        # If the directory doesn't exist, then try to create it
        LookFileBakeAPI.CreateLookFileDirectory(tmpDir)

        # Form the file path for this pass
        tmpPath = os.path.join(tmpDir, os.path.basename(passData.filePath))

        # Modify the given pass data to contain the new temp file path
        passData["filePath"] = tmpPath

        # Get KlfLookFileBake to do the work of writing to a KLF file
        result = KlfLookFileBakeOutputFormat.writeSinglePass(self, passData)

        return result

    def postProcess(self, filePaths):
        """
        Performs post-processing after a set of passes has been written.

        Zips the files and cleans up the temp directory.

        @type filePaths: C{list} of C{str}
        @rtype: C{list} of C{str}
        @param filePaths: A list of files which have been written so far.
        @return: A list containing the ZIP file which was created.
        """
        # Get the name of the zip file from the "outputPath" setting
        outputPath = self._settings.outputPath

        # Zip up all the files
        try:
            with zipfile.ZipFile(outputPath, "w",
                                 compression=zipfile.ZIP_DEFLATED) as fileOut:
                for filePath in filePaths:
                    fileOut.write(filePath, os.path.basename(filePath))

        except IOError as exception:
            raise LookFileBakeException(
                "Writing to %s has failed: %s" % (outputPath, exception))

        # Clean up the temporary directory
        for filePath in filePaths:
            os.unlink(filePath)

        return [outputPath]


LookFileBakeAPI.RegisterOutputFormat(KlfZipLookFileBakeOutputFormat)
LookFileBakeAPI.SetDefaultOutputFormat(KlfZipLookFileBakeOutputFormat)
