# Copyright (c) 2018 The Foundry Visionmongers Ltd. All Rights Reserved.


import os
import time

import LookFileBakeAPI
import PyFnGeolibProducers as FnGeolibProducers
import PyXmlIO as XmlIO
from LookFileBakeAPI import LookFileBakeException

BaseOutputFormat = LookFileBakeAPI.BaseLookFileBakeOutputFormat


class KlfLookFileBakeOutputFormat(BaseOutputFormat):

    """
    Class implementing a KLF look file bake output format.
    """

    DisplayName = "as directory"
    FileExtension = ""
    PassFileExtension = "klf"

    def writeSinglePass(self, passData):
        """
        @type passData: C{LookFileBakeAPI.LookFilePassData}
        @rtype: C{list} of C{str}
        @param passData: The data representing a single look file pass to be
            baked.
        @return: A list of paths to files which have been written.
        """
        filePath = passData.filePath

        # If the enclosing directory doesn't exist, then try to create it
        LookFileBakeAPI.CreateLookFileDirectory(os.path.dirname(filePath))

        # TODO(TM): This is legacy. Consider removing "format" from KLF files.
        legacyFormatField = 2

        result = [filePath]

        attrsFilePath = "%s.attrs" % filePath

        try:
            attrWriter = FnGeolibProducers.BinaryAttrWriter(attrsFilePath)

        except RuntimeError as exception:
            raise LookFileBakeException(
                "Writing to %s has failed: %s" % (attrsFilePath, exception))

        # Add the attrs file to our list of paths to return
        result.append(attrsFilePath)

        root = XmlIO.Element('kstd')
        root.setAttr('time', time.ctime())
        root.setAttr('user', os.environ.get('USER', ''))
        root.setAttr('format', legacyFormatField)
        root.setAttr('katanaVersion', os.environ.get('KATANA_VERSION', ''))
        root.setAttr('katanaRelease', os.environ.get('KATANA_RELEASE', ''))

        if self._settings.sourceFile:
            root.setAttr('sourceFile', self._settings.sourceFile)
        if self._settings.sourceAsset:
            root.setAttr('sourceAsset', self._settings.sourceAsset)

        materials = root.addChild(XmlIO.Element('materials'))

        materialKeys = sorted(passData.materialDict.keys())

        attrOffsets = {}
        def writeBinaryAttr(attr, element):
            attrHash = attr.getHash()
            if attrHash in attrOffsets:
                element.setAttr('attroffset', attrOffsets[attrHash])
            else:
                pos = attrWriter.tell()
                attrOffsets[attrHash] = pos
                element.setAttr('attroffset', pos)
                attrWriter.writeAttr(attr)

        for i in materialKeys:
            materialType, materialAttr = passData.materialDict[i]
            material = materials.addChild(XmlIO.Element('material'))

            material.setAttr('type', materialType)
            material.setAttr('name', i)

            if materialAttr:
                writeBinaryAttr(materialAttr, material)

        if passData.sharedOverridesDict:
            sharedoverrides = root.addChild(XmlIO.Element('sharedoverrides'))

            keys = sorted(passData.sharedOverridesDict.keys())

            for key in keys:
                locationElement = sharedoverrides.addChild(
                    XmlIO.Element('overridegroup'))
                locationElement.setAttr('id', key)

                overrideDict = passData.sharedOverridesDict[key]

                for name, attr in overrideDict.items():
                    override = locationElement.addChild(
                        XmlIO.Element('override'))
                    override.setAttr('name', name)

                    if attr:
                        writeBinaryAttr(attr, override)

        for outputDict, rootName, rootType in passData.outputDictList:
            overrides = root.addChild(XmlIO.Element('overrides'))
            overrides.setAttr('rootid', rootName)
            overrides.setAttr('roottype', rootType)

            for location, overrideDict in outputDict.items():
                locationElement = overrides.addChild(
                    XmlIO.Element('overridegroup'))
                locationElement.setAttr('path', location)

                if isinstance(overrideDict, dict):
                    for name, attr in overrideDict.items():
                        override = locationElement.addChild(
                            XmlIO.Element('override'))
                        override.setAttr('name', name)

                        if attr:
                            writeBinaryAttr(attr, override)
                else:
                    locationElement.setAttr('sharedid', overrideDict)

        if passData.rootOverrideDict:
            overrides = root.addChild(XmlIO.Element('rootoverrides'))
            for name, attr in passData.rootOverrideDict.items():
                override = overrides.addChild(XmlIO.Element('override'))
                override.setAttr('name', name)

                if attr:
                    writeBinaryAttr(attr, override)

        try:
            with open(filePath, 'w') as f:
                root.write(f)
        except IOError as exception:
            raise LookFileBakeException(
                    "Writing to %s has failed: %s" % (filePath, exception))

        return result



LookFileBakeAPI.RegisterOutputFormat(KlfLookFileBakeOutputFormat)
