# Copyright (c) 2020 The Foundry Visionmongers Ltd. All Rights Reserved.
"""
Module containing an implementation of L{FarmAPI.BaseFarmPlugin} that provides
a bridge to I{Katana Queue}, Katana's local render queue.

Depends on the C{kq} Python package.
"""


import json
import logging
import os.path
import shlex
import shutil
import signal
import sys

from Callbacks import Callbacks
from Katana import (
    Configuration,
    FarmAPI,
    Nodes2DAPI,
)

from RenderingAPI import RenderPlugins

import kq
from kq.Client.KatanaQueue import GetSharedKatanaQueueWrapper
from kq.Job import Job
from kq.Task import Task
from kq.Util.LoggingConfig import GetBaseDirectory

log = logging.getLogger('kq.KatanaQueueFarmPlugin')


# Class Definitions -----------------------------------------------------------

class KatanaQueueFarmPlugin(FarmAPI.BaseFarmPlugin):
    """
    Implementation of L{FarmAPI.BaseFarmPlugin} that enables Katana to send
    remote render jobs to the I{Katana Queue} (which runs locally on the user's
    workstation).
    """

    # Mapping of Katana Queue job state to FarmAPI job state
    kJobStateMap = {
        Job.kNoState: FarmAPI.Job.State.kWaiting,
        Job.kSubmitted: FarmAPI.Job.State.kWaiting,
        Job.kAllocated: FarmAPI.Job.State.kRunning,
        Job.kRunning: FarmAPI.Job.State.kRunning,
        Job.kFinished: FarmAPI.Job.State.kCompleted,
        Job.kFailed: FarmAPI.Job.State.kFailed,
        Job.kStopping: FarmAPI.Job.State.kCancelled,
        Job.kStopped: FarmAPI.Job.State.kCancelled,
    }

    # Tuple of names of environment variables that must not be overwritten by a
    # KQ Agent environment file specified via `KQ_AGENT_ENVIRONMENT_FILE`
    kProtectedEnvironmentVariables = (
        '__KATANA_CONTROLLING_PROCESS',
    )

    # Initializer -------------------------------------------------------------

    def __init__(self):
        """
        Initializes an instance of the class.
        """
        self.queueClientWrapper = GetSharedKatanaQueueWrapper()

        self.agentEnvironmentSpec = self._getAgentEnvironmentSpec()

        # Flag to avoid spamming the log, as `getJobs()` is call periodically.
        self.canLogGetJobsFailure = True

    # Instance Methods --------------------------------------------------------

    def _getAgentEnvironmentSpec(self):
        """
        @rtype: C{dict}
        @return: A dictionary with names and values of environment variables
            to set for running Agents.
        """
        agentEnvironmentFilename = os.getenv('KQ_AGENT_ENVIRONMENT_FILE')
        if not agentEnvironmentFilename:
            return {}

        if not os.path.exists(agentEnvironmentFilename):
            log.warning('File referred to in KQ_AGENT_ENVIRONMENT_FILE '
                        'environment variable not found: "%s": no custom '
                        'Agent environment will be exported.',
                        agentEnvironmentFilename)
            return {}

        result = {}
        protectedEnvironmentVariables = \
            KatanaQueueFarmPlugin.kProtectedEnvironmentVariables

        with open(agentEnvironmentFilename, 'r') as agentEnvironmentFile:
            for i, line in enumerate(agentEnvironmentFile.readlines(),
                                     start=1):
                # Ignore empty lines
                if not line.strip():
                    continue

                if '=' not in line:
                    log.warning('%s: Line %d does not contain a "=" '
                                'character: "%s". Line ignored.',
                                agentEnvironmentFilename, i, line.strip())
                    continue

                name, value = line.split('=', 1)
                name = name.strip()
                value = value.strip()
                if name:
                    # Check if the given name matches the name of an
                    # environment variable that must not be overwritten
                    if name in protectedEnvironmentVariables:
                        log.warning('%s: Line %d uses the name of an '
                                    'environment variable that is protected: '
                                    '"%s". Line ignored.',
                                    agentEnvironmentFilename, i, line.strip())
                        continue

                    result[name] = value
                else:
                    log.warning('%s: Line %d does not match `name=value` '
                                'pattern: "%s". Line ignored.',
                                agentEnvironmentFilename, i, line.strip())

        return result

    def getRemoteInstallationMetadata(self):
        # For users running KQ with remote agents KQ_OVERRIDE_KATANA_ROOT
        # provides a mechanism to override the default KATANA_ROOT (in the case
        # where Katana is installed our mounted at a different path on the
        # host running the agent. If not specified KATANA_ROOT is used.
        katanaRoot = os.getenv('KQ_OVERRIDE_KATANA_ROOT',
                               default=os.getenv('KATANA_ROOT'))

        # Build a mapping with names of registered render plug-ins as keys, and
        # absolute paths to the folders containing the renderers` DSOs as
        # values, e.g.
        # {
        #     'dl': ('/usr/local/3delight-2.1.4/Linux-x86_64/'
        #            '3DelightForKatana/Libs'),
        #     'prman': ('/opt/pixar/RenderManForKatana-23.3-katana3.5/plugins/'
        #               'Resources/PRMan23/Libs'),
        #     'customRenderer': '/path/to/customRenderer/Libs'
        # }
        # Allow user to override install path of each renderer's plugin path
        renderPluginDsoPaths = {}
        for rendererName in RenderPlugins.GetRendererPluginNames(
                includeViewer=False, includeInternal=True):
            pluginPathOverrideKey = 'KQ_OVERRIDE_{}_PLUGIN_PATH'.format(
                rendererName.upper())
            renderPluginDsoPaths[rendererName] =\
                os.getenv(pluginPathOverrideKey,
                          RenderPlugins.GetRendererPluginDir(rendererName))

        renderbootPath = os.path.join(katanaRoot, 'bin', 'renderboot')

        return FarmAPI.BaseFarmPlugin.InstallationMetadata(
            katanaRoot,
            renderPluginDsoPaths,
            renderbootPath
        )

    def getEnvironmentForJob(self, renderContextName):
        # pylint: disable=unused-argument
        result = self.agentEnvironmentSpec.copy()

        foundryLicense = os.getenv('foundry_LICENSE')
        if foundryLicense:
            result['foundry_LICENSE'] = foundryLicense

        return result

    def _projectDirectoryForContext(self, renderContextName):
        """
        Produces the path to a unique directory for the specified
        C{renderContextName}.

        @type renderContextName: C{str}
        @rtype: C{str}
        @param renderContextName: A globally unique identifier for the render
            job for which to create a tempoary directory.
        @return: Full path to a unique directory for the given render context.
        """
        baseDirectory = GetBaseDirectory()
        projectDirectory = os.path.join(baseDirectory, 'kq')
        return os.path.join(projectDirectory, renderContextName)

    def createTemporaryDirectory(self, renderContextName):
        renderDirectory = self._projectDirectoryForContext(renderContextName)
        os.makedirs(renderDirectory)
        return renderDirectory

    def copyRenderFiles(self, renderFiles, renderContextName):
        _projectDirectory = self._projectDirectoryForContext(renderContextName)

        result = {}
        for renderFile in renderFiles:
            fileName = os.path.basename(renderFile)
            remoteFile = os.path.join(_projectDirectory, fileName)
            shutil.copy(renderFile, remoteFile)
            result[renderFile] = remoteFile
        return result

    def _extractRendererName(self, commandLine):
        """
        Extracts the renderer plugin name that will be used when the specified
        command line is run.

        @type commandLine: C{str}
        @rtype: C{str} or C{None}
        @param commandLine: The command line that should be parsed.
        @return: The renderer name if one is present in the command line.
        """
        arguments = shlex.split(commandLine)
        for i in range(0, len(arguments)):
            if arguments[i] == '-renderer':
                return arguments[i + 1]
        return None

    def _getTerminationSignalForCommand(self, command):
        # Windows doesn't have the concept of signal termination.
        if sys.platform.startswith('win'):
            return signal.SIGTERM

        terminationSignal = signal.SIGTERM

        # Each renderer may use a different signal to terminate and shutdown
        # cleanly so use the associated environment variable determine which
        # to use.
        renderer = self._extractRendererName(command)
        if not renderer:
            return terminationSignal

        return Nodes2DAPI.GetRenderTerminationSignal(renderer)

    # pylint: disable=missing-docstring
    def submitJob(self, jobRequest):
        # Create a new job and submit to the farm

        # If the client is not responsive, don't attempt to submit the job, as
        # this would stall the UI thread.
        queueClient = self.queueClientWrapper.getClient()
        if queueClient is None or not queueClient.isAlive():
            log.error('Failed to submit job. Katana Queue not available.')
            return ''

        environment = ['{key}={value}'.format(key=key, value=value)
                       for key, value in sorted(
                           jobRequest.environment.items())]

        job = Job(jobRequest.nodeName)
        for command in jobRequest.commands:
            # Determine which termination signal we should use.
            terminationSignal = self._getTerminationSignalForCommand(command)
            task = Task(command, environment, terminationSignal,
                        jobRequest.frames)
            job.appendTask(task)

        queueClient.submitJob(job)

        # Return string repr of the uuid4 object that identifies this job.
        return str(job.jobId)

    def _getJobs(self, filters):
        # Gets the list of jobs.

        # If the client is not responsive, don't attempt to get jobs, as this
        # would stall the UI thread.
        queueClient = self.queueClientWrapper.getClient()
        if queueClient is None or not queueClient.isAlive():
            if self.canLogGetJobsFailure:
                self.canLogGetJobsFailure = False
                log.error('Failed to get jobs. Katana Queue not available.')
            return []
        self.canLogGetJobsFailure = True

        return queueClient.getJobs(filters=filters)

    def getRenderLogFilename(self, jobId):
        # Query for just our job.
        jobs = self._getJobs(filters=['jobId={}'.format(jobId)])
        if not jobs:
            raise FarmAPI.JobNotFoundException(jobId,
                                               'Retrieve render log filename')
        return jobs[0].logFilename

    def getJobState(self, jobId):
        # Query for just our job.
        jobs = self._getJobs(filters=['jobId={}'.format(jobId)])
        if not jobs:
            raise FarmAPI.JobNotFoundException(jobId, 'Retrieve job state')

        job = jobs[0]
        return KatanaQueueFarmPlugin.kJobStateMap[job.state]

    def getJobs(self, filters):
        # Convert filters to match our internal representation
        kqFilters = []
        for jobFilter in filters:
            if jobFilter.startswith('jobId'):
                kqFilters.append(jobFilter.replace('jobId', 'jobId'))
            else:
                kqFilters.append(jobFilter)

        jobs = self._getJobs(kqFilters) or []

        # Convert our job representation to Katana's representation.
        result = []
        for job in jobs:
            result.append(FarmAPI.Job(job.jobId, job.jobName,
                                      job.submissionTime, job.completionTime,
                                      KatanaQueueFarmPlugin.kJobStateMap[
                                          job.state],
                                      job.logFilename))

        return result

    def startJob(self, jobId):
        log.info('Starting job "%s"...', jobId)

        # If the client is not responsive, don't attempt to start the job, as
        # this would stall the UI thread.
        queueClient = self.queueClientWrapper.getClient()
        if queueClient is None or not queueClient.isAlive():
            log.error('Failed to start job "%s". Katana Queue not available.',
                      jobId)
            return

        jobs = self._getJobs(filters=['jobId={}'.format(jobId)])
        if not jobs:
            raise FarmAPI.JobNotFoundException(jobId, 'Start job')

        # Only start the Job if it's not currently in a running state
        job = jobs[0]
        if job.state in Job.kRunningStates:
            log.info('Job "%s" is already running (state: "%s").', job.jobName,
                     job.state)
        else:
            if not queueClient.startJob(job):
                log.error('Job "%s" could not be started.', job.jobName)

    def stopJob(self, jobId):
        log.info('Stopping job "%s"...', jobId)

        # If the client is not responsive, don't attempt to stop the job, as
        # this would stall the UI thread.
        queueClient = self.queueClientWrapper.getClient()
        if queueClient is None or not queueClient.isAlive():
            log.error('Failed to stop job "%s". Katana Queue not available.',
                      jobId)
            return

        jobs = self._getJobs(filters=['jobId={}'.format(jobId)])
        if not jobs:
            raise FarmAPI.JobNotFoundException(jobId, 'Stop job')

        # Only stop the Job if it's not currently in a stoppped state
        job = jobs[0]
        if job.state in Job.kStoppedStates:
            log.info('Job "%s" has already been stopped (state: "%s").',
                     job.jobName, job.state)
        else:
            if not queueClient.stopJob(job):
                log.error('Job "%s" could not be stopped.', job.jobName)


def StartKatanaQueue(objectHash):
    """
    Starts the Katana Queue (kq).
    """
    # pylint: disable=unused-argument
    import kq

    kq.StartQueue()


# Only register the plug-in if Katana Queue is enabled
if kq.IsDisabled():
    log.info('KQ is disabled. '
             'Katana Queue farm plug-in will not be registered.')
else:
    if Configuration.get('KATANA_UI_MODE'):
        Callbacks.addCallback(Callbacks.Type.onStartupComplete,
                              StartKatanaQueue)

    PluginRegistry = [('FarmPlugin', 2.0, 'Katana Queue',
                       KatanaQueueFarmPlugin)]
