// Copyright (c) 2018 The Foundry Visionmongers Ltd. All Rights Reserved.

#ifndef KATANA_PLUGINS_PROFILINGMOCKRENDERER_PROFILINGMOCKRENDERPLUGIN_H_
#define KATANA_PLUGINS_PROFILINGMOCKRENDERER_PROFILINGMOCKRENDERPLUGIN_H_

#include <string>

#include <FnAttribute/FnAttribute.h>
#include <FnRender/plugin/RenderBase.h>

namespace ProfilingMockRenderer
{
/**
 * \defgroup ProfilingMockRenderer Profiling Mock Renderer Plug-in
 * \{
 */

/**
 * \brief The <b>Profiling Mock Renderer</b> is a \em mock renderer plug-in
 * that provides scene traversal profiling information.
 *
 * The Profiling Mock Renderer performs no rendering, but instead traverses the
 * entire scene graph as quickly as possible using a given traversal strategy.
 *
 * <b>Preview Render</b> and <b>Disk Render</b> (from a <b>Render</b> node) are
 * the two render methods supported. When rendering a scene with either of
 * these methods, the plug-in will time and log how long it took for a location
 * to be expanded, as well as the rate of locations expanded per second.
 * In UI mode, the log will be visible in the <b>Render Log</b> tab. This is
 * the format in which the log messages are reported:
 * \code
 * >> Depth-first search: 0.00s 0.00 ms/loc        1  /root
 * >> Depth-first search: 0.00s 0.48 ms/loc        2  /root/world
 * >> Depth-first search: 0.00s 0.45 ms/loc        3  /root/world/geo
 * >> Depth-first search: 0.01s 2.03 ms/loc        4  /root/world/geo/primitive
 * >> Depth-first search: 0.01s 1.70 ms/loc        5  /root/world/cam
 * >> Depth-first search: 0.01s 1.46 ms/loc        6  /root/world/cam/camera
 * >> Depth-first search: 0.01s 1.31 ms/loc        7  /root/world/lgt
 * >> Depth-first search: 0.01s 1.19 ms/loc        8  /root/world/lgt/light
 *    --------1---------  --2-- -----3-----       -4- -----------5-------------
 *
 *  - 1 Traversal strategy log prefix
 *  - 2 Elapsed time since the beginning of the render
 *  - 3 Current traversal rate (time per location)
 *  - 4 Number of locations that have been expanded
 *  - 5 The current location path
 * \endcode
 *
 * <b>Traversal strategies</b>
 *
 * Three traversal strategies are available:
 *  - Depth-first search (\c dfs)
 *  - Breadth-first search (\c bfs)
 *  - Parallel search (\c parallel)
 *
 * <b>Depth-first search</b>
 *
 * This approach starts at the root location and explores as far as possible
 * along each branch before backtracking.
 *
 * Depth-first traversal within a single thread makes best use of scene data
 * caching within that thread's allocated Geolib runtime, while minimising
 * memory consumption (matching the current best practice for single-threaded
 * rendering).
 *
 * <b>Breadth-first search</b>
 *
 * This approach starts at the root location and explores all of the neighbor
 * locations at the present depth prior to moving on to the locations at the
 * next depth level.
 *
 * Breadth-first traversal within a single thread makes poor use of scene data
 * caching. Evaluating data from distant subtrees evicts data from previously
 * computed locations. By the time the program execution returns to a certain
 * subtree, all its previous data may have already been evicted.
 *
 * <b>Parallel search</b>
 *
 * This approach uses multiple threads to traverse subtrees, with each thread
 * using a depth-first search. Idle threads acquire work from busy threads by
 * stealing their \em oldest, and therefore shallowest task. This constitutes
 * breadth-first division of work across multiple threads. Since different
 * threads spawn distinct Geolib runtimes for concurrent evaluation, this
 * strategy aims to minimise redundancy in scene cooking.
 *
 * See <a href="https://software.intel.com/en-us/node/506103">How Task
 * Scheduling Works</a> for detailed information.
 *
 * <b>Global parameters</b>
 *
 * The <b>ProfilingMockRendererGlobalSettings</b> node provides several
 * configuration parameters:
 * \li \c profilingMockRendererGlobalStatements.options.traversalStrategy:
 *     Available traversal strategies are \c dfs, \c bfs and \c parallel.
 *     Default is \c dfs.
 * \li \c profilingMockRendererGlobalStatements.options.maxLogDepth:
 *     Log timing information for locations up to this depth. Default is \c 5.
 * \li \c profilingMockRendererGlobalStatements.options.maxThreads:
 *     Max threads to use for the traversal. Relevant for the \c parallel
 *     traversal strategy only. If set to \c 0, the number of threads will be
 *     automatically determined based on hardware configuration. Default is
 *     \c 0.
 *
 * \note Profiling Mock Renderer plug-in does not support live rendering.
 */
class RenderPlugin : public FnKat::Render::RenderBase
{
public:
    /** \brief Constructor.
     *
     * \param rootIterator The root \c FnScenegraphIterator used to traverse
     *     the Katana recipe.
     * \param arguments Render arguments (e.g. render time).
     */
    RenderPlugin(FnKat::FnScenegraphIterator rootIterator,
                 FnAttribute::GroupAttribute arguments);

    /** \brief Destructor. */
    ~RenderPlugin() override;

    /** \brief Constructs an intance of the class.
     *
     * This is used by the \c DEFINE_RENDER_PLUGIN macro.
     *
     * \param rootIterator The root \c FnScenegraphIterator used to traverse
     *     the Katana recipe.
     * \param arguments Render arguments (e.g. render time).
     * \return A pointer to the new instance.
     */
    static FnKat::Render::RenderBase* create(
        FnKat::FnScenegraphIterator rootIterator,
        FnAttribute::GroupAttribute arguments);

    static void flush() {}

    /* RenderBase Methods */

    /** \brief Reimplemented from \c RenderBase.
     *
     * This function is called at the start of the render process.
     *
     * Reads the configurable renderer global setting from the \c /root
     * location.
     */
    int start() override;

    /** \brief Reimplemented from \c RenderBase.
     *
     * Called at the end of the render process. The Profiling Mock Renderer
     * does not perform any action when this message is received.
     */
    int stop() override;

    /** \brief Reimplemented from \c RenderBase.
     *
     * The Profiling Mock Renderer disables the monitor output, as no image is
     * generated (\c Render::RenderAction::setLoadOutputInMonitor).
     */
    void configureDiskRenderOutputProcess(
        FnKat::Render::DiskRenderOutputProcess& diskRenderOutputProcess,
        const std::string& outputName,
        const std::string& outputPath,
        const std::string& renderMethodName,
        const float& frameTime) const override;
};

/** \} */

}  // namespace ProfilingMockRenderer

#endif  // KATANA_PLUGINS_PROFILINGMOCKRENDERER_PROFILINGMOCKRENDERPLUGIN_H_
