// Copyright (c) 2017 The Foundry Visionmongers, Ltd.

#include <FnGeolib/op/FnGeolibOp.h>
#include <FnAttribute/FnAttribute.h>
#include <FnGeolibServices/FnGeolibCookInterfaceUtilsService.h>
#include <mutex>

namespace
{

static FnKat::Attribute 
lookupMetaShaderParam(FnKat::GeolibCookInterface& interface,
                      const std::string & shaderName)
{
    FnKat::StringAttribute nameAttr = interface.getAttr("material.meta." + shaderName);

    if (nameAttr.isValid())
    {
        FnKat::StringAttribute::array_type values = nameAttr.getNearestSample(0.0);

        for (const auto& value : values)
        {
            FnKat::Attribute shaderAttr =
                interface.getAttr(std::string("material.") + value);

            if (shaderAttr.isValid())
                return shaderAttr;
        }
    }

    return FnKat::Attribute();
}

/// Specifically look for a Hydra shader.
static bool lookupHydraParam(FnKat::GeolibCookInterface& interface,
                             std::string const& type)
{
    FnKat::StringAttribute attr = interface.getAttr("material.hydraLightParams.type");
    if (attr.isValid())
    {
        std::string attrValue = attr.getValue("", false);
        return attrValue == type;
    }

    return false;
}

/// Determine whether this is a Spot light.
static bool isSpotLight(FnKat::GeolibCookInterface& interface)
{
    // Early check for Hydra shaders.
    if (lookupHydraParam(interface, "spot"))
        return true;

    /// Not a Hydra shader, so keep looking.
    FnKat::FloatAttribute attr = lookupMetaShaderParam(interface, "coneAngleName");
    if (attr.isValid())
        return true;

    static const std::string attrNames[] =
    {
        "material.prmanLightParams.Cone_Outer_Angle",
        "material.prmanLightParams.OuterAngle",
        "material.arnoldLightParams.wide_angle",
        "material.arnoldLightParams.cone_angle",
        "material.testLightParams.coneAngle",
        "material.parameters.Cone_Outer_Angle",
        "material.parameters.OuterAngle",
        "material.parameters.wide_angle",
        "material.parameters.cone_angle",
        "material.viewerLightParams.Cone_Outer_Angle",
        "material.parameters.coneAngle",
        "material.vrayLightParams.coneAngle",
    };

    for (const auto& attrName : attrNames)
    {
        attr = interface.getAttr(attrName);
        if (attr.isValid())
            return true;
    }

    return false;
}

/// Determine whether this is a Distance light.
static bool isDistantLight(FnKat::GeolibCookInterface& interface)
{
    // Early check for Hydra shaders.
    if (lookupHydraParam(interface, "distant"))
        return true;

    /// Not a Hydra shader, so keep looking.
    static const std::string attrNames[] =
    {
        "material.arnoldLightParams.direction",
        "material.parameters.direction",
        "material.testLightParams.physical_sun",
        "material.vrayLightParams.beamRadius",
        "material.parameters.beamRadius",
        "material.vrayLightParams.ozone",
        "material.parameters.ozone",
    };

    for (const auto& attrName : attrNames)
    {
        // One of these attributes only needs to be valid
        if (interface.getAttr(attrName).isValid())
            return true;
    }

    return false;
}

/// Determine whether this is a Quad light.
static bool isQuadLight(FnKat::GeolibCookInterface& interface)
{
    static const std::string attrNames[] =
    {
        "material.arnoldLightParams.vertices",
        "material.parameters.vertices",
        "material.testLightParams.disc",
    };

    for (const auto& attrName : attrNames)
    {
        // These attributes only need to be valid
        if (interface.getAttr(attrName).isValid())
            return true;
    }

    // V-Ray Rectangle Light
    FnKat::FloatAttribute uSizeAttr =
        interface.getAttr("material.vrayLightParams.u_size");
    FnKat::FloatAttribute vSizeAttr =
        interface.getAttr("material.vrayLightParams.v_size");
    if (uSizeAttr.isValid() && vSizeAttr.isValid())
        return true;

    uSizeAttr = interface.getAttr("material.parameters.u_size");
    vSizeAttr = interface.getAttr("material.parameters.v_size");
    if (uSizeAttr.isValid() && vSizeAttr.isValid())
        return true;

    return false;
}

/// Determine whether this is a Mesh light.
static bool isMeshLight(FnKat::GeolibCookInterface& interface)
{
    if (interface.getAttr("material.arnoldLightParams.mesh").isValid())
        return true;

    return false;
}

/// Determine whether this is a Sphere light.
static bool isSphereLight(FnKat::GeolibCookInterface& interface)
{
    static const std::string attrNames[] =
    {
        "material.vrayLightParams.radius",
        "material.parameters.radius",
    };

    FnKat::FloatAttribute attr;
    for (const auto& attrName : attrNames)
    {
        attr = interface.getAttr(attrName);
        if (attr.isValid())
            return true;
    }

    return false;
}

/// Determine whether this is a dome light.
static bool isDomeLight(FnKat::GeolibCookInterface& interface)
{
    static const std::string attrNames[] =
    {
        "material.vrayLightParams.dome_targetRadius",
        "material.parameters.dome_targetRadius",
    };

    for (const auto& attrName : attrNames)
    {
        if (interface.getAttr(attrName).isValid())
            return true;
    }

    return false;
}

// "Light-Type Meta Data" - op that generates attribute on light locations.
class LightTypeMetaDataOp : public Foundry::Katana::GeolibOp
{
public:
    // Boilerplate that indicates the Op's cook() function is safe to be called
    // concurrently.
    static void setup(Foundry::Katana::GeolibSetupInterface& interface)
    {
        interface.setThreading(
            Foundry::Katana::GeolibSetupInterface::ThreadModeConcurrent);
    }

    static void cook(Foundry::Katana::GeolibCookInterface& interface)
    {
        FnAttribute::StringAttribute celAttr = interface.getOpArg("CEL");
        if (celAttr.isValid())
        {
            FnGeolibServices::FnGeolibCookInterfaceUtils::MatchesCELInfo matchInfo;
            FnGeolibServices::FnGeolibCookInterfaceUtils::matchesCEL(
                matchInfo, interface, celAttr);

            if (!matchInfo.canMatchChildren)
            {
                interface.stopChildTraversal();
            }

            if (!matchInfo.matches)
            {
                return;
            }
        }

        const char* lightType = nullptr;

        if (isSpotLight(interface))
            lightType = "spot";
        else if (isDistantLight(interface))
            lightType = "distant";
        else if (isQuadLight(interface))
            lightType = "quad";
        else if (isMeshLight(interface))
            lightType = "mesh";
        else if (isSphereLight(interface))
            lightType = "sphere";
        else if (isDomeLight(interface))
            lightType = "dome";
        else
            lightType = "omni";

        if (lightType)
        {
            if (!interface.getAttr("viewer.lightType").isValid())
            {
                FnAttribute::StringAttribute typeAttr(lightType);

                interface.setAttr("viewer.lightType", typeAttr);
            }
        }
    }
};
DEFINE_GEOLIBOP_PLUGIN(LightTypeMetaDataOp)
}  // namespace

void registerPlugins()
{
    REGISTER_PLUGIN(LightTypeMetaDataOp, "LightTypeMetaData", 0, 1);
}
