#include <vector>

#include <FnAttribute/FnAttribute.h>
#include <FnAttribute/FnGroupBuilder.h>
#include <FnAttribute/FnDataBuilder.h>

#include <FnPluginSystem/FnPlugin.h>

#include <FnGeolib/op/FnGeolibOp.h>

namespace { //anonymous

class HierarchyCopyOp : public Foundry::Katana::GeolibOp
{
public:

    class ArgsUpdater
    {
    public:
        ArgsUpdater(Foundry::Katana::GeolibCookInterface &interface)
        : m_interface(interface)
        , m_argsUpdated(false)
        {}

        FnAttribute::GroupBuilder & get()
        {
            if (!m_argsUpdated)
            {
                m_gb.update(m_interface.getOpArg());
                m_argsUpdated = true;
            }
            return m_gb;
        }

        ~ArgsUpdater()
        {
            if (m_argsUpdated)
            {
                m_interface.replaceChildTraversalOp(m_interface.getOpType(),
                            m_gb.build());
            }
        }

    private:
        Foundry::Katana::GeolibCookInterface & m_interface;
        bool m_argsUpdated;
        FnAttribute::GroupBuilder m_gb;

    };

    static void setup(Foundry::Katana::GeolibSetupInterface &interface)
    {
        interface.setThreading(
            Foundry::Katana::GeolibSetupInterface::ThreadModeConcurrent);
    }

    static void cook(Foundry::Katana::GeolibCookInterface &interface)
    {
        //will update child arguments if necessary upon destruction
        ArgsUpdater argsUpdater(interface);

        bool canMatchChildren = false;

        const std::string rootLocationPath = interface.getRootLocationPath();
        FnAttribute::GroupAttribute entries = interface.getOpArg("entries");

        Foundry::Katana::CreateLocationInfo createLocationInfo;

        // Avoid std::string reallocations by allocating a reasonable buffer up-
        // front and clearing back to the "entries." prefix each iteration.
        constexpr int kEntryNamePrefixLen = 8;  // =strlen("entries.")
        std::string entryName;
        entryName.reserve(256);
        entryName += "entries.";

        for (int64_t i = 0, e = entries.getNumberOfChildren(); i < e; ++i)
        {
            FnAttribute::GroupAttribute entry = entries.getChildByIndex(i);

            entryName.erase(kEntryNamePrefixLen);
            entryName += entries.getChildNameCStr(i);

            FnAttribute::StringAttribute src = entry.getChildByName(
                "src", static_cast<int32_t>(strlen("src")));

            // The index of the node inputs to copy the hierarchy from
            FnAttribute::IntAttribute srcIndexAttr = entry.getChildByName(
                "srcIndex", static_cast<int32_t>(strlen("srcIndex")));

            bool entryCanMatchChildren = false;

            if (!src.isValid())
            {
                argsUpdater.get().del(entryName);
                continue;
            }

            FnAttribute::StringAttribute dst = entry.getChildByName(
                "dst", static_cast<int32_t>(strlen("dst")));
            if (dst.isValid())
            {
                FnAttribute::StringConstVector dstValues =
                        dst.getNearestSample(0);

                std::vector<bool> omitIndices(dstValues.size(), false);
                size_t numOmitIndices = 0;

                size_t index = 0;
                for (FnAttribute::StringConstVector::const_iterator I =
                        dstValues.begin(), E = dstValues.end(); I != E;
                                ++I, ++index)
                {
                    // noticed that an empty string will keep building
                    // forever. Probably something that should be fixed
                    // in CreateLocation. For now, let's validate here.
                    if (strncmp((*I), rootLocationPath.c_str(),
                                rootLocationPath.size()) != 0)
                    {
                        omitIndices[index] = true;
                        ++numOmitIndices;
                        continue;
                    }


                    Foundry::Katana::CreateLocation(
                            createLocationInfo, interface, (*I));

                    if (createLocationInfo.atLeaf)
                    {
                        const int srcIndex = srcIndexAttr.getValue(
                            kFnKatGeolibDefaultInput, false);

                        interface.replaceAttrs(
                            src.getValue("", false), srcIndex);
                        interface.replaceChildren(
                            src.getValue("", false), srcIndex);
                        omitIndices[index] = true;
                        ++numOmitIndices;
                    }
                    else if (createLocationInfo.canMatchChildren)
                    {
                        entryCanMatchChildren = true;
                    }
                    else
                    {
                        omitIndices[index] = true;
                        ++numOmitIndices;
                    }
                }


                if (entryCanMatchChildren)
                {
                    if (numOmitIndices > 0)
                    {
                        if (numOmitIndices < dstValues.size())
                        {
                            std::vector<const char*> newValues;
                            newValues.reserve(
                                dstValues.size() - numOmitIndices);

                            for (FnAttribute::StringConstVector::const_iterator
                                    I = dstValues.begin(), E = dstValues.end();
                                            I != E; ++I, ++index)
                            {

                                if (omitIndices[index])
                                {
                                    newValues.push_back((*I));
                                }
                            }

                            entryName += ".dst";
                            argsUpdater.get().set(
                                entryName,
                                FnAttribute::StringAttribute(
                                    newValues.data(), newValues.size(), 1));
                        }
                        else
                        {
                            // Shouldn't be necessary as we won't get an
                            // entryCanMatchChildren if we've removed everything
                            // being safe never hurts.
                            argsUpdater.get().del(entryName);
                        }
                    }

                    canMatchChildren = true;
                }
                else
                {
                    argsUpdater.get().del(entryName);
                }
            }
        }

        if (!canMatchChildren)
        {
            interface.stopChildTraversal();
        }
    }

};
DEFINE_GEOLIBOP_PLUGIN(HierarchyCopyOp)

} // anonymous

void registerPlugins()
{
    REGISTER_PLUGIN(HierarchyCopyOp, "HierarchyCopy", 0, 1);
}
