#include <iostream>
#include <string>

#include <FnAttribute/FnAttribute.h>
#include <FnAttribute/FnGroupBuilder.h>

#include <FnPluginSystem/FnPlugin.h>

#include <FnGeolib/op/FnGeolibOp.h>

#include <FnGeolib/util/Path.h>

#include <FnGeolibServices/FnGeolibCookInterfaceUtilsService.h>
#include <FnGeolibServices/FnMaterialResolveUtil.h>

namespace { //anonymous

// Creates, modifies or deletes attributes at scene graph locations.
//
// AttributeSetOp provides a flexible means of manipulating scene graph
// attributes. AttributeSetOp applies changes to the scene graph in batches
// of instructions. The format of a given set of instructions is as follows:
//
// Instruction
// -----------
//  - CEL/locationPaths [] - CEL expressions or scene graph locations to apply
//  operations to.
//  - setAttrs [] - GroupAttributes (name, value) to set at matching locations
//  - deleteAttrNames [] - attributes to delete at matching locations.
//  - execOps [] - GroupAttributes (opType, opArgs) to exec at matching
//  locations.
//
// In the common case AttributeSet contains only one set of Instructions at the
// root of the Op's op args i.e.:
//
// GroupAttribute
//      locationPaths: "/root"
//      setAttrs: [("foo", StringAttr("bar")]
//      deleteAttrNames: ["renderSettings"]
//      execOps: [("Prune", GroupAttribute(//Prune args//)]
//
// However AttributeSetOp can apply multiple sets of instructions during a
// single cook call. This can be achieved through the use of "batches" in which
// case the Op's args will look as follows:
//
// GroupAttribute
//      locationPaths: "/root"
//      setAttrs: [("foo", StringAttr("bar")]
//      deleteAttrNames: ["renderSettings"]
//      execOps: [("Prune", GroupAttribute(//Prune args//)]
//
//      batch.[batch_name_1]: (Same instruction format as above)
//      batch.[batch_name_2]: (Same instruction format as above)
class AttributeSetOp : public Foundry::Katana::GeolibOp
{
private:
    typedef std::vector<FnAttribute::GroupAttribute> AttributeSetGroups;

    // Called at each location this Op is executed at to determine if the
    // AttributeSet op should apply any changes to this location or child
    // locations.
    //
    // If the user specifies both locationPaths and CEL then locationPaths
    // will be used.
    static void matchesLocation(
        const Foundry::Katana::GeolibCookInterface& interface,
        const FnAttribute::GroupAttribute args,
        bool& canMatchChildren,
        bool& localMatch)
    {
        FnAttribute::StringAttribute locationsAttr =
            args.getChildByName("locationPaths");
        FnAttribute::StringAttribute celAttr =
            args.getChildByName("CEL");

        if (locationsAttr.isValid())
        {
            FnGeolibUtil::Path::FnMatchInfo matchInfo;
            FnAttribute::StringConstVector locationPaths =
                locationsAttr.getNearestSample(0.0);

            for (auto locationPath : locationPaths)
            {
                FnGeolibUtil::Path::FnMatch(
                    matchInfo, interface.getOutputLocationPath(), locationPath);
                if (matchInfo.match)
                    localMatch = true;
                if (matchInfo.canMatchChildren)
                    canMatchChildren = true;
                if (localMatch && canMatchChildren)
                    break;
            }
        }
        else
        {
            FnGeolibServices::FnGeolibCookInterfaceUtils::MatchesCELInfo
                matchInfo;
            FnGeolibServices::FnGeolibCookInterfaceUtils::matchesCEL(
                matchInfo, interface, celAttr);

            if (matchInfo.matches)
                localMatch = true;
            if (matchInfo.canMatchChildren)
                canMatchChildren = true;
        }
    }

    // Applies any modificiations to the attributes at the Op's current
    // location.
    static void apply(Foundry::Katana::GeolibCookInterface& interface,
                      const FnAttribute::GroupAttribute& args)
    {
        // setAttrs
        FnAttribute::GroupAttribute setAttrs = args.getChildByName("setAttrs");
        if (setAttrs.isValid())
        {
            for (int i = 0; i < setAttrs.getNumberOfChildren(); ++i)
            {
                FnAttribute::GroupAttribute groupAttr =
                    setAttrs.getChildByIndex(i);
                if (!groupAttr.isValid())
                    continue;

                FnAttribute::StringAttribute nameAttr =
                    groupAttr.getChildByName("name");
                FnAttribute::Attribute attr = groupAttr.getChildByName("attr");
                if (!nameAttr.isValid() || !attr.isValid())
                    continue;

                bool groupInherit = FnAttribute::IntAttribute(
                                        groupAttr.getChildByName("inherit"))
                                        .getValue(1, false);

                std::string name = nameAttr.getValue("", false);
                if (name.empty())
                    continue;

                interface.setAttr(name, attr, groupInherit);
            }
        }

        // deleteAttrs
        FnAttribute::StringAttribute deleteAttrNames =
            args.getChildByName("deleteAttrNames");
        if (deleteAttrNames.isValid())
        {
            FnAttribute::StringConstVector deleteAttrNameValues =
                deleteAttrNames.getNearestSample(0);

            for (auto deleteAttrNameValue : deleteAttrNameValues)
            {
                interface.deleteAttr(deleteAttrNameValue);
            }
        }

        // execOps
        FnAttribute::GroupAttribute execOps = args.getChildByName("execOps");
        if (execOps.isValid())
        {
            for (int i = 0; i < execOps.getNumberOfChildren(); ++i)
            {
                FnAttribute::GroupAttribute groupAttr =
                    execOps.getChildByIndex(i);
                if (!groupAttr.isValid())
                    continue;

                FnAttribute::StringAttribute opType =
                    groupAttr.getChildByName("opType");

                FnAttribute::GroupAttribute opArgs =
                    groupAttr.getChildByName("opArgs");

                if (!opType.isValid() || !opArgs.isValid())
                    continue;

                interface.execOp(opType.getValue("", false), opArgs);
            }
        }
    }

    class ChildTraversalManager
    {
    public:
        explicit ChildTraversalManager(
            Foundry::Katana::GeolibCookInterface& interface)
            : _continueToChildren(false), _interface(interface)
        {
        }

        ~ChildTraversalManager()
        {
            if (!_continueToChildren)
                _interface.stopChildTraversal();
        }

        void runAtChildren() { _continueToChildren = true; }

    private:
        bool _continueToChildren;
        Foundry::Katana::GeolibCookInterface& _interface;
    };

    static bool isInstructionValid(FnAttribute::GroupAttribute& instruction)
    {
        FnAttribute::GroupAttribute setAttrs =
            instruction.getChildByName("setAttrs");
        FnAttribute::StringAttribute deleteAttrNames =
            instruction.getChildByName("deleteAttrNames");
        FnAttribute::GroupAttribute execOps =
            instruction.getChildByName("execOps");

        if (!setAttrs.isValid() && !deleteAttrNames.isValid() &&
            !execOps.isValid())
        {
            return false;
        }

        FnAttribute::StringAttribute locationsAttr =
            instruction.getChildByName("locationPaths");
        FnAttribute::StringAttribute celAttr =
            instruction.getChildByName("CEL");

        if (!locationsAttr.isValid() && !celAttr.isValid())
        {
            return false;
        }
        return true;
    }

public:
    static void setup(Foundry::Katana::GeolibSetupInterface& interface)
    {
        interface.setThreading(
            Foundry::Katana::GeolibSetupInterface::ThreadModeConcurrent);
    }

    static void cook(Foundry::Katana::GeolibCookInterface& interface)
    {
        ChildTraversalManager childTraversalManager(interface);

        // Common case occurs where our arguments contain only one set
        // of AttributeSet instructions...
        FnAttribute::GroupAttribute batchAttr = interface.getOpArg("batch");
        if (!batchAttr.isValid())
        {
            FnAttribute::GroupAttribute instruction = interface.getOpArg();
            if (instruction.isValid())
            {
                if (!AttributeSetOp::isInstructionValid(instruction))
                    return;

                // Take the optimised path.
                bool canMatchChildren = false;
                bool localMatch = false;

                AttributeSetOp::matchesLocation(interface, instruction,
                                                canMatchChildren, localMatch);
                if (localMatch)
                {
                    AttributeSetOp::apply(interface, instruction);
                }

                if (canMatchChildren)
                    childTraversalManager.runAtChildren();
            }
        }
        else
        {
            const int64_t numBatchAttr = batchAttr.getNumberOfChildren();

            // We need to prepare to apply multiple AttributeSet operations.
            AttributeSetGroups attributeSetGroups;
            attributeSetGroups.reserve(1 + numBatchAttr);

            // Include top-level arg definition
            attributeSetGroups.push_back(interface.getOpArg());

            for (int64_t i = 0; i < numBatchAttr; ++i)
            {
                FnAttribute::GroupAttribute batchChild =
                    batchAttr.getChildByIndex(i);
                if (batchChild.isValid())
                    attributeSetGroups.push_back(batchChild);
            }

            for (auto& instruction : attributeSetGroups)
            {
                if (!AttributeSetOp::isInstructionValid(instruction))
                    continue;

                bool canMatchChildren = false;
                bool localMatch = false;
                AttributeSetOp::matchesLocation(interface, instruction,
                                                canMatchChildren, localMatch);

                if (localMatch)
                {
                    AttributeSetOp::apply(interface, instruction);
                }

                if (canMatchChildren)
                {
                    childTraversalManager.runAtChildren();
                }
            }
        }
    }
};

DEFINE_GEOLIBOP_PLUGIN(AttributeSetOp)

///////////////////////////////////////////////////////////////////////////////

bool matchesLocation(Foundry::Katana::GeolibCookInterface &interface,
        bool & outCanMatchChildren)
{
    bool localMatch = false;
    bool canMatchChildren = false;

    FnAttribute::StringAttribute locationsAttr =
            interface.getOpArg("locationPaths");

    FnAttribute::StringAttribute celAttr =
            interface.getOpArg("CEL");

    if (!locationsAttr.isValid() && !celAttr.isValid())
    {
        outCanMatchChildren = false;
        return false;
    }

    if (locationsAttr.isValid())
    {
        FnGeolibUtil::Path::FnMatchInfo matchInfo;
        FnAttribute::StringConstVector locationPaths =
            locationsAttr.getNearestSample(0.0);
        for (auto locationPath : locationPaths)
        {
            FnGeolibUtil::Path::FnMatch(
                matchInfo, interface.getOutputLocationPath(), locationPath);
            if(matchInfo.match) localMatch = true;
            if(matchInfo.canMatchChildren) canMatchChildren = true;
            if(localMatch && canMatchChildren) break;
        }
    }
    else
    {
        FnGeolibServices::FnGeolibCookInterfaceUtils::MatchesCELInfo matchInfo;
        FnGeolibServices::FnGeolibCookInterfaceUtils::matchesCEL(
                matchInfo, interface, celAttr);

        if(matchInfo.matches) localMatch = true;
        if(matchInfo.canMatchChildren) canMatchChildren = true;
    }

    outCanMatchChildren = canMatchChildren;
    return localMatch;
}

///////////////////////////////////////////////////////////////////////////////


class AttributeSetIfNotPresentOp : public Foundry::Katana::GeolibOp
{
public:
    static void setup(Foundry::Katana::GeolibSetupInterface &interface)
    {
        interface.setThreading(
                Foundry::Katana::GeolibSetupInterface::ThreadModeConcurrent);
    }


    static void cook(Foundry::Katana::GeolibCookInterface &interface)
    {

        bool canMatchChildren = false;
        bool localMatch = matchesLocation(interface, canMatchChildren);

        if (!canMatchChildren)
        {
            interface.stopChildTraversal();
        }

        if (!localMatch)
        {
            return;
        }

        FnAttribute::GroupAttribute setAttrs = interface.getOpArg("setAttrs");
        for (int64_t i = 0, e = setAttrs.getNumberOfChildren(); i < e; ++i)
        {
            FnAttribute::GroupAttribute groupAttr =
                    setAttrs.getChildByIndex(i);

            if (!groupAttr.isValid())
            {
                continue;
            }

            FnAttribute::StringAttribute nameAttr =
                    groupAttr.getChildByName("name");
            FnAttribute::Attribute attr = groupAttr.getChildByName("attr");
            if (!nameAttr.isValid() || !attr.isValid())
            {
                continue;
            }

            bool groupInherit = FnAttribute::IntAttribute(
                    groupAttr.getChildByName("inherit")).getValue(1, false);

            std::string name = nameAttr.getValue("", false);
            if (name.empty())
            {
                continue;
            }

            bool queryGlobal = FnAttribute::IntAttribute(
                    groupAttr.getChildByName("queryGlobal")).getValue(
                            0, false);


            bool attrExists = false;

            if (queryGlobal)
            {
                 attrExists = Foundry::Katana::GetGlobalAttr(
                        interface, name).isValid();
            }
            else
            {
                attrExists = interface.getAttr(name).isValid();
            }


            if (!attrExists)
            {
                interface.setAttr(name, attr, groupInherit);
            }
        }
    }




};

DEFINE_GEOLIBOP_PLUGIN(AttributeSetIfNotPresentOp)

//////////////////////////////////////////////////////////////////////////////

class OverrideDefineOp : public Foundry::Katana::GeolibOp
{
public:
    static void setup(Foundry::Katana::GeolibSetupInterface &interface)
    {
        interface.setThreading(
                Foundry::Katana::GeolibSetupInterface::ThreadModeConcurrent);
    }



    static void cook(Foundry::Katana::GeolibCookInterface &interface)
    {
        bool canMatchChildren = false;
        bool localMatch = matchesLocation(interface, canMatchChildren);

        if (!canMatchChildren)
        {
            interface.stopChildTraversal();
        }

        if (!localMatch)
        {
            return;
        }


        std::string attrPrefix;

        FnAttribute::GroupAttribute typeSpecificPrefixes =
                interface.getOpArg("typePrefixes");
        FnAttribute::StringAttribute typeSpecificPrefixAttr =
                typeSpecificPrefixes.getChildByName(
                        Foundry::Katana::GetInputLocationType(interface));

        //used only for layeredMaterialOverride case
        FnAttribute::GroupAttribute inputMaterial;

        if (typeSpecificPrefixAttr.isValid())
        {
            attrPrefix = typeSpecificPrefixAttr.getValue("", false);

            //special case for layeredMaterialOverride
            if (FnAttribute::StringAttribute(interface.getOpArg(
                    "defaultPrefix")).getValue(
                        "", false) == "layeredMaterialOverride.")
            {
                inputMaterial = FnGeolibServices::FnMaterialResolveUtil::
                        resolveMaterialReferences(
                                Foundry::Katana::GetGlobalAttr(
                                        interface, "material"), false);
            }
        }
        else
        {
            attrPrefix = FnAttribute::StringAttribute(
                interface.getOpArg("defaultPrefix")).getValue("", false);
        }

        FnAttribute::GroupAttribute setAttrs = interface.getOpArg("setAttrs");
        for (int64_t i = 0, e = setAttrs.getNumberOfChildren(); i < e; ++i)
        {
            FnAttribute::GroupAttribute groupAttr =
                    setAttrs.getChildByIndex(i);

            if (!groupAttr.isValid())
            {
                continue;
            }

            FnAttribute::StringAttribute nameAttr =
                    groupAttr.getChildByName("name");
            FnAttribute::Attribute attr = groupAttr.getChildByName("attr");
            if (!nameAttr.isValid() || !attr.isValid())
            {
                continue;
            }

            bool groupInherit = FnAttribute::IntAttribute(
                    groupAttr.getChildByName("inherit")).getValue(1, false);

            std::string name = nameAttr.getValue("", false);
            if (name.empty())
            {
                continue;
            }

            if (inputMaterial.isValid())
            {
                //will only be reached in the layeredMaterialOverride case
                FnAttribute::GroupBuilder gb;
                gb.set(attrPrefix + name, attr);

                setLeafAttrs(interface,
                        FnGeolibServices::FnMaterialResolveUtil::
                                combineLayeredMaterialOverrides(
                                        inputMaterial,
                                        FnAttribute::GroupAttribute(true),
                                        gb.build()),
                                                groupInherit);

            }
            else
            {
                interface.setAttr(attrPrefix + name, attr, groupInherit);
            }
        }
    }

    static void setLeafAttrs(Foundry::Katana::GeolibCookInterface &interface,
            FnAttribute::GroupAttribute groupAttr, bool groupInherit,
                    const std::string & attrPath=std::string())
    {
        int64_t numberOfChildren = groupAttr.getNumberOfChildren();

        for (int64_t i = 0; i < numberOfChildren; ++i)
        {
            std::string childName = groupAttr.getChildName(i);

            std::string childPath;
            if (attrPath.empty())
            {
                childPath = childName;
            }
            else
            {
                childPath = attrPath + "." + childName;
            }

            FnAttribute::Attribute attr = groupAttr.getChildByIndex(i);

            FnAttribute::GroupAttribute childGroup(attr);
            if (childGroup.isValid())
            {
                setLeafAttrs(interface, childGroup, groupInherit, childPath);
            }
            else
            {
                interface.setAttr(childPath, attr);
            }
        }
    }

};

DEFINE_GEOLIBOP_PLUGIN(OverrideDefineOp)


} // anonymous

void registerPlugins()
{
    REGISTER_PLUGIN(AttributeSetOp, "AttributeSet", 0, 1);
    REGISTER_PLUGIN(AttributeSetIfNotPresentOp, "AttributeSetIfNotPresent",
            0, 1);
    REGISTER_PLUGIN(OverrideDefineOp, "OverrideDefine", 0, 1);
}
