#include "ArrayPropUtils.h"

#include <boost/container/small_vector.hpp>
#include <boost/optional.hpp>

#include <FnAttribute/FnDataBuilder.h>

namespace AlembicIn
{
// Converts the specified Alembic property array to an FnAttribute array type.
template <typename attrT, typename podT>
void convertAlembicTypesToKatanaTypes(
    ArrayProp& iProp,
    Alembic::Abc::ISampleSelector sampleSelector,
    typename attrT::value_type* sampleBuffer,
    size_t numPodValues)
{
    try
    {
        // Perform a conversion from the stored POD type to the proper
        // POD type for the Katana attribute.
        Alembic::Util::PlainOldDataType alembicType =
            FnAttrTypeToPODType(attrT::getKatAttributeType());
        iProp.prop.getAs(sampleBuffer, alembicType, sampleSelector);
    }
    catch (Alembic::Abc::Exception&)
    {
        // Because certain types cannot be automatically converted by
        // certain backends (HDF5 does not do well converting float16 to
        // float32, for example), catch the exception and perform a direct
        // cast below.  We don't do this for everything, because the
        // above is faster and works for the vast majority of cases.

        Alembic::Abc::ArraySamplePtr valuePtr;
        iProp.prop.get(valuePtr, sampleSelector);
        if (valuePtr && valuePtr->valid())
        {
            const podT* typedData =
                static_cast<const podT*>(valuePtr->getData());
            typename attrT::value_type* buffer =
                (typename attrT::value_type*)sampleBuffer;
            for (size_t i = 0; i < numPodValues; ++i)
            {
                buffer[i] =
                    static_cast<typename attrT::value_type>(typedData[i]);
            }
        }
    }
}

////////////////////////////////////////////////////////////////////////////////
/// For arithmetic types that provide the zero copy builder.
////////////////////////////////////////////////////////////////////////////////
namespace
{
template <typename attrT>
void CleanUpAttrData(void* context)
{
    typename attrT::value_type* valueBuffer =
            static_cast<typename attrT::value_type*>(context);
    delete[] valueBuffer;
}
}   // namespace

template <typename attrT, typename builderT, typename podT>
void createAndStoreAttr(ArrayProp& iProp,
                        const OpArgs& iArgs,
                        SampleTimes& sampleTimes,
                        int64_t tupleSize,
                        const size_t bufOffset,
                        const size_t numValues,
                        const size_t numPodValues,
                        FnAttribute::GroupBuilder& oGb)
{
    // Number of sample times is typically small so allocate on stack.
    boost::container::small_vector<float, 16> katanaSampleTimes;
    boost::container::small_vector<const typename attrT::value_type*, 16>
        sampleDataPtrs;

    std::unique_ptr<typename attrT::value_type[]> valueBufferPtr(
        new typename attrT::value_type[numPodValues * sampleTimes.size()]);
    typename attrT::value_type* valuePtr = &valueBufferPtr[0];

    // Load the time samples.
    boost::optional<double> previousSampleTime;
    for (auto sampleTime : sampleTimes)
    {
        Alembic::Abc::ISampleSelector sampleSelector(sampleTime);

        valuePtr[0] = typename attrT::value_type();

        convertAlembicTypesToKatanaTypes<attrT, podT>(
            iProp, sampleSelector, valuePtr + bufOffset, numPodValues);

        if (iProp.name == "geometry.poly.startIndex" && numValues != 0)
        {
            for (size_t i = 2; i < numPodValues; ++i)
            {
                valuePtr[i] += valuePtr[i - 1];
            }
        }

        // Reject the sample if it is <= the previous one as float,
        // frame-relative time (which would otherwise invalidate the attribute).
        if (previousSampleTime &&
            static_cast<float>(previousSampleTime.get()) ==
            static_cast<float>(sampleTime))
        {
            continue;
        }

        katanaSampleTimes.push_back(iArgs.getRelativeSampleTime(sampleTime));
        sampleDataPtrs.push_back(valuePtr);

        previousSampleTime = sampleTime;

        // Move to next location in the valueBuffer
        valuePtr = &(valueBufferPtr[numPodValues * katanaSampleTimes.size()]);
    }

    oGb.set(iProp.name,
            attrT(katanaSampleTimes.data(), katanaSampleTimes.size(),
                  sampleDataPtrs.data(), numPodValues, tupleSize,
                  &(valueBufferPtr.release()[0]), CleanUpAttrData<attrT>));
}

////////////////////////////////////////////////////////////////////////////////
/// For non-arithmetic types that don't support the zero copy FnAtttributes
////////////////////////////////////////////////////////////////////////////////
template <>
void createAndStoreAttr<FnAttribute::StringAttribute,
                        FnAttribute::StringBuilder,
                        std::string>(ArrayProp& iProp,
                                     const OpArgs& iArgs,
                                     SampleTimes& sampleTimes,
                                     int64_t tupleSize,
                                     const size_t bufOffset,
                                     const size_t numValues,
                                     const size_t numPodValues,
                                     FnAttribute::GroupBuilder& oGb)
{
    if (sampleTimes.size() == 1)
    {
        Alembic::Abc::ISampleSelector sampleSelector(sampleTimes[0]);
        std::vector<std::string> values;

        values.resize(numPodValues + bufOffset);

        convertAlembicTypesToKatanaTypes<FnAttribute::StringAttribute,
                                         std::string>(
            iProp, sampleSelector, (values.data() + bufOffset), numPodValues);

        // Is this an array index?
        if (iProp.name == "geometry.poly.startIndex" && numValues != 0)
        {
            for (size_t i = 2; i < values.size(); ++i)
            {
                values[i] += values[i - 1];
            }
        }
        oGb.set(iProp.name, FnAttribute::StringAttribute(
                                values.data(), values.size(), tupleSize));
    }
    else
    {
        FnAttribute::StringBuilder b(tupleSize);

        for (auto sampleTime : sampleTimes)
        {
            Alembic::Abc::ISampleSelector sampleSelector(sampleTime);

            std::vector<std::string> values =
                std::vector<std::string>(numPodValues);
            convertAlembicTypesToKatanaTypes<FnAttribute::StringAttribute,
                                             std::string>(
                iProp, sampleSelector, (values.data() + bufOffset),
                numPodValues);

            if (iProp.name == "geometry.poly.startIndex" && numValues != 0)
            {
                for (size_t i = 2; i < numPodValues; ++i)
                {
                    values[i] += values[i - 1];
                }
            }

            b.set(values, iArgs.getRelativeSampleTime(sampleTime));
        }

        oGb.set(iProp.name, b.build());
    }
}

// read iProp.prop into oGb
template <typename attrT, typename builderT, typename podT>
void readArrayProp(
        ArrayProp & iProp,
        const OpArgs & iArgs,
        FnAttribute::GroupBuilder & oGb)
{
    // Extent refers to the number of values this data type is made up of
    const int64_t tupleSize = iProp.prop.getDataType().getExtent();

    // How many sample times are present at this property (and how many are
    // relevant to the scene)
    Alembic::Abc::TimeSamplingPtr ts = iProp.prop.getTimeSampling();
    SampleTimes sampleTimes;
    iArgs.getRelevantSampleTimes(ts, iProp.prop.getNumSamples(), sampleTimes);

    // Determine if all the samples are the same dimension.
    size_t numVals = 0;
    Alembic::Util::Dimensions dims;
    if (!sampleTimes.empty())
    {
        // Use the dimension of the first sample
        SampleTimes::iterator it = sampleTimes.begin();
        iProp.prop.getDimensions(dims, Alembic::Abc::ISampleSelector(*it));
        numVals = dims.numPoints();
        ++it;

        // make sure every sample we are using is the same size
        bool sameSize = true;
        for (; it != sampleTimes.end(); ++it)
        {
            iProp.prop.getDimensions(dims, Alembic::Abc::ISampleSelector(*it));
            if (numVals != dims.numPoints())
            {
                sameSize = false;
                break;
            }
        }

        // not the same, use just a single time
        if (!sameSize)
        {
            sampleTimes.clear();
            sampleTimes.push_back(iArgs.getAbcFrameTime());
            Alembic::Abc::ISampleSelector ss(*sampleTimes.begin());
            iProp.prop.getDimensions(dims, ss);
            numVals = dims.numPoints();
        }
    }

    // How many PoD types do we have?
    const size_t bufOffset =
        (iProp.name == "geometry.poly.startIndex" && numVals != 0) ? 1 : 0;
    const size_t numPodValues = (tupleSize * numVals) + bufOffset;

    createAndStoreAttr<attrT, builderT, podT>(iProp, iArgs, sampleTimes,
                                              tupleSize, bufOffset, numVals,
                                              numPodValues, oGb);
}

////////////////////////////////////////////////////////////////////////////////

void arrayPropertyToAttr(ArrayProp & iProp,
    const OpArgs & iArgs,
    FnAttribute::GroupBuilder & oGb)
{
    switch(iProp.asPod)
    {
        case Alembic::Util::kBooleanPOD:
            readArrayProp<FnAttribute::IntAttribute,
                          FnAttribute::IntBuilder,
                          Alembic::Util::bool_t>(
                iProp, iArgs, oGb);
        break;

        case Alembic::Util::kUint8POD:
            readArrayProp<FnAttribute::IntAttribute,
                          FnAttribute::IntBuilder,
                          Alembic::Util::uint8_t>(
                iProp, iArgs, oGb);
        break;

        case Alembic::Util::kInt8POD:
            readArrayProp<FnAttribute::IntAttribute,
                          FnAttribute::IntBuilder,
                          Alembic::Util::int8_t>(
                iProp, iArgs, oGb);
        break;

        case Alembic::Util::kUint16POD:
            readArrayProp<FnAttribute::IntAttribute,
                          FnAttribute::IntBuilder,
                          Alembic::Util::uint16_t>(
                iProp, iArgs, oGb);
        break;

        case Alembic::Util::kInt16POD:
            readArrayProp<FnAttribute::IntAttribute,
                          FnAttribute::IntBuilder,
                          Alembic::Util::int16_t>(
                iProp, iArgs, oGb);
        break;

        case Alembic::Util::kUint32POD:
            readArrayProp<FnAttribute::IntAttribute,
                          FnAttribute::IntBuilder,
                          Alembic::Util::uint32_t>(
                iProp, iArgs, oGb);
        break;

        case Alembic::Util::kInt32POD:
            readArrayProp<FnAttribute::IntAttribute,
                          FnAttribute::IntBuilder,
                          Alembic::Util::int32_t>(
                iProp, iArgs, oGb);
        break;

        case Alembic::Util::kFloat16POD:
            readArrayProp<FnAttribute::FloatAttribute,
                          FnAttribute::FloatBuilder,
                          Alembic::Util::float16_t>(
                iProp, iArgs, oGb);
        break;

        case Alembic::Util::kFloat32POD:
            readArrayProp<FnAttribute::FloatAttribute,
                          FnAttribute::FloatBuilder,
                          Alembic::Util::float32_t>(
                iProp, iArgs, oGb);
        break;

        case Alembic::Util::kFloat64POD:
            readArrayProp<FnAttribute::DoubleAttribute,
                          FnAttribute::DoubleBuilder,
                          Alembic::Util::float64_t>(
                iProp, iArgs, oGb);
        break;

        case Alembic::Util::kStringPOD:
            readArrayProp<FnAttribute::StringAttribute,
                          FnAttribute::StringBuilder,
                          std::string>(
                iProp, iArgs, oGb);
        break;

        default:
        break;
    }
}

void arrayPropertyToAttr(Alembic::Abc::ICompoundProperty & iParent,
    const Alembic::Abc::PropertyHeader & iPropHeader,
    const std::string & iPropName,
    FnKatAttributeType iType,
    AbcCookPtr ioCook,
    FnAttribute::GroupBuilder & oStaticGb)
{
    Alembic::Abc::IArrayProperty prop(iParent, iPropHeader.getName());

    // bad prop don't bother with it
    if (!prop.valid() || prop.getNumSamples() == 0)
    {
        return;
    }

    ArrayProp item;
    item.name = iPropName;
    item.prop = prop;
    item.asPod = iPropHeader.getDataType().getPod();

    if (!prop.isConstant() && item.asPod != Alembic::Util::kUnknownPOD)
    {
        ioCook->arrayProps.push_back(item);
        return;
    }

    OpArgs defaultArgs;
    arrayPropertyToAttr(item, defaultArgs, oStaticGb);
}

}
