// Copyright (c) 2012 The Foundry Visionmongers Ltd. All Rights Reserved.

#include "FileSequencePlugin.h"
#include <iterator>
#include <string>
#include <utility>
#include <vector>
#include <boost/filesystem.hpp>
#include <boost/format.hpp>
#include <boost/regex.hpp>

#include "FileSequenceInternal.h"
#include <FnAsset/FnDefaultAssetPlugin.h>
#include <FnAsset/plugin/FnAsset.h>
#include <FnAsset/plugin/FnFileSequence.h>
#include <FnAttribute/FnDataBuilder.h>
#include <FnAttribute/FnGroupBuilder.h>
#include <FnLogging/FnLogging.h>
#include "FnAsset/plugin/FnFileSequence.h"
#include "SequenceMatching.h"

std::pair<std::vector<FnKat::FileSequence*>, std::vector<std::string>>
resolveSequences(std::vector<std::string> files)
{
    std::vector<FileNameParts> candidates;
    std::vector<std::string> nonSequenceFiles;
    for (auto& file : files)
    {
        FileNameParts fileNameParts = getFilePartsWithRegex(file);
        if (fileNameParts)
        {
            candidates.emplace_back(std::move(fileNameParts));
        }
        else
        {
            nonSequenceFiles.emplace_back(file);
        }
    }
    if (candidates.empty())
    {
        return {std::vector<FnKat::FileSequence*>(), std::move(files)};
    }
    std::sort(candidates.begin(), candidates.end(), FileNamePartsComparator);

    std::vector<FnKat::FileSequence*> sequences;

    auto it = std::begin(candidates);
    auto end = std::end(candidates);
    auto fs = Sequence(it->prefix, it->number, it->number,
                       it->padding, it->suffix);

    uint32_t sequenceLength(1);
    ++it;

    auto writeToOutputContainers = [&]() {
        if (sequenceLength < 2)
        {
            nonSequenceFiles.emplace_back((it - 1)->filename);
        }
        else
        {
            sequences.emplace_back(new FileSeq(
                fs.prefix, fs.suffix, fs.firstFrame, fs.lastFrame, fs.padding));
        }
    };

    while (it != end)
    {
        if (it->suffix == fs.suffix && it->prefix == fs.prefix &&
            it->number == fs.lastFrame + 1)
        {
            ++fs.lastFrame;
            ++sequenceLength;
            ++it;
        }
        else
        {
            writeToOutputContainers();
            fs = Sequence(it->prefix, it->number, it->number,
                          it->padding, it->suffix);
            sequenceLength = 1;
            ++it;
        }
    }
    if (sequenceLength != 0)
    {
        writeToOutputContainers();
    }

    return {std::move(sequences), std::move(nonSequenceFiles)};
}

std::pair<std::vector<FileNameParts>, std::vector<std::string>>
getFileNamePartsIfValid(std::vector<std::string>& files)
{
    std::vector<FileNameParts> seq;
    std::vector<std::string> nonSeq;
    auto begin = std::begin(files);
    auto end = std::end(files);
    while (begin != end)
    {
        auto fileParts = getFilePartsWithRegex(*begin);
        if (fileParts.isValid)
        {
            seq.push_back(std::move(fileParts));
        }
        else
        {
            nonSeq.emplace_back(*begin);
        }
        begin++;
    }
    return std::make_pair(seq, nonSeq);
}

FileSeq::FileSeq(std::string prefix,
                 std::string suffix,
                 int32_t firstFrame,
                 int32_t lastFrame,
                 size_t padding)
    : _prefix(std::move(prefix)),
      _suffix(std::move(suffix)),
      _firstFrame(firstFrame),
      _lastFrame(lastFrame),
      _padding(padding)
{
    _sequenceStringForm = generateStringForm();
}

FileSeq::FileSeq(const std::string& assetID)
{
    const std::string fileSequenceString = convertAssetIDToFilePath(assetID);
    SequenceMatch match = makeFileSequenceMatch(fileSequenceString);
    if (!match.isValid)
    {
        throw std::invalid_argument(
            "String provided is not a valid file sequence");
    }
    if (match.firstFrame.empty())
    {
        _prefix = match.prefix;
        _suffix = match.suffix;
        _padding = match.getPadding();
        _sequenceStringForm = fileSequenceString;
        _hasFrameSet = false;
    }
    else
    {
        _prefix = match.prefix;
        _suffix = match.suffix;
        _firstFrame = std::stoi(match.firstFrame);
        _lastFrame = std::stoi(match.lastFrame);
        _padding = match.getPadding();
        _sequenceStringForm = fileSequenceString;
    }
}

FnPlugStatus FileSeq::setHost(FnPluginHost* host)
{
    m_host = host;
    FnKat::DefaultAssetPlugin::setHost(host);
    FnKat::FnLogging::setHost(host);
    FnKat::GroupBuilder::setHost(host);
    return FnKat::Attribute::setHost(host);
}

FnPluginHost* FileSeq::getHost()
{
    return m_host;
}

bool FileSeq::isValid() const
{
    return _isValid;
}

FnKat::FileSequence* FileSeq::clone() const
{
    return new FileSeq(*this);
}

void FileSeq::getAsString(std::string& retValue)
{
    retValue = _sequenceStringForm;
}

void FileSeq::getDirectory(std::string& retValue)
{
    boost::filesystem::path p(_prefix);
    boost::filesystem::path dir = p.parent_path();
    retValue = dir.string();
}

void FileSeq::getPrefix(std::string& retValue)
{
    retValue = _prefix;
}
void FileSeq::getSuffix(std::string& retValue)
{
    retValue = _suffix;
}

unsigned int FileSeq::getPadding()
{
    return _padding;
}

void FileSeq::getResolvedPath(const int frame, std::string& retValue)
{
    if (_hasFrameSet && (frame < _firstFrame || _lastFrame < frame))
    {
        throw std::out_of_range(
            "Frame number is not in the range of frames in this frame-set");
    }
    retValue = generateFileNameString(_prefix, _padding, _suffix, frame);
}

bool FileSeq::hasFrameSet()
{
    return _hasFrameSet;
}

bool FileSeq::isFrameInFrameSet(const int frame)
{
    if (_hasFrameSet)
    {
        return _firstFrame <= frame && frame <= _lastFrame;
    }
    return true;
}

int FileSeq::getFirstFrameInFrameSet()
{
    if (_hasFrameSet)
    {
        return _firstFrame;
    }
    return 0;
}

int FileSeq::getLastFrameInFrameSet()
{
    if (_hasFrameSet)
    {
        return _lastFrame;
    }
    return 0;
}

void FileSeq::getNearestFramesInFrameSet(int frame,
                                         bool* hasLeft,
                                         int* nearestLeft,
                                         bool* hasRight,
                                         int* nearestRight)
{
    *hasLeft = false;
    *hasRight = false;
    *nearestLeft = 0;
    *nearestRight = 0;
    if (!_hasFrameSet)
    {
        return;
    }

    if (_firstFrame < frame)
    {
        *nearestLeft = std::min(frame - 1, _lastFrame);
        *hasLeft = true;
    }
    if (frame < _lastFrame)
    {
        *nearestRight = std::max(frame + 1, _firstFrame);
        *hasRight = true;
    }
}

void FileSeq::getFrameListFromFrameSet(std::vector<int>& returnList)
{
    if (!_hasFrameSet)
    {
        returnList = std::vector<int>();
        return;
    }
    int frame = _firstFrame;
    const size_t size(_lastFrame - _firstFrame + 1);
    std::vector<int> range(size);
    std::generate(std::begin(range), std::end(range),
                  [&frame] { return frame++; });
    returnList = std::move(range);
}

FnKat::FileSequence* FileSeq::create(const char* cstr)
{
    const std::string sequenceString(cstr);
    return new FileSeq(sequenceString);
}

bool FileSeq::isFileSequence(const char* assetID)
{
    const std::string assetIDStr = std::string(assetID);
    try
    {
        FileSeq seq = FileSeq(assetIDStr);
        const std::string seqString = seq.convertAssetIDToFilePath(assetIDStr);
        auto match = makeFileSequenceMatch(seqString);
        return match.isValid;
    }
    catch(...)
    {
        return false;
    }
}

std::string generateNoRangeStringForm(const std::string& prefix,
                                      int padding,
                                      const std::string& suffix)
{
    const char* noRangeFormat = "%s%%0%dd%s";
    std::string result =
        str(boost::format(noRangeFormat) % prefix % padding % suffix);
    return result;
}

std::string FileSeq::buildFileSequenceString(const std::string& prefix,
                                             const std::string& suffix,
                                             int padding)
{
    return generateNoRangeStringForm(prefix, padding, suffix);
}

struct Sequences
{
};

FnKat::FileSequenceArray* FileSeq::findSequence(const char** fileList,
                                                unsigned int fileCount)
{
    std::vector<std::string> files(fileList, fileList + fileCount);
    auto sequencesAndFiles = resolveSequences(files);

    return new FnKat::FileSequenceArray(sequencesAndFiles.first,
                                        sequencesAndFiles.second);
}

std::string FileSeq::generateStringForm()
{
    if (_firstFrame != std::numeric_limits<int32_t>::min())
    {
        return generateFilesequenceString(_prefix, _padding, _suffix,
                                          _firstFrame, _lastFrame);
    }
    return generateNoRangeStringForm(_prefix, _padding, _suffix);
}

// Private Methods

std::string FileSeq::convertAssetIDToFilePath(
    const std::string& assetID) const
{
    std::string result = "";
    if (FnKat::DefaultAssetPlugin::isAssetId(assetID))
    {
        result = FnKat::DefaultAssetPlugin::resolveAsset(assetID);
    }
    else
    {
        result = assetID;
    }
    return result;
}

FnPluginHost* FileSeq::m_host = 0x0;

DEFINE_FS_PLUGIN(FileSeq)

void registerPlugins()
{
    REGISTER_PLUGIN(FileSeq, "FileSeq", 0, 1);
}
