#pragma once

#include <cstddef>
#include <iostream>
#include <sstream>
#include <string>

#include <boost/regex.hpp>

struct FilenameMatch
{
    explicit FilenameMatch(bool isValid) : isValid(isValid) {}
    FilenameMatch(bool isValid,
                  const std::string prefix,
                  const std::string number,
                  const std::string suffix)
        : isValid(isValid),
          prefix(std::move(prefix)),
          number(std::move(number)),
          suffix(std::move(suffix))
    {
    }
    bool operator==(const FilenameMatch& rhs) const
    {
        return isValid == rhs.isValid && prefix == rhs.prefix &&
               number == rhs.number && suffix == rhs.suffix;
    }
    bool operator!=(const FilenameMatch& rhs) const { return !(*this == rhs); }
    std::string str() const
    {
        if (!isValid)
        {
            return "Not a match";
        }
        std::stringstream ss;
        ss << prefix << '\n' << number << '\n' << suffix;
        return ss.str();
    }

    bool isValid;
    std::string prefix;
    std::string number;
    std::string suffix;
};

struct SequenceMatch
{
    explicit SequenceMatch(bool isValid) : isValid(isValid) {}
    SequenceMatch(bool isValid,
                  std::string prefix,
                  std::string range,
                  std::string firstFrame,
                  std::string lastFrame,
                  std::string printfSyntaxPadding,
                  std::string format,
                  std::string suffix)
        : isValid(std::move(isValid)),
          prefix(std::move(prefix)),
          range(std::move(range)),
          firstFrame(std::move(firstFrame)),
          lastFrame(std::move(lastFrame)),
          printfSyntaxPadding(std::move(printfSyntaxPadding)),
          format(std::move(format)),
          suffix(std::move(suffix))
    {
    }

    size_t getPadding() const
    {
        size_t result(format.length());
        if (format.empty())
        {
            result = std::stoi(printfSyntaxPadding);
        }
        return result;
    }
    bool operator==(const SequenceMatch& rhs) const
    {
        return isValid == rhs.isValid && prefix == rhs.prefix &&
               range == rhs.range && firstFrame == rhs.firstFrame &&
               lastFrame == rhs.lastFrame &&
               printfSyntaxPadding == rhs.printfSyntaxPadding &&
               format == rhs.format && suffix == rhs.suffix;
    }
    friend std::ostream& operator<<(std::ostream& os,
                                    const SequenceMatch& match);

    bool isValid;
    std::string prefix;
    std::string range;
    std::string firstFrame;
    std::string lastFrame;
    std::string printfSyntaxPadding;
    std::string format;
    std::string suffix;
};

inline std::ostream& operator<<(std::ostream& os, const SequenceMatch& match)
{
    if (!match.isValid)
    {
        os << "Not a sequence";
        return os;
    }
    os << " prefix: " << match.prefix << " range: " << match.range
       << " firstFrame: " << match.firstFrame
       << " lastFrame: " << match.lastFrame
       << " printf Style padding:" << match.printfSyntaxPadding
       << " hash style paddding: " << match.format
       << " suffix: " << match.suffix;
    return os;
}

inline FilenameMatch makeFilenameMatch(const std::string& s)
{
    boost::regex expr{R"((.*[-\._]+)(\d+)(\..*))"};
    boost::smatch matches;
    if (!boost::regex_match(s, matches, expr))
    {
        return FilenameMatch(false);
    }
    return FilenameMatch(true, matches[1], matches[2], matches[3]);
}

inline SequenceMatch makeFileSequenceMatch(const std::string& s)
{
    boost::regex expr{
        R"((.*[-\._]+)(\((\d+)-(\d+)\))?((%(\d+)d)|(#{1,10}))(\..*))"};
    boost::smatch matches;
    if (!boost::regex_match(s, matches, expr))
    {
        return SequenceMatch(false);
    }
    return SequenceMatch(true, matches[1], matches[2], matches[3], matches[4],
                         matches[7], matches[8], matches[9]);
}
