// Copyright (c) 2012 The Foundry Visionmongers Ltd. All Rights Reserved.
#pragma once

#include <cstdint>
#include <limits>
#include <sstream>
#include <string>
#include <boost/format.hpp>
#include "SequenceMatching.h"

inline std::string generateFileNameString(const std::string& prefix,
                                          const uint32_t padding,
                                          const std::string& suffix,
                                          const uint32_t frame)
{
    std::string noRangeSequenceRep =
        boost::str(boost::format("%s%%0%di%s") % prefix % padding % suffix);
    return boost::str(boost::format(noRangeSequenceRep) % frame);
}

inline std::string generateFilesequenceString(const std::string& prefix,
                                              const int32_t padding,
                                              const std::string& suffix,
                                              const int32_t first,
                                              const int32_t last)
{
    std::string sequenceRep =
        boost::str(boost::format("%s(%%0%di-%%0%di)%%%%0%dd%s") % prefix %
                   padding % padding % padding % suffix);
    return boost::str(boost::format(sequenceRep) % first % last);
}

struct FileNameParts
{
    explicit FileNameParts(bool isValid) : isValid(isValid) {}
    FileNameParts(bool isValid,
                  std::string prefix,
                  std::string suffix,
                  size_t padding,
                  int32_t number,
                  std::string filename)
        : isValid(isValid),
          prefix(std::move(prefix)),
          suffix(std::move(suffix)),
          padding(padding),
          number(number),
          filename(std::move(filename))
    {
    }

    explicit operator bool() const { return isValid; }
    friend std::ostream& operator<<(std::ostream& os,
                                    const FileNameParts& fileNameParts);

    bool isValid;
    std::string prefix;
    std::string suffix;
    size_t padding;
    int32_t number;
    std::string filename;
};

inline std::ostream& operator<<(std::ostream& os,
                                const FileNameParts& fileNameParts)
{
    os << "prefix: " << fileNameParts.prefix
       << " suffix: " << fileNameParts.suffix
       << " padding: " << fileNameParts.padding
       << " number: " << fileNameParts.number
       << " filename: " << fileNameParts.filename << '\n';
    return os;
}

inline bool FileNamePartsComparator(const FileNameParts& a,
                                    const FileNameParts& b)
{
    return std::tie(a.prefix, a.suffix, a.padding, a.number) <
           std::tie(b.prefix, b.suffix, b.padding, b.number);
};

inline bool isValidNumber(const std::string& s)
{
    if (s.size() < 1 && 10 < s.size())
        return false;
    if (s.find_first_not_of("0123456789") == std::string::npos)
        return true;
    return false;
}

inline FileNameParts getFilePartsWithRegex(const std::string& fileName)
{
    FilenameMatch match = makeFilenameMatch(fileName);
    if (!match.isValid)
    {
        return FileNameParts(false);
    }
    int32_t number;
    size_t padding;
    const std::string& numString(match.number);
    if (!isValidNumber(numString))
    {
        return FileNameParts(false);
    }
    number = std::stoi(numString);
    padding = numString.length();
    return FileNameParts{true,    match.prefix, match.suffix,
                         padding, number,       fileName};
}

struct Sequence
{
    Sequence(std::string prefix,
             int32_t firstFrame,
             int32_t lastFrame,
             size_t padding,
             std::string suffix)
        : prefix(prefix),
          firstFrame(firstFrame),
          lastFrame(lastFrame),
          padding(padding),
          suffix(suffix)
    {
    }
    std::string prefix;
    int32_t firstFrame;
    int32_t lastFrame;
    size_t padding;
    std::string suffix;
};
