# Copyright (c) 2021 The Foundry Visionmongers Ltd. All Rights Reserved.
"""
Module contains the Nuke Bridge render components
"""
import json
import logging
import os
import socket
import time
import tempfile
import threading

import nuke

# Provide a basic logging config
logging.basicConfig(level=logging.INFO,
                    format='[%(levelname)s Nuke Bridge]: %(message)s')

log = logging.getLogger('NukeBridge')

gKatanaWriterNode = None
gCurrentFrame = 1
gWaitForAOVs = True
gWaitForUpdateRequest = True


def SetCurrentFrame(frame):
    """
    Sets the current frame to render from.

    @type frame: C{int}
    @param frame: The frame to render from.
    """
    # pylint: disable=global-statement
    global gCurrentFrame
    gCurrentFrame = frame
    logging.debug('Live Rendering from frame {frame}'.format(frame=frame))


def SetKatanaWriterNode(writerNodeName):
    """
    Sets the Nuke render node that we should be render from.

    @type writerNodeName:
    @param writerNodeName:
    """
    # pylint: disable=global-statement
    global gKatanaWriterNode
    writerNodeName = str(writerNodeName)
    if gKatanaWriterNode == writerNodeName:
        return
    gKatanaWriterNode = writerNodeName

    global gWaitForUpdateRequest
    gWaitForUpdateRequest = False

    logging.debug('Live Render node set to {nodeName}'.format(
        nodeName=writerNodeName))


def SetAOVsReady():
    """
    Indicate to the render loop that all AOVs have been received from Katana
    for the initial first pass.
    """
    # pylint: disable=global-statement
    global gWaitForAOVs
    gWaitForAOVs = False

    global gWaitForUpdateRequest
    gWaitForUpdateRequest = False

    logging.debug('Pixel data ready.')


def RenderLoop(renderMethod, interactive):
    """
    The main render loop, waits for AOVs to be sent from Katana before rendering
    the initial render pass, if this is a live render then the render loop
    will continue, updating after updates are received.

    @type renderMethod: C{str}
    @type: interactive: C{bool}
    @param renderMethod: C{'previewRender'} or C{'liveRender'}
    @param interactive: Whether the Nuke process has been launched in UI mode.
    @raise RuntimeError: If the global KatanaWriter node (C{gKatanaWriterNode})
        has not been set before calling this function.
    """
    # pylint: disable=global-statement
    global gWaitForAOVs
    global gWaitForUpdateRequest

    if gKatanaWriterNode is None:
        raise RuntimeError('No KatanaWriter render node has been set.')

    if interactive:
        # In interactive mode, the KatanaWriter nodes need to be viewed by an
        # active Viewer node, or else the image data that flows through them
        # won't be sent to Katana.
        # This following logic will make sure that the user is notified (by
        # showing an icon on the KatanaWriter node) when these conditions are
        # not met.

        CheckKatanaWritersStatus()

        def onKatanaWriterCreated():
            CheckKatanaWritersStatus()

        def onInputChanged():
            if nuke.thisKnob().name() == 'inputChange':
                CheckKatanaWritersStatus()

        def onViewerInputNumberChanged():
            if nuke.thisKnob().name() == 'input_number':
                CheckKatanaWritersStatus()

        def onViewerDestroyed():
            CheckKatanaWritersStatus(disregard=nuke.thisNode())

        nuke.addOnCreate(onKatanaWriterCreated, nodeClass='KatanaWriter')
        nuke.addKnobChanged(onInputChanged)
        nuke.addKnobChanged(onViewerInputNumberChanged, nodeClass='Viewer')
        nuke.addOnDestroy(onViewerDestroyed, nodeClass='Viewer')

        return

    # Wait for initial AOV data...
    logging.info('Waiting for pixel data...')
    while gWaitForAOVs:
        time.sleep(0.05)
    logging.info('Initial pixel data ready.')

    # Render initial pass.
    nuke.execute(gKatanaWriterNode, gCurrentFrame, gCurrentFrame, 1)
    gWaitForUpdateRequest = True

    # Enter some kind of loop to get updates and re-trigger a render...
    if renderMethod == 'liveRender':
        while True:
            # Wait for updates from Katana
            while gWaitForUpdateRequest:
                time.sleep(0.25)

            # Got an update so re-render the scene...
            nuke.execute(gKatanaWriterNode, gCurrentFrame, gCurrentFrame, 1)
            gWaitForUpdateRequest = True


def CheckKatanaWritersStatus(disregard=None):
    """
    Checks whether the instances of the KatanaWriter node type are being viewed
    by an active Viewer node.

    @type disregard: C{Viewer}
    @param disregard: An optional Viewer node that will not be considered for
        evaluating the status of the KatanaWriter nodes. Useful when a certain
        node is known to be about to be destroyed.
    """
    def getDownstreamViewers(node):
        """
        Generates a list of Viewer nodes that are connected to the given node.

        @type node: C{Node}
        @rtype: C{generator}
        @param node: The node whose downstream Viewer nodes will be evaluated.
        @return: A generated list of Viewer nodes and most immediate upstream
            node to reach to them.
        """
        pending = [node]
        while pending:
            node = pending.pop(0)
            for dependent in node.dependent():
                if dependent.Class() == 'Viewer':
                    yield dependent, node
                else:
                    pending.append(dependent)

    for node in nuke.allNodes(filter='KatanaWriter', recurseGroups=True):
        for viewer, upstream in getDownstreamViewers(node):
            if viewer == disregard:
                continue

            # The active input port needs to match the upstream node from which
            # this Viewer node was found in the traversal.
            inputNumber = int(viewer.knob('input_number').value())
            if 0 <= inputNumber < viewer.inputs():
                if viewer.input(inputNumber) == upstream:
                    node.clearCustomIcon()
                    break
        else:
            if not hasattr(CheckKatanaWritersStatus, 'iconPath'):
                # An icon from Katana is used.
                scriptPath = os.path.abspath(os.path.realpath(__file__))
                scriptDir = os.path.dirname(scriptPath)
                katanaRoot = os.path.dirname(
                    os.path.dirname(os.path.dirname(
                        os.path.dirname(scriptDir))))
                iconPath = os.path.join(katanaRoot, 'bin', 'python', 'UI4',
                                        'Resources', 'Icons', 'Scenegraph',
                                        'warning32.png')
                CheckKatanaWritersStatus.iconPath = iconPath

            node.setCustomIcon(CheckKatanaWritersStatus.iconPath, 1.0, -32, 8)


class NukeMicroservice(object):
    """
    Class provides a simple TCP interface exposed on the localhost to allow
    the main Nuke service (written in C++) to communicate with the Nuke
    python NDK and avoid python version conflicts.
    """
    def __init__(self):
        """
        Initialises an instance of this class.
        """
        self.__started = False
        self.__thread = threading.Thread(target=self.__eventLoop)
        self.__thread.daemon = True

    def start(self):
        """
        Starts the server.
        """
        self.__thread.start()

        now = time.time()
        while not self.__started:
            time.sleep(0)  # Yield the current thread.
            if time.time() - now > 10:
                log.error('[Nuke Microservice] Unable to start event loop.')
                os._exit(1)

    def startServer(self):
        """
        Starts the NukeMicroservice and writes the local TCP port to a file
        so the Nuke Service (its parent) can find it.
        """
        serverSocket = socket.socket(socket.AF_INET, socket.SOCK_STREAM)
        serverSocket.bind(('localhost', 0))
        port = serverSocket.getsockname()[1]
        serverSocket.listen(1)
        log.debug('[Nuke Microservice] Listening on loopback @ %s.', port)

        # Write the server details to KATANA_TMPDIR so renderboot can find them
        tempDir = os.getenv('KATANA_TMPDIR', tempfile.gettempdir())
        processId = os.getpid()
        nukeServiceFilePath = os.path.join(
            tempDir, 'nuke_service_{}.json'.format(processId))
        log.debug('[Nuke Microservice] Writing temp file to "%s".',
                  nukeServiceFilePath)

        with open(nukeServiceFilePath, 'w') as nukeService:
            # Omit the host as we will only ever listen on localhost for this
            # python based service, this will prevent information leakage across
            # the network unless you have local access (in which case you can
            # just look at the Nuke script.
            nukeService.write(json.dumps({
                'port': port
            }))

        self.__started = True
        while True:
            conn, _ = serverSocket.accept()
            msg = NukeMicroservice.ReceiveMessage(conn)
            if msg:
                self.handleMessage(msg, conn)
            conn.close()


    def handleMessage(self, message, sock):
        """
        Handles new connections from the asyncio TCP server.

        @type: message: C{bytearray}
        @type: sock: C{socket.socket}
        @param: message: The message
        @param sock: TCP socket.
        """
        try:
            # Attempt to parse a JSON object from the message
            request = json.loads(message.decode('latin-1'))
        except Exception as exception:
            log.exception(exception)
            return

        if 'command' not in request:
            NukeMicroservice.SendErrorResponse(
                sock, 'No command given')
            return

        command = request['command']
        if command == 'setFrame':
            args = request['args']
            nodeName = args['katanaWriterNode']
            frame = args['frame']

            # Get the KatanaWriterNode
            writerNode = nuke.toNode(nodeName)
            if not writerNode:
                NukeMicroservice.SendErrorResponse(
                    sock,
                    'No node called {} found.'.format(nodeName),
                    'Nodegraph Error')
                return

            if writerNode.Class() != 'KatanaWriter':
                NukeMicroservice.SendErrorResponse(
                    sock,
                    '{} is not a KatanaWriter node.'.format(nodeName),
                    'Nodegraph Error')
                return

            # Checks pass so set update the knob.
            writerNode['frame'].setValue(frame)

            NukeMicroservice.SendResponse(
                sock, {'katanaWriterNode': nodeName, 'frame': frame})
        elif command == 'getNodes':
            NukeMicroservice.SendResponse(
                sock, {'nodes': self.__getNodeNames()})
        elif command == 'setOutputNodeName':
            args = request['args']
            SetKatanaWriterNode(args['outputNodeName'])
        elif command == 'setAOVsReady':
            # This is synchronisation command sent by Katana once it has sent
            # the initial batch of AOVs
            SetAOVsReady()

            NukeMicroservice.SendResponse(sock, {})
        else:
            NukeMicroservice.SendErrorResponse(
                sock, 'Unknown command {}'.format(command),
                'Unsupported command')
            return

    def __eventLoop(self):
        """
        The NukeService event processing loop.
        """
        self.startServer()

    def __getNodeNames(self):
        """
        :rtype: C{list} of C{str}
        :return: A list of nodes in the current Nuke script.
        """
        return [node.fullName() for node in nuke.root()]


    @staticmethod
    def SendMessage(sock, msg):
        """
        Sends a message through a socket

        @type: C{sock} of C{socket.socket}
        @type: C{msg} of C{str}
        @param sock: The socket to send the response to.
        @param msg: The contents of the message to send
        """
        sock.sendall(msg)

    @staticmethod
    def ReceiveMessage(sock):
        """
        Reads a socket for any pending data. Will continue to read
        data from the stream until a null-terminator is found.

        @type: C{sock} of C{socket.socket}
        @param sock: The socket to send the response to.
        @rtype: C{bytearray}
        @return: The contents of the message from the stream, excluding
            the null-terminator. C{None} if the end of the stream is reached
            before a null-terminator.
        """
        data = bytearray()
        nullTerminatorFound = False
        while not nullTerminatorFound:
            packet = sock.recv(1024)
            if not packet:
                if data:
                    NukeMicroservice.SendErrorResponse(
                        sock, 'Invalid message received (non-terminated)')
                return None
            data.extend(packet)
            nullTerminatorFound = b'\0' in packet
        return data[:-1]

    @staticmethod
    def SendErrorResponse(sock, message, status='error'):
        """
        Formats an error response and sends it to the client via writer then
        closes the stream held by writer.

        @type sock: C{socket.socket}
        @type message: C{str}
        @type status: C{str}
        @param sock: The socket to send the response to.
        @param message: A descriptive message to send to help the client
            debug the error.
        @param status: An error category.
        """
        response = json.dumps({'status': status, 'message': message}).encode(
            'latin-1')
        NukeMicroservice.SendMessage(sock, response)

    @staticmethod
    def SendResponse(sock, response, status='success'):
        """
        Formats a successful response message and sends it to the client via
        writer then closes the stream held by writer.

        @type sock: C{socket.socket}
        @type response: C{dict}
        @type status: C{status}
        @param sock: The socket to send the response to.
        @param response: A C{dict}, convertible to JSON that will be returned to
            the client.
        @param status: A sub-category for the success response.
        """
        jsonResponse = json.dumps(
            {'status': status, 'response': response}).encode('latin-1')
        NukeMicroservice.SendMessage(sock, jsonResponse)


################################################################################


def StartNukeService():
    """
    Starts the NukeService.
    """
    if not hasattr(StartNukeService, 'nukeMicroservice'):
        StartNukeService.nukeMicroservice = NukeMicroservice()
        StartNukeService.nukeMicroservice.start()

        # The creation of the first KatanaWriter node starts the Nuke Service.
        nuke.delete(nuke.createNode('KatanaWriter'))
