"""
NAME: Launch Remote Agents...
ICON: Icons/shelfDefault.png

Simple UI to manage starting agents on a selection of remote machines.
"""


import getpass
import logging
import json
import os
import subprocess
import textwrap

from PyQt5 import (
    QtCore,
    QtGui,
    QtWidgets,
)

from Katana import (
    UI4,
    QT4FormWidgets,
)

from kq import ReadConnectionDetails

log = logging.getLogger('AgentManager')


# Class Definitions -----------------------------------------------------------

class AgentManagerDialog(QtWidgets.QDialog):
    """
    Class implementing a dialog for configuring settings for launching Katana
    Queue agents on remote render farm machines.
    """

    # Initializer -------------------------------------------------------------

    def __init__(self,):
        """
        Initializes an instance of the class.
        """
        QtWidgets.QDialog.__init__(self, UI4.App.MainWindow.GetMainWindow())

        self.setWindowTitle('Launch Remote Agents')
        self.move(QtGui.QCursor.pos())
        self.setMinimumWidth(400)

        connectionDetails = ReadConnectionDetails()
        if not connectionDetails:
            connectionDetails = {
                'host': '',
                'worker_request_port': -1,
                'workerUpdatesPort': -1,
                'broadcastPort': -1,
            }

        pageHints = {'open': True}
        portHints = {'int': True, 'min': 0, 'max': 65535}
        readOnlyHint = {'readOnly': True}

        settingsPolicyData = {
            'KQ Instance': {
                'Host': str(connectionDetails['host']),
                'Agent Request Port':
                    connectionDetails['worker_request_port'],
                'Agent Update Port': connectionDetails['workerUpdatesPort'],
                'Broadcast Port': connectionDetails['broadcastPort'],
                'Data Root': GetDataRoot(),
                '__childHints': {
                    'Agent Request Port': portHints,
                    'Agent Update Port': portHints,
                    'Broadcast Port': portHints,
                    'Data Root': readOnlyHint,
                },
                '__childOrder': [
                    'Host',
                    'Agent Request Port',
                    'Agent Update Port',
                    'Broadcast Port',
                    'Data Root',
                ],
            },
            'SSH': {
                'User Name': getpass.getuser(),
                'Identity File': '~/.ssh/id_rsa',
                '__childOrder': [
                    'User Name',
                    'Identity File',
                ],
            },
            'Agent': {
                'Host': str(connectionDetails['host']),
                'Num Agents': 2,
                'Katana Install': os.getenv('KQ_OVERRIDE_KATANA_ROOT',
                                            os.getenv('KATANA_ROOT')),
                '__childHints': {
                    'Num Agents': {'int': True, 'min': 1, 'max': 256},
                    'Katana Install': readOnlyHint,
                },
                '__childOrder': [
                    'Host',
                    'Num Agents',
                    'Katana Install',
                ],
            },
            '__childHints': {
                'KQ Instance': pageHints,
                'SSH': pageHints,
                'Agent': pageHints,
            },
            '__childOrder': [
                'KQ Instance',
                'SSH',
                'Agent',
            ],
        }

        # Create a Python value policy from the dictionary defined above,
        # configure its top-level widget hints, and create a form widget tree
        # from it
        self.__settingsPolicy = QT4FormWidgets.PythonValuePolicy(
            '', settingsPolicyData)
        self.__settingsPolicy.getWidgetHints()['hideTitle'] = True
        self.__settingsPolicy.getWidgetHints()['open'] = True
        self.__settingsPolicy.getWidgetHints()['widget'] = 'hideTitleGroup'
        self.__settingsWidget = UI4.FormMaster.KatanaWidgetFactory.buildWidget(
            self, self.__settingsPolicy)

        # Create and setup buttons
        copyCommandButton = QtWidgets.QPushButton('Copy Command')
        copyCommandButton.setObjectName('copyCommandButton')
        copyCommandButton.setToolTip('Copies the command for starting agents '
                                     'to the clipboard.')
        stopAgentsButton = QtWidgets.QPushButton('Stop Agents')
        stopAgentsButton.setObjectName('stopAgentsButton')
        startAgentsButton = QtWidgets.QPushButton('Start Agents')
        startAgentsButton.setObjectName('startAgentsButton')

        # Set up button layout
        buttonLayout = QtWidgets.QHBoxLayout()
        buttonLayout.addWidget(copyCommandButton)
        buttonLayout.addWidget(stopAgentsButton)
        buttonLayout.addStretch()
        buttonLayout.addWidget(startAgentsButton)

        # Set up dialog layout
        layout = QtWidgets.QVBoxLayout()
        layout.addWidget(self.__settingsWidget)
        layout.addStretch()
        layout.addLayout(buttonLayout)

        self.setLayout(layout)

        # Set up signal/slot connections based on object names and signal names
        QtCore.QMetaObject.connectSlotsByName(self)

    # QWidget Event Handlers --------------------------------------------------

    def showEvent(self, event):
        """
        Event handler for widget show events.

        @type event: C{QtGui.QShowEvent}
        @param event: An object containing details about the widget show event
            to process.
        """
        # pylint: disable=unused-argument

        # Select all text in the Agent > Host line edit widget, and focus it
        agentWidget = self.__settingsWidget.getFormWidgetChild('Agent')
        hostWidget = agentWidget.getFormWidgetChild('Host')
        lineEditWidget = hostWidget.getControlWidgets()[0]
        lineEditWidget.setFocus()
        QtCore.QTimer.singleShot(0, lineEditWidget.selectAll)

    # Slots -------------------------------------------------------------------

    @QtCore.pyqtSlot()
    def on_copyCommandButton_clicked(self):
        # pylint: disable=invalid-name

        commandArguments = self.__getStartAgentsCommandLine()

        clipboard = QtGui.QGuiApplication.clipboard()
        clipboard.setText(' '.join(commandArguments))

    @QtCore.pyqtSlot()
    def on_stopAgentsButton_clicked(self):
        # pylint: disable=invalid-name

        commandArguments = self.__getStopAgentsCommandLine()

        self.__runProcess(self.sender().text(), commandArguments)

    @QtCore.pyqtSlot()
    def on_startAgentsButton_clicked(self):
        # pylint: disable=invalid-name

        commandArguments = self.__getStartAgentsCommandLine()

        self.__runProcess(self.sender().text(), commandArguments)

    # Private Instance Functions ----------------------------------------------

    def __getStartAgentsCommandLine(self):
        values = []
        for name in [
                'KQ Instance > Host',
                'KQ Instance > Agent Request Port',
                'KQ Instance > Agent Update Port',
                'KQ Instance > Broadcast Port',
                'KQ Instance > Data Root',
                'SSH > User Name',
                'SSH > Identity File',
                'Agent > Host',
                'Agent > Num Agents',
                'Agent > Katana Install']:
            parentName, childName = name.split(' > ')
            parentPolicy = self.__settingsPolicy.getChildByName(parentName)
            childPolicy = parentPolicy.getChildByName(childName)
            if childPolicy.getWidgetHints().get('int', False):
                values.append(int(childPolicy.getValue()))
            else:
                values.append(childPolicy.getValue())

        return GetStartAgentsCommandLine(*values)

    def __getStopAgentsCommandLine(self):
        values = []
        for name in [
                'SSH > User Name',
                'SSH > Identity File',
                'Agent > Host']:
            parentName, childName = name.split(' > ')
            parentPolicy = self.__settingsPolicy.getChildByName(parentName)
            childPolicy = parentPolicy.getChildByName(childName)
            values.append(childPolicy.getValue())

        return GetStopAgentsCommandLine(*values)

    def __runProcess(self, title, commandArguments):
        command = ' '.join(commandArguments)
        log.info('%s: Running command "%s"...', title, command)
        process = subprocess.Popen(commandArguments,
                                   stdout=subprocess.PIPE,
                                   stderr=subprocess.PIPE,
                                   text=True)
        output, errors = process.communicate()
        if process.returncode != 0:
            # Create and configure a message box to show the output of the tool
            messageBox = QtWidgets.QMessageBox(self)
            messageBox.setWindowTitle(title)
            messageBox.setIcon(QtWidgets.QMessageBox.Critical)
            messageBox.setText('Process failed with return code %d.'
                               % process.returncode)
            messageBox.setMinimumWidth(600)
            if output and errors:
                output = output.strip()
                errors = errors.strip()
                messageBox.setInformativeText(errors.splitlines()[0])
                messageBox.setDetailedText('Command:\n%s\n\n'
                                           'Full error output:\n%s\n\n'
                                           'Full standard output:\n%s<'
                                           % (command, errors, output))
            elif errors:
                errors = errors.strip()
                messageBox.setInformativeText(errors.splitlines()[0])
                messageBox.setDetailedText('Command:\n%s\n\n'
                                           'Full error output:\n%s'
                                           % (command, errors))
            elif output:
                output = output.strip()
                messageBox.setInformativeText(output.splitlines()[0])
                messageBox.setDetailedText('Command:\n%s\n\n'
                                           'Full standard output:\n%s'
                                           % (command, output))

            # Make the text edit widget showing the detailed text appear as a
            # console
            consoleStyle = textwrap.dedent('''
                background-color: palette(shadow);
                font-family: mono, Bitstream Vera Sans Mono, Liberation Mono,
                monospace;
                ''').strip()
            messageBox.setStyleSheet('QTextEdit { %s }' % consoleStyle)

            # Modify the label that shows the informative text to appear as a
            # console as well, and give it a minimum width, so that it doesn't
            # get squashed (which it normally would)
            informativeTextLabel = messageBox.findChild(
                QtWidgets.QLabel, 'qt_msgbox_informativelabel')
            if informativeTextLabel is not None:
                informativeTextLabel.setCursor(QtCore.Qt.IBeamCursor)
                informativeTextLabel.setFixedWidth(550)
                informativeTextLabel.setStyleSheet('QLabel { %s }'
                                                   % consoleStyle)
                informativeTextLabel.setTextInteractionFlags(
                    QtCore.Qt.TextSelectableByMouse
                    | QtCore.Qt.TextSelectableByKeyboard)

            # Log the text shown in the message box in order to show it in the
            # View Messages popup, Messages tabs, and in the console
            log.error('%s: %s\n%s', messageBox.text()[:-1],
                      messageBox.informativeText(), messageBox.detailedText())

            messageBox.show()


# Module Functions ------------------------------------------------------------


def GetDataRoot():
    """
    Returns the root where data should be stored by the agents.
    """
    baseDirectory = os.getenv('KQ_DATA_ROOT')
    if baseDirectory:
        return baseDirectory

    baseDirectory = os.getenv('KATANA_TMPDIR', os.getcwd())
    return baseDirectory


def GetStartAgentsCommandLine(*args):
    """
    Returns an array containing the command line to be executed on the remote
    machine.
    """
    (
        # KQ Instance
        kqHost,
        kqAgentRequestPort,
        kqAgentUpdatePort,
        kqBroadcastPort,
        kqDataRootDir,
        # SSH
        user,
        identityFilename,
        # Agent
        host,
        numberOfAgents,
        katanaRootDir,
    ) = args

    commandArguments = ['ssh']

    # Identity
    commandArguments.append('-i')
    commandArguments.append(identityFilename)

    # User and Host
    userAndHost = '{user}@{host}'.format(user=user, host=host)
    commandArguments.append(userAndHost)

    # Command to Run
    if os.name == 'nt':
        commandArguments.append('powershell')
        commandArguments.append('-file')
        commandArguments.append('.\\StartAgents.ps1')
        commandArguments.append('-x')
    else:
        commandArguments.append('$HOME/StartAgents.sh')

    commandArguments.append('"{}"'.format(katanaRootDir))
    commandArguments.append('"{}"'.format(kqDataRootDir))
    commandArguments.append(kqHost)
    commandArguments.append(str(kqAgentRequestPort))
    commandArguments.append(str(kqAgentUpdatePort))
    commandArguments.append(str(kqBroadcastPort))
    commandArguments.append(str(numberOfAgents))

    return commandArguments


def GetStopAgentsCommandLine(*args):
    """
    Returns an array containing the command line to be executed on the remote
    machine.
    """
    (
        # SSH
        user,
        identityFilename,
        # Agent
        host,
    ) = args

    commandArguments = ['ssh']

    # Identity
    commandArguments.append('-i')
    commandArguments.append(identityFilename)

    # User and Host
    userAndHost = '{user}@{host}'.format(user=user, host=host)
    commandArguments.append(userAndHost)

    # Command to Run
    if os.name == 'nt':
        commandArguments.append('powershell')
        commandArguments.append('-file')
        commandArguments.append('.\\StopAgents.ps1')
    else:
        commandArguments.append('$HOME/StopAgents.sh')

    return commandArguments


# Main Code -------------------------------------------------------------------

# Create an instance of the dialog class, and show the dialog
agentManagerDialog = AgentManagerDialog()
agentManagerDialog.show()
