#version 420

in vec4 Color;
in vec2 UV;
in float outlineScale;

layout(location = 0) out vec4 FragColor;

uniform sampler2D screenTexture;

/**
 * The following matrices are a modified Sobel (edge-detection) kernels for
 * horizontal (sx) and vertical (sy) directions respectively.
 **/

// clang-format off
mat3 sx = mat3(
    1.0 * outlineScale,  1.0 * outlineScale,  1.0 * outlineScale,
    0.0,                 0.0,                 0.0,
   -1.0 * outlineScale, -1.0 * outlineScale, -1.0 * outlineScale
);
mat3 sy = mat3(
    1.0 * outlineScale,  0.0,                -1.0 * outlineScale,
    1.0 * outlineScale,  0.0,                -1.0 * outlineScale,
    1.0 * outlineScale,  0.0,                -1.0 * outlineScale
);
// clang-format on

void main()
{
    vec4 idPass = texture(screenTexture, UV);
    vec3 idPassRGB = idPass.rgb;

    /**
     * Fetch the texels from the texture surrouding the particualr UV pair.
     * and store them in I 3x3 matrix.
     **/
    mat3 I;
    for (int i = 0; i < 3; i++)
    {
        for (int j = 0; j < 3; j++)
        {
            vec3 texel =
                texelFetch(screenTexture,
                           ivec2(gl_FragCoord) + ivec2(i - 1, j - 1), 0).rgb;
            I[i][j] = length(texel);
        }
    }

    /**
     * Multiply row by row (vec3 * vec3) each element of both vectors and sum
     * them up (dot product) for both horizontal (gx) and vertical (gy)
     * directions separately. This gives as gradient value in each direction
     **/
    float gx = dot(sx[0], I[0]) + dot(sx[1], I[1]) + dot(sx[2], I[2]);
    float gy = dot(sy[0], I[0]) + dot(sy[1], I[1]) + dot(sy[2], I[2]);

    /**
     * Calculate the overall gradient value using Pythagoras theorem with a
     * magic number 0.3 that makes the edge appear with certain brightness.
     */
    float g = sqrt(pow(gx, 2.0) + pow(gy, 2.0)) * 0.3;
    g = clamp(g, 0.0, 1.0);
    // Multiply the gradient by the input color.
    FragColor = vec4(Color.rgb, g);
}
