#version 420

in vec2 UV;
in float Sigma;

layout(location = 0) out vec4 FragColor;

uniform sampler2D screenTexture;

// Normal distribution probability function to cacluate the blur kernel where
// 1 / sqrt(2 * PI) = 0.39894
// https://en.wikipedia.org/wiki/Normal_distribution
float normpdf(in float x, in float sigma)
{
    return 0.39894 * exp(-0.5 * x * x / (sigma * sigma)) / sigma;
}

void main()
{
    vec4 idPass = texture(screenTexture, UV);
    vec3 idPassRGB = idPass.rgb;

    //
    const int mSize = 3;
    const int kSize = (mSize - 1) / 2;
    float kernel[mSize];
    vec3 final_colour = vec3(0.0);

    float Z = 0.0;
    // We only calculate the values of normal distribution for 9 elements
    // with the last 3 (9 + 3 = mSize) being zero. The trick is that instead
    // of storing all the values in 9x9 or 11x11 matrix we store it only in
    // in a float[11] as the values of the distribution are symetrical in
    // both x and y axis.
    for (int j = 0; j <= kSize; ++j)
    {
        kernel[kSize + j] = kernel[kSize - j] = normpdf(float(j), Sigma);
    }

    for (int j = 0; j < mSize; ++j)
    {
        Z += kernel[j];
    }

    // Read texels in a 11x11 grid and multiply by the values of the normal
    // distribution
    for (int i = -kSize; i <= kSize; ++i)
    {
        for (int j = -kSize; j <= kSize; ++j)
        {
            final_colour +=
                kernel[kSize + j] * kernel[kSize + i] *
                texelFetch(screenTexture,
                           ivec2(gl_FragCoord) + ivec2(i - 1, j - 1), 0).rgb;
        }
    }

    // We normalize result by Z * Z
    vec4 minVal = vec4(0.0, 0.0, 0.0, 0.0);
    vec4 maxVal = vec4(1.0, 1.0, 1.0, 1.0);
    vec4 val = vec4(final_colour / (Z * Z), 1.0);
    FragColor = clamp(val, minVal, maxVal);
}
