// Copyright (c) 2016 The Foundry Visionmongers Ltd. All Rights Reserved.

#ifdef _WIN32
    #include <FnPlatform/Windows.h>
#endif
#include "FnViewer/utils/FnGLShaderProgram.h"
#include "FnViewer/utils/FnDrawingHelpers.h"
#include <iostream>
#include <fstream>
#include <sstream>


namespace Foundry
{
namespace Katana
{
namespace ViewerUtils
{

GLShaderProgram::GLShaderProgram() : m_handle(0), m_isLinked(false)
{

}

GLShaderProgram::~GLShaderProgram()
{
    cleanup();
}

void GLShaderProgram::cleanup()
{
    if (m_handle)
    {
        // Query the number of attached shaders
        GLint numShaders = 0;
        glGetProgramiv(m_handle, GL_ATTACHED_SHADERS, &numShaders);

        // Get the shader names
        std::vector<GLuint> shaderNames(numShaders);
        glGetAttachedShaders(m_handle, numShaders, NULL, shaderNames.data());

        // Delete the shaders
        for (int i = 0; i < numShaders; i++)
        {
            glDeleteShader(shaderNames[i]);
        }

        // Delete the program
        glDeleteProgram(m_handle);

        m_handle = 0;
    }

    m_isLinked = false;
    m_uniformLocations.clear();
}

const std::string GLShaderProgram::readFile(const std::string& filename)
{
    std::ifstream in(filename.c_str());
    if (in.is_open())
    {
        std::ostringstream contents;
        contents << in.rdbuf();
        in.close();
        return(contents.str());
    }

    return "";
}

void GLShaderProgram::compileShader(const std::string& filename, ShaderType type)
{
    const std::string shaderSource = readFile(filename);
    compileShader(filename, type, shaderSource);
}

void GLShaderProgram::compileShader(const std::string& name, ShaderType type, const std::string& shaderSource)
{
    if (m_handle <= 0)
    {
        m_handle = glCreateProgram();
    }

    GLuint shader = glCreateShader(type);
    if (shader == 0)
    {
        std::cerr << "[ GLShaderCompiler ] Error creating GL shader '" << name << "'\n";
        return;
    }

    // Load shader as string
    const GLchar* shaderCode = shaderSource.c_str();

    glShaderSource(shader, 1, &shaderCode, NULL);
    glCompileShader(shader);

    GLint compileResult;
    glGetShaderiv(shader, GL_COMPILE_STATUS, &compileResult);
    if (compileResult == GL_FALSE)
    {
        std::cerr << "[ GLShaderCompiler ] Compilation of shader '" << name << "' failed!\n";
        GLint logLen;
        glGetShaderiv(shader, GL_INFO_LOG_LENGTH, &logLen);

        if (logLen > 0)
        {
            std::vector<char> log(logLen);
            GLsizei written;
            glGetShaderInfoLog(shader, logLen, &written, log.data());
            std::cerr << "[ GLShaderCompiler ] Shader compile log:\n"
                      << log.data() << "\n";
        }

        glDeleteShader(shader);
    }
    else
    {
        glAttachShader(m_handle, shader);
    }
}

GLuint GLShaderProgram::getUniformLocation(const std::string& name)
{
    std::map<std::string, GLuint>::iterator pos = m_uniformLocations.find(name);

    if (pos == m_uniformLocations.end())
    {
        m_uniformLocations[name] = glGetUniformLocation(m_handle, name.c_str());
    }

    return m_uniformLocations[name];
}


void GLShaderProgram::link()
{
    if (m_isLinked && m_handle > 0)
    {
        // Already linked
        return;
    }

    glLinkProgram(m_handle);
    GLint status;
    glGetProgramiv(m_handle, GL_LINK_STATUS, &status);
    if (status == GL_FALSE)
    {
        // Link failed!
        GLint logLength;
        glGetProgramiv(m_handle, GL_INFO_LOG_LENGTH, &logLength);
        if (logLength > 0)
        {
            char* log = new char[logLength];
            GLsizei written;
            glGetProgramInfoLog(m_handle, logLength, &written, log);
            std::cerr << "[ GLShaderCompiler ] Failed to link shader program:\n " << log;
            delete[] log;
        }
        else
        {
            std::cerr << "[ GLShaderCompiler ] Failed to link shader program\n";
        }
    }
    else
    {
        // Link successful!
        m_isLinked = true;
        m_uniformLocations.clear();
    }
}

void GLShaderProgram::use()
{
    if (!m_isLinked || m_handle <= 0)
    {
        std::cerr << "[ GLShaderCompiler ] Cannot use shader program because it is invalid.\n";
        return;
    }

    glUseProgram(m_handle);
}

void GLShaderProgram::bindAttribLocation(GLuint location, const std::string& name)
{
    glBindAttribLocation(m_handle, location, name.c_str());
}

void GLShaderProgram::bindFragDataLocation(GLuint location, const std::string& name)
{
    glBindFragDataLocation(m_handle, location, name.c_str());
}

void GLShaderProgram::setUniform(const std::string& name, float val)
{
    GLint loc = getUniformLocation(name.c_str());
    if(loc == -1)
        return;
    glUniform1f(loc, val);
}

void GLShaderProgram::setUniform(const std::string& name, float x, float y)
{
    GLint loc = getUniformLocation(name.c_str());
    if(loc == -1)
        return;
    glUniform2f(loc, x, y);
}

void GLShaderProgram::setUniform(const std::string& name, float x, float y, float z)
{
    GLint loc = getUniformLocation(name.c_str());
    if (loc == -1)
        return;
    glUniform3f(loc, x, y, z);
}

void GLShaderProgram::setUniform(const std::string& name, float x, float y, float z, float w)
{
    GLint loc = getUniformLocation(name.c_str());
    if(loc == -1)
        return;
    glUniform4f(loc, x, y, z, w);
}

void GLShaderProgram::setUniform(const std::string& name, int val)
{
    GLint loc = getUniformLocation(name.c_str());
    if (loc == -1)
        return;
    glUniform1i(loc, val);
}

void GLShaderProgram::setUniform(const std::string& name, bool val)
{
    setUniform(name, (int)val);
}

void GLShaderProgram::setUniform(const std::string& name, FnAttribute::FloatAttribute val)
{
    GLint loc = getUniformLocation(name.c_str());
    if (loc == -1)
        return;

    FnAttribute::FloatConstVector rawValues = val.getNearestSample(0);
    switch (val.getNumberOfValues())
    {
    case 1: // Single float value
        glUniform1f(loc, rawValues[0]);
        break;
    case 2: // Vec2
        glUniform2f(loc, rawValues[0], rawValues[1]);
        break;
    case 3: // Vec3
        glUniform3f(loc, rawValues[0], rawValues[1], rawValues[2]);
        break;
    case 4: // Vec4
        glUniform4f(
            loc, rawValues[0], rawValues[1], rawValues[2], rawValues[3]);
        break;
    case 9: // 3x3 matrix
        glUniformMatrix3fv(loc, 1, GL_FALSE, &rawValues[0]);
        break;
    case 16: // 4x4 matrix
        glUniformMatrix4fv(loc, 1, GL_FALSE, &rawValues[0]);
        break;
    }
}

void GLShaderProgram::setUniform(const std::string& name, FnAttribute::DoubleAttribute val)
{
    GLint loc = getUniformLocation(name.c_str());
    if (loc == -1)
        return;

    FnAttribute::DoubleConstVector rawValues = val.getNearestSample(0);
    switch (val.getNumberOfValues())
    {
    case 1: // Single value
        glUniform1d(loc, rawValues[0]);
        break;
    case 2: // Vec2
        glUniform2d(loc, rawValues[0], rawValues[1]);
        break;
    case 3: // Vec3
        glUniform3d(loc, rawValues[0], rawValues[1], rawValues[2]);
        break;
    case 4: // Vec4
        glUniform4d(
            loc, rawValues[0], rawValues[1], rawValues[2], rawValues[3]);
        break;
    case 9: // 3x3 matrix
        glUniformMatrix3dv(loc, 1, GL_FALSE, &rawValues[0]);
        break;
    case 16: // 4x4 matrix
        glUniformMatrix4dv(loc, 1, GL_FALSE, &rawValues[0]);
        break;
    }
}

void GLShaderProgram::setUniform(const std::string& name,
    const Matrix44d& matrix)
{
    GLint loc = getUniformLocation(name.c_str());
    if(loc == -1)
        return;

    glUniformMatrix4dv(loc, 1, GL_FALSE, matrix.data);
}

void GLShaderProgram::setUniformMatrix4d(
    const std::string& name, const double* matrix)
{
    GLint loc = getUniformLocation(name.c_str());
    if (loc == -1)
        return;

    glUniformMatrix4dv(loc, 1, GL_FALSE, matrix);
}

}
}
}

