// Copyright (c) 2016 The Foundry Visionmongers Ltd. All Rights Reserved.
#include <FnViewer/utils/FnGLManipulator.h>

#include <cstdlib>

#include <FnViewer/utils/FnDrawingHelpers.h>
#include <FnViewer/utils/FnImathHelpers.h>
#include <FnViewer/plugin/FnViewportCamera.h>
#include <FnAttribute/FnAttribute.h>

namespace Foundry
{
namespace Katana
{
namespace ViewerUtils
{

////////////////////////////
// GLManipulator implementation
////////////////////////////

const std::string GLManipulator::kTechnology = "KatanaGLManipulators";

GLManipulator::GLManipulator() {}

GLManipulator::~GLManipulator() {}

////////////////////////////
// GLManipulatorHandle implementation
////////////////////////////

GLManipulatorHandle::GLManipulatorHandle() :
        m_processingManipulation(false),
        m_latestKeyboardModifiers(FnEventWrapper::kNoModifier),
        m_isDragging(false),
        m_hasBeenDragged(false)
{}

GLManipulatorHandle::~GLManipulatorHandle() {}

void GLManipulatorHandle::cancelManipulation()
{
    // Make sure the viewport view is thawed when the manipulation is aborted.
    getViewport()->setViewFrozen(false);
}

void GLManipulatorHandle::draw() {}

void GLManipulatorHandle::pickerDraw(int64_t pickerID) {}

bool GLManipulatorHandle::event(const FnEventWrapper& eventData)
{
    std::string type = eventData.getType();
    FnAttribute::GroupAttribute dataAttr = eventData.getData();
    bool isFinal = false;

    const bool isClick = type == "MouseButtonPress" ||
        type == "MouseButtonDblClick";
    if (isClick || type == "MouseMove")
    {
        // Check if the left button is pressed
        FnAttribute::IntAttribute leftButtonAttr =
            dataAttr.getChildByName("LeftButton");
        if (!leftButtonAttr.getValue(0, false))
        {
            return false;
        }

        // Check if these set of modifiers can be processed by the manipulator
        // handle implementation. If not, let's ignore the event right away so
        // that other layers can process it.
        const FnAttribute::IntAttribute modifiersAttr(
                dataAttr.getChildByName("modifiers"));
        const int modifiers = modifiersAttr.getValue(0, false);
        if (!canProcessKeyboardModifiers(modifiers))
        {
            // ...unless the manipulation action has started, in which case we
            // will process them.
            if (!m_processingManipulation)
            {
                return false;
            }
        }

        // Keep track of the modifiers.
        m_latestKeyboardModifiers = modifiers;

        // Indicate that we have received a click event. From now on we must
        // accept all events until the manipulation is final.
        if (isClick)
        {
            m_processingManipulation = true;
        }
    }
    else if (type == "MouseButtonRelease")
    {
        isFinal = true;
    }
    else
    {
        // We don't care about other events
        return false;
    }

    // Get mouse position
    FnAttribute::IntAttribute xAttr = dataAttr.getChildByName("x");
    FnAttribute::IntAttribute yAttr = dataAttr.getChildByName("y");
    Vec2i mousePosition(xAttr.getValue(), yAttr.getValue());
    m_latestMousePosition = mousePosition;

    // Get the handle's specific dragging plane information
    Vec3d planeOrigin;
    Vec3d planeNormal;
    if (!getDraggingPlane(planeOrigin, planeNormal))
    {
        // If there is no dragging plane, then there is no dragging.
        return false;
    }

    // Get the camera.
    const ViewportCameraWrapperPtr camera = getViewport()->getActiveCamera();

    // Get the mouse point on the manipulation plane in world space
    Vec3d pointOnPlane;
    const bool validPoint = camera->getPointOnPlane(
        mousePosition.x, mousePosition.y, planeOrigin, planeNormal,
        pointOnPlane);

    // This member holds the state of the dragging, delayed one iteration.
    m_hasBeenDragged = m_isDragging;

    // If we are not dragging yet and this is not the final value, then it we
    // are starging to drag.
    if (!m_isDragging && !isFinal)
    {
        // If it's not a valid intersection, do not start dragging.
        if (!validPoint)
        {
            return true;
        }

        // We are not dragging yet set the inital point
        if (isClick)
        {
            m_initialHandleXform = getXform();
            m_initialPointOnPlane = pointOnPlane;
            m_initialMousePosition = mousePosition;
        }
        else
        {
            // Freeze the viewport view while the manipulation is ongoing.
            getViewport()->setViewFrozen(true);

            // First drag has been detected
            m_isDragging = true;
            m_hasBeenDragged = true;
            startDrag(m_initialPointOnPlane, m_initialMousePosition);
        }
    }
    else if (m_isDragging)
    {
        if (!validPoint)
        {
            // Since we don't have a valid point on plane, we will snap back
            // to the initial point on plane.
            pointOnPlane = m_initialPointOnPlane;
        }

        // Use a manipulation group so that the Live render system gets
        // the entire manipulation as a single event.
        GLManipulator* manip = getGLManipulator();
        std::vector<std::string> locationPaths;
        getGLManipulator()->getLocationPaths(locationPaths);
        for (std::vector<std::string>::const_iterator it =
            locationPaths.begin();
            it != locationPaths.end(); ++it)
        {
            manip->openManipulationGroup(*it);
        }

        drag(m_initialPointOnPlane, m_previousPointOnPlane, pointOnPlane,
            m_initialMousePosition, m_previousMousePosition, mousePosition,
            isFinal);

        // Close the groups now.
        for (std::vector<std::string>::const_iterator it =
            locationPaths.begin();
            it != locationPaths.end(); ++it)
        {
            manip->closeManipulationGroup(*it);
        }
    }

    // End dragging if the mouse was released
    if (isFinal)
    {
        // Mark as not dragging
        m_isDragging = false;

        endDrag();

        // Thaw the viewport view now that we're finished.
        getViewport()->setViewFrozen(false);
    }

    // Update the previous point on plane
    m_previousPointOnPlane = pointOnPlane;
    m_previousMousePosition = mousePosition;

    // Reset the manipulation state when it is final.
    if (isFinal)
    {
        m_processingManipulation = false;
    }

    return true;
}

FnAttribute::Attribute GLManipulatorHandle::getOption(
        Foundry::Katana::ViewerAPI::OptionIdGenerator::value_type optionId)
{
    static const OptionIdGenerator::value_type s_hideMousePointer =
        OptionIdGenerator::GenerateId("HideMousePointer");
    if (s_hideMousePointer == optionId)
    {
        return FnAttribute::IntAttribute(1);
    }

    static const OptionIdGenerator::value_type s_newMousePos =
        OptionIdGenerator::GenerateId("NewMousePosition");
    if (s_newMousePos == optionId)
    {
        // This is to avoid moving the pointer when the user has only clicked
        // on the handler.
        if (!hasBeenDragged())
            return FnAttribute::Attribute();

        // Given the initial position (when the drag started), the initial
        // xform and the final xform, calculate the final position.
        const IMATH_NAMESPACE::M44d xformCompensation =
            toImathMatrix44d(m_initialHandleXform).inverse() *
            toImathMatrix44d(getXform());
        const IMATH_NAMESPACE::V4d finalPosition4d =
            IMATH_NAMESPACE::V4d(toImathV3d(m_initialPointOnPlane)) *
            xformCompensation;
        const Vec3d finalPosition(
            finalPosition4d.x, finalPosition4d.y, finalPosition4d.z);

        // Convert the 3D position into window space.
        ViewportCameraWrapperPtr cam = getViewport()->getActiveCamera();
        const Vec3d windowPos = cam->projectObjectIntoWindow(finalPosition);

        // Build the attribute and ship it.
        const int pos[2] = { static_cast<int>(windowPos.x),
                             static_cast<int>(windowPos.y) };
        return FnAttribute::IntAttribute(pos, 2, 2);
    }

    return FnAttribute::Attribute();
}

GLManipulator* GLManipulatorHandle::getGLManipulator()
{
    return getManipulator()->getPluginInstance<GLManipulator>();
}

ViewportWrapperPtr GLManipulatorHandle::getViewport()
{
    return getManipulator()->getViewport();
}

ViewerDelegateWrapperPtr GLManipulatorHandle::getViewerDelegate()
{
    return getManipulator()->getViewport()->getViewerDelegate();
}

bool GLManipulatorHandle::isDragging()
{
    return m_isDragging;
}

bool GLManipulatorHandle::hasBeenDragged()
{
    return m_hasBeenDragged;
}

bool GLManipulatorHandle::getDraggingPlane(Vec3d& origin, Vec3d& normal)
{
    return false;
}

void GLManipulatorHandle::startDrag(const Vec3d& initialPointOnPlane,
    const Vec2i& initialMousePosition) {}

void GLManipulatorHandle::drag(const Vec3d& initialPointOnPlane,
                               const Vec3d& previousPointOnPlane,
                               const Vec3d& currentPointOnPlane,
                               const Vec2i& initialMousePosition,
                               const Vec2i& previousMousePosition,
                               const Vec2i& currentMousePosition,
                               bool isFinal)
{}

void GLManipulatorHandle::endDrag() {}

void GLManipulatorHandle::useDrawingShader(
    const Matrix44d& xform,
    const Vec4f& color,
    bool isFlat)
{
    const RenderMode renderMode = isFlat ? kShadingFlat : kShading3D;

    // Default shader initialization
    useShader(getViewport(), xform, renderMode, m_shaderProgram);

    // Set the render mode
    m_shaderProgram.setUniform("mode", renderMode);

    // The color of the handle
    m_shaderProgram.setUniform("color", color.x, color.y, color.z, color.w);
}

void GLManipulatorHandle::useSnappingShader(const Matrix44d& xform)
{
    // Default shader initialization
    useShader(getViewport(), xform, kTextured, m_shaderProgram);

    // Set the render mode
    m_shaderProgram.setUniform("mode", kTextured);
}

void GLManipulatorHandle::useLineStippleDrawingShader(
    const Matrix44d& xform,
    const Vec4f& color,
    int pattern,
    float factor)
{
    // Default shader initialization
    useShader(getViewport(), xform, kLineStipple, m_shaderProgramLineStipple);

    // Set the render mode
    m_shaderProgramLineStipple.setUniform("mode", kLineStipple);

    // The color of the handle
    m_shaderProgramLineStipple.setUniform(
        "color", color.x, color.y, color.z, color.w);

    // For the line stipple mode, we need to pass the pattern to the shader.
    m_shaderProgramLineStipple.setUniform("pattern", pattern);
    m_shaderProgramLineStipple.setUniform("factor", factor);
}

void GLManipulatorHandle::usePickingShader(
    const Matrix44d& xform,
    int handleId,
    int handlePriority)
{
    // Default shader initialization
    useShader(getViewport(), xform, kPicking, m_shaderProgram);

    // Set the render mode
    m_shaderProgram.setUniform("mode", kPicking);

    // Handle id and priority
    assert(handleId <= 255);
    assert(handlePriority <= 255);
    m_shaderProgram.setUniform("handleId", handleId);
    m_shaderProgram.setUniform("handlePriority", handlePriority);
}

/*static*/
void GLManipulatorHandle::useShader(
        ViewportWrapperPtr viewport,
        const Matrix44d& xform,
        RenderMode renderMode,
        GLShaderProgram& shaderProgram)
{
    // Compile and link the shader program if it hasn't happened before
    if (!shaderProgram.isLinked())
    {
        const char* const katanaRoot = getenv("KATANA_ROOT");
        std::string fullPath(katanaRoot ? katanaRoot : "");
        fullPath.append("/plugins/Resources/Core/Shaders/"
                        "TransformManipulators/Manipulators");
        shaderProgram.compileShader(fullPath + ".vert", VERTEX);
        if (renderMode == kLineStipple)
        {
            shaderProgram.compileShader(fullPath + ".geom", GEOMETRY);
        }
        shaderProgram.compileShader(fullPath + ".frag", FRAGMENT);
        shaderProgram.link();

        if (!shaderProgram.isLinked())
        {
            return;
        }
    }

    // Get the projection and view matrices
    const double* const viewMatrix = viewport->getViewMatrix();
    const double* const projectionMatrix = viewport->getProjectionMatrix();

    // Enable the shader and set its attributes
    shaderProgram.use();
    shaderProgram.setUniform("worldMatrix", xform);
    shaderProgram.setUniformMatrix4d("viewMatrix", viewMatrix);
    shaderProgram.setUniformMatrix4d("projectionMatrix", projectionMatrix);
    shaderProgram.setUniform(
        "viewportSize",
        static_cast<float>(viewport->getWidth()),
        static_cast<float>(viewport->getHeight()));
}

}
}
}
