// Copyright (c) 2018 The Foundry Visionmongers Ltd. All Rights Reserved.

#include "FnViewer/utils/FnBaseLocator.h"
#include <algorithm>
#include "pystring/pystring.h"

#include <FnViewer/plugin/FnGLStateHelper.h>

namespace Foundry
{
namespace Katana
{
namespace ViewerUtils
{
FnLogSetup("ViewerUtils.BaseLocator");


FnBaseLocatorVDC::LocatorContainerVector
    FnBaseLocatorVDC::m_registeredLocators;

FnBaseLocatorVDC::SceneNode::SceneNode(
    const std::string& locationPath)
    : m_locationPath(locationPath), m_selected(false), m_enabled(true),
      m_hidden(InheritableFlag::kNotSet), m_pickable(InheritableFlag::kNotSet)
{
}

FnBaseLocatorVDC::SceneNode::~SceneNode() {}

void FnBaseLocatorVDC::SceneNode::addChild(
    const std::string& name,
    FnBaseLocatorVDC::SceneNode::Ptr child)
{
    std::vector<std::string> pathTokens;

    pystring::split(pystring::lstrip(name, "/"), pathTokens, "/", 1);

    if (pathTokens.size() > 1)
    {
        if (m_children.count(pathTokens[0]) == 0)
        {
            auto immediateChild = FnBaseLocatorVDC::SceneNode::Ptr(
                new SceneNode(m_locationPath + "/" + pathTokens[0]));
            m_children[pathTokens[0]] = immediateChild;
            immediateChild->setParent(shared_from_this());
        }
        m_children[pathTokens[0]]->addChild(pathTokens[1], child);
    }
    else
    {
        // Ensure that any existing child with the same name
        // is removed
        removeChild(pathTokens[0]);
        m_children[pathTokens[0]] = child;

        child->setParent(shared_from_this());
    }
}

void FnBaseLocatorVDC::SceneNode::removeChild(
    const std::string& name)
{
    std::vector<std::string> pathTokens;

    pystring::split(pystring::lstrip(name, "/"), pathTokens, "/", 1);

    if (pathTokens.size() > 1)
    {
        if (m_children.count(pathTokens[0]) == 0)
        {
            m_children[pathTokens[0]]->removeChild(pathTokens[1]);
        }
    }
    else
    {
        if (m_children.count(pathTokens[0]))
        {
            m_children.erase(pathTokens[0]);
        }
    }
}

FnBaseLocatorVDC::SceneNode::WeakPtr
FnBaseLocatorVDC::SceneNode::getChild(const std::string& name)
{
    std::vector<std::string> pathTokens;

    pystring::split(pystring::lstrip(name, "/"), pathTokens, "/", 1);

    if (!m_children.count(pathTokens[0]))
    {
        return FnBaseLocatorVDC::SceneNode::WeakPtr();
    }
    // Find child matching the first token
    if (pathTokens.size() > 1)
    {
        return m_children[pathTokens[0]]->getChild(pathTokens[1]);
    }
    else
    {
        if (m_children.count(pathTokens[0]))
        {
            return m_children[pathTokens[0]];
        }
    }

    return FnBaseLocatorVDC::SceneNode::WeakPtr();
}

FnBaseLocatorVDC::SceneNode::WeakPtr
FnBaseLocatorVDC::SceneNode::getParent() const
{
    return m_parent;
}

void FnBaseLocatorVDC::SceneNode::setParent(
    FnBaseLocatorVDC::SceneNode::WeakPtr parent)
{
    m_parent = parent;
}

std::map<std::string, FnBaseLocatorVDC::SceneNode::Ptr>
FnBaseLocatorVDC::SceneNode::getChildren() const
{
    return m_children;
}

void FnBaseLocatorVDC::SceneNode::getDescendantLocatorLocations(
    LocatorLocationsMap& locatorLocations)
{
    for (const auto& child : m_children)
    {
        for (const auto& locatorName : child.second->getLocatorNames())
        {
            locatorLocations[locatorName].insert(
                child.second->getLocationPath());
        }
        child.second->getDescendantLocatorLocations(locatorLocations);
    }
}

bool FnBaseLocatorVDC::SceneNode::isSelected() const
{
    if (m_selected)
    {
        return true;
    }
    else if (auto parent = getParent().lock())
    {
        return parent->isSelected();
    }

    return false;
}

bool FnBaseLocatorVDC::SceneNode::isEnabled() const
{
    return m_enabled;
}

/// Returns true if the location is hidden
bool FnBaseLocatorVDC::SceneNode::isHidden() const
{
    if (m_hidden == InheritableFlag::kNotSet)
    {
        if (auto parent = getParent().lock())
        {
            return parent->isHidden();
        }
        else
        {
            return false;
        }
    }

    return m_hidden == InheritableFlag::kTrue;
}

void FnBaseLocatorVDC::SceneNode::setHidden(InheritableFlag hidden)
{
    m_hidden = hidden;
}

/// Returns true if the location is hidden
bool FnBaseLocatorVDC::SceneNode::isPickable() const
{
    if (m_pickable == InheritableFlag::kNotSet)
    {
        if (auto parent = getParent().lock())
        {
            return parent->isPickable();
        }
        else
        {
            return true;
        }
    }

    return m_pickable == InheritableFlag::kTrue;
}

void FnBaseLocatorVDC::SceneNode::setPickable(InheritableFlag pickable)
{
    m_pickable = pickable;
}

std::string FnBaseLocatorVDC::SceneNode::getLocationPath() const
{
    return m_locationPath;
}

void FnBaseLocatorVDC::SceneNode::addLocatorName(
    const std::string& locatorName)
{
    m_locatorNames.insert(locatorName);
}

void FnBaseLocatorVDC::SceneNode::removeLocatorName(
    const std::string& locatorName)
{
    m_locatorNames.erase(locatorName);
}

void FnBaseLocatorVDC::SceneNode::clearLocatorNames()
{
    m_locatorNames.clear();
}

bool FnBaseLocatorVDC::SceneNode::hasLocator() const
{
    return !m_locatorNames.empty();
}

std::set<std::string> FnBaseLocatorVDC::SceneNode::getLocatorNames() const
{
    return m_locatorNames;
}

void FnBaseLocatorVDC::SceneNode::setSelected(bool selected)
{
    m_selected = selected;
}

void FnBaseLocatorVDC::SceneNode::setEnabled(bool enabled)
{
    m_enabled = enabled;
}

//-----------------------------------------------------------------------------
FnBaseLocator::FnBaseLocator() : m_viewerDelegateComponent(nullptr),
    m_viewportLayer(nullptr),
    m_shader(nullptr)
{
}

FnBaseLocator::~FnBaseLocator()
{
}

FnBaseLocatorVDC* FnBaseLocator::getViewerDelegateComponent() const
{
    return m_viewerDelegateComponent;
}

void FnBaseLocator::setViewerDelegateComponent(FnBaseLocatorVDC* vdc)
{
    m_viewerDelegateComponent = vdc;
}

/// Returns the ViewportLayer that owns this object
FnBaseLocatorViewportLayer* FnBaseLocator::getViewportLayer() const
{
    return m_viewportLayer;
}

/// Sets the ViewportLayer that owns this object
void FnBaseLocator::setViewportLayer(FnBaseLocatorViewportLayer* layer)
{
    m_viewportLayer = layer;
}

FnAttribute::Attribute FnBaseLocator::getAttribute(
    const std::string& locationPath,
    const std::string& attrName /* = "" */)
{
    assert(m_viewerDelegateComponent);
    FnAttribute::GroupAttribute attrs =
        m_viewerDelegateComponent->getAttributes(locationPath);

    if (attrName.empty())
    {
        return attrs;
    }

    return attrs.getChildByName(attrName);
}

bool FnBaseLocator::isLocationSelected(const std::string& locationPath)
{
    assert(m_viewerDelegateComponent);
    return m_viewerDelegateComponent->isLocationSelected(locationPath);
}

void FnBaseLocator::onFrameBegin(bool isPicking)
{
    assert(m_shader);

    // Get the camera and prepare the shader
    m_shader->use();
    auto camera = getViewportLayer()->getViewport()->getActiveCamera();
    m_shader->setUniform("ProjectionMatrix", camera->getProjectionMatrix44d());
    m_shader->setUniform("ViewMatrix", camera->getViewMatrix44d());
}

void FnBaseLocator::onFrameEnd(bool isPickingon)
{
}

GLShaderProgram* FnBaseLocator::getDefaultShaderProgram() const
{
    return m_shader;
}

void FnBaseLocator::setDefaultShaderProgram(GLShaderProgram* shaderProgram)
{
    m_shader = shaderProgram;
}

void FnBaseLocator::locationSetup(const std::string& locationPath, bool isPicking)
{
    if (!m_shader)
    {
        return;
    }
    // Set the rest of the shader args
    auto delegate = getViewerDelegateComponent()->getViewerDelegate();
    m_shader->setUniform("WorldMatrix", delegate->getWorldXform(locationPath));

    // The color to be used in the render
    FnKat::ViewerAPI::Vec4f color;
    if (isPicking)
    {
        // Build a location entry attribute:
        // The convention for a location will be a GroupAttribute with
        // a child called "location" and the value must be a
        // StringAttribtue containing the location path in its value
        FnAttribute::GroupAttribute locationAttr =
            FnAttribute::GroupBuilder()
            .set("location",
                FnAttribute::StringAttribute(locationPath))
            .build();

        // Add the pickable object
        FnKat::ViewerAPI::FnPickId pickId =
            getViewportLayer()->addPickableObject(locationAttr);

        // Get the color for this ID (see FnPickingTypes.h)
        pickIdToColor(pickId, color);
        m_shader->setUniform("Color", color.x, color.y, color.z,
            color.w);
    }
    else
    {
        color = FnKat::ViewerAPI::Vec4f(1.0f, 1.0f, 1.0f, 1.0f);
        m_shader->setUniform("Color", color.x, color.y, color.z,
            color.w);
    }
}

void FnBaseLocator::locationCleanup(const std::string& locationPath, bool isPicking)
{

}
//-----------------------------------------------------------------------------

FnBaseLocatorVDC::FnBaseLocatorVDC()
    : m_rootSceneNode(new SceneNode("")),
      m_locatorLocations(new LocatorLocationsMap())
{
}

FnBaseLocatorVDC::~FnBaseLocatorVDC() {}

void FnBaseLocatorVDC::flush() {}

void FnBaseLocatorVDC::registerLocator(
    const std::string& name,
    LocatorCreateCallback create,
    LocatorMatchesCallback matches,
    LocatorMatchesCallback overridesBaseGeometry,
    LocatorGetBoundsCallback getBounds,
    LocatorComputeExtentCallback computeExtent)
{
    LocatorContainer* locatorContainer = new LocatorContainer();
    locatorContainer->name = name;
    locatorContainer->create = create;
    locatorContainer->matches = matches;
    locatorContainer->overridesBaseGeometry = overridesBaseGeometry;
    locatorContainer->getBounds = getBounds;
    locatorContainer->computeExtent = computeExtent;

    m_registeredLocators.push_back(locatorContainer);
}

void FnBaseLocatorVDC::dirtyAllViewports()
{
    for (unsigned int i = 0; i < getViewerDelegate()->getNumberOfViewports();
         ++i)
    {
        FnKat::ViewerAPI::ViewportWrapperPtr vp =
            getViewerDelegate()->getViewport(i);
        vp->setDirty(true);
    }
}

FnBaseLocatorVDC::SceneNode::Ptr
FnBaseLocatorVDC::createSceneNode(
    const FnKat::ViewerAPI::ViewerLocationEvent& event)
{
    return FnBaseLocatorVDC::SceneNode::Ptr(
        new SceneNode(event.locationPath));
}

std::weak_ptr<LocatorLocationsMap>
FnBaseLocatorVDC::getLocatorLocations()
{
    return m_locatorLocations;
}

void FnBaseLocatorVDC::removeLocation(const std::string& locationPath)
{
    SceneNode::WeakPtr sceneNodeWeakPtr =
        m_rootSceneNode->getChild(locationPath);
    SceneNode::Ptr sceneNode = sceneNodeWeakPtr.lock();
    if (sceneNode)
    {
        LocatorLocationsMap locatorLocations;
        for (const auto& locatorName : sceneNode->getLocatorNames())
        {
            locatorLocations[locatorName].insert(locationPath);
        }

        sceneNode->getDescendantLocatorLocations(locatorLocations);

        for (auto& locatorLocationsPair : locatorLocations)
        {
            const std::string& locatorName = locatorLocationsPair.first;
            const std::set<std::string>& locations =
                locatorLocationsPair.second;
            for (const auto& location : locations)
            {
                (*m_locatorLocations.get())[locatorName].erase(location);
            }
        }
    }
    m_rootSceneNode->removeChild(locationPath);
    dirtyAllViewports();
}

bool FnBaseLocatorVDC::locationEvent(
    const FnKat::ViewerAPI::ViewerLocationEvent& event,
    bool locationHandled)
{
    if (event.stateChanges.locationRemoved)
    {
        removeLocation(event.locationPath);
        return false;
    }

    SceneNode::WeakPtr sceneNodeWeakPtr =
        m_rootSceneNode->getChild(event.locationPath);
    SceneNode::Ptr sceneNode = sceneNodeWeakPtr.lock();
    if (!sceneNode)
    {
        sceneNode = createSceneNode(event);
        m_rootSceneNode->addChild(event.locationPath, sceneNode);

        if (m_selectedLocations.count(event.locationPath))
        {
            sceneNode->setSelected(true);
        }
    }

    if (event.stateChanges.attributesUpdated)
    {
        // Update 'hidden' state
        FnAttribute::IntAttribute hideAttr =
            event.attributes.getChildByName("viewer.default.drawOptions.hide");
        if (hideAttr.isValid())
        {
            sceneNode->setHidden(hideAttr.getValue() ?
                                SceneNode::InheritableFlag::kTrue
                                : SceneNode::InheritableFlag::kFalse);
        }
        else
        {
            sceneNode->setHidden(SceneNode::InheritableFlag::kNotSet);
        }

        // Update 'pickable' state
        FnAttribute::IntAttribute pickableAttr =
            event.attributes.getChildByName("viewer.default.pickable");
        if (pickableAttr.isValid())
        {
            sceneNode->setPickable(pickableAttr.getValue() ?
                SceneNode::InheritableFlag::kTrue
                : SceneNode::InheritableFlag::kFalse);
        }
        else
        {
            sceneNode->setPickable(SceneNode::InheritableFlag::kNotSet);
        }
    }

    const bool locationWasEnabled = sceneNode->isEnabled();
    const bool locationIsEnabled = !(event.excluded || locationHandled);
    sceneNode->setEnabled(locationIsEnabled);

    if (!locationIsEnabled)
    {
        if (sceneNode->hasLocator())
        {
            // Dissociate location from all locator plug-ins.
            for (const std::string& locatorName : sceneNode->getLocatorNames())
            {
                std::set<std::string>& locations =
                    (*m_locatorLocations.get())[locatorName];
                locations.erase(event.locationPath);
            }
            sceneNode->clearLocatorNames();
            dirtyAllViewports();
        }

        return false;
    }

    bool overrideBaseGeometry = false;

    for (LocatorContainer* locatorContainer : m_registeredLocators)
    {
        bool locatorMatches = locatorContainer->matches(event);
        std::set<std::string>& locations =
            (*m_locatorLocations.get())[locatorContainer->name];
        if (locatorMatches)
        {
            sceneNode->addLocatorName(locatorContainer->name);
            locations.insert(event.locationPath);

            // Allow the locator to decide if it should override
            // existing handlers
            if (locatorContainer->overridesBaseGeometry(event))
            {
                overrideBaseGeometry = true;
            }
        }
        else
        {
            sceneNode->removeLocatorName(locatorContainer->name);
            locations.erase(event.locationPath);
        }
    }

    if (!locationWasEnabled || event.stateChanges.attributesUpdated)
    {
        dirtyAllViewports();
    }

    return overrideBaseGeometry;
}

void* FnBaseLocatorVDC::getPrivateData(void* inputData)
{
    return nullptr;
}

FnAttribute::DoubleAttribute FnBaseLocatorVDC::getBounds(
    const std::string& location)
{
    FnKat::ViewerAPI::ViewerLocationEvent event;
    event.locationPath = location;
    event.attributes = getAttributes(location);

    for (LocatorContainer* locatorContainer : m_registeredLocators)
    {
        if (locatorContainer->matches(event))
        {
            return locatorContainer->getBounds(event);
        }
    }

    return FnAttribute::DoubleAttribute();
}

/// Computes the extent of the location.
FnAttribute::DoubleAttribute FnBaseLocatorVDC::computeExtent(
    const std::string& location)
{
    FnKat::ViewerAPI::ViewerLocationEvent event;
    event.locationPath = location;
    event.attributes = getAttributes(location);

    for (LocatorContainer* locatorContainer : m_registeredLocators)
    {
        if (locatorContainer->matches(event))
        {
            return locatorContainer->computeExtent(event);
        }
    }

    return FnAttribute::DoubleAttribute();
}

void FnBaseLocatorVDC::locationsSelected(
    const std::vector<std::string>& locations)
{
    // Deselect existing locations
    for (auto location : m_selectedLocations)
    {
        SceneNode::WeakPtr sceneNodeWeakPtr = m_rootSceneNode->getChild(
            location);
        if (SceneNode::Ptr sceneNode = sceneNodeWeakPtr.lock())
        {
            sceneNode->setSelected(false);
        }
    }
    m_selectedLocations.clear();

    // Select new locations
    for (auto location : locations)
    {
        m_selectedLocations.emplace(location);

        SceneNode::WeakPtr sceneNodeWeakPtr = m_rootSceneNode->getChild(
            location);
        if (SceneNode::Ptr sceneNode = sceneNodeWeakPtr.lock())
        {
            sceneNode->setSelected(true);
        }
    }
}

FnBaseLocatorVDC::LocatorContainerVector
FnBaseLocatorVDC::getRegisteredLocators()
{
    return m_registeredLocators;
}

bool FnBaseLocatorVDC::isLocationSelected(
    const std::string& locationPath)
{
    SceneNode::Ptr childPtr = getRootSceneNode()->getChild(
        locationPath).lock();
    if (childPtr)
    {
        return childPtr->isSelected();
    }
    return false;
}

bool FnBaseLocatorVDC::isLocationHidden(
    const std::string& locationPath)
{
    SceneNode::Ptr childPtr = getRootSceneNode()->getChild(locationPath).lock();
    if (childPtr)
    {
        return childPtr->isHidden();
    }
    return false;
}

/***** Viewport Layer *****/

FnBaseLocatorViewportLayer::FnBaseLocatorViewportLayer()
    : ViewportLayer(false /* usePickingOnHover */),
      m_viewerDelegateComponent(nullptr), m_initialized(false)
{
}

FnBaseLocatorViewportLayer::~FnBaseLocatorViewportLayer() {}

void FnBaseLocatorViewportLayer::cleanup()
{
    if (!m_initialized)
    {
        return;
    }

    m_viewport->makeGLContextCurrent();
    for (auto& locatorPair : m_locators)
    {
        FnBaseLocator::Ptr locator = locatorPair.second;
        locator->cleanup();
    }
}

void FnBaseLocatorViewportLayer::setOption(
    FnKat::ViewerAPI::OptionIdGenerator::value_type optionId,
    FnAttribute::Attribute attr)
{
    static const FnKat::ViewerAPI::OptionIdGenerator::value_type s_vdcNameHash =
        FnKat::ViewerAPI::OptionIdGenerator::GenerateId("vdc_name");

    static const auto kExcludedLocationsPickingHash =
        FnKat::ViewerAPI::OptionIdGenerator::GenerateId(
            "ViewportLayer.ExcludedLocationsPicking");

    if (optionId == s_vdcNameHash)
    {
        FnAttribute::StringAttribute vdcNameAttr(attr);
        m_vdcName = vdcNameAttr.getValue("", false);

        if (m_initialized )
        {
            initializeLocators();
        }
    }
    else if (optionId == kExcludedLocationsPickingHash)
    {
        const auto excludedLocationsPicking(
            FnAttribute::StringAttribute(attr).getNearestSample(0.0f));
        m_excludedLocationsPicking.clear();
        m_excludedLocationsPicking.reserve(excludedLocationsPicking.size());
        m_excludedLocationsPicking.insert(m_excludedLocationsPicking.end(),
                                          excludedLocationsPicking.begin(),
                                          excludedLocationsPicking.end());
    }
}

void FnBaseLocatorViewportLayer::initializeLocators()
{
    if (m_vdcName.empty())
    {
        return;
    }

    if (!m_viewerDelegate)
    {
        // Cache the Viewport and ViewerDelegate, since they will not change
        // during the lifetime of this object.
        m_viewport = getViewport();
        m_viewerDelegate = m_viewport->getViewerDelegate();
    }
    // Get the reference to the ViewerDelegateComponent plugin
    FnKat::ViewerAPI::ViewerDelegateComponentWrapperPtr component =
        m_viewerDelegate->getComponent(m_vdcName);
    if (!component)
    {
        FnLogError("Error: FnBaseLocatorViewportLayer could not get component "
                   "named \"" << m_vdcName << "\" from viewer delegate.");
        return;
    }

    // Get the ViewerDelegateComponent plugin instance - this only works if the
    // other plugin is built into the same binary as this.
    m_viewerDelegateComponent =
        component->getPluginInstance<FnBaseLocatorVDC>();

    m_locatorLocations = m_viewerDelegateComponent->getLocatorLocations();

    // Create local instances of locators
    m_viewport->makeGLContextCurrent();
    for (auto& locatorContainer : FnBaseLocatorVDC::getRegisteredLocators())
    {
        if (m_locators.find(locatorContainer->name) == m_locators.end())
        {
            // Create new locator
            FnBaseLocator::Ptr locator = FnBaseLocator::Ptr(
                locatorContainer->create());
            locator->setViewerDelegateComponent(m_viewerDelegateComponent);
            locator->setViewportLayer(this);
            locator->setDefaultShaderProgram(&m_shader);
            locator->setup();
            m_locators[locatorContainer->name] = locator;
        }
    }
}

void FnBaseLocatorViewportLayer::setup()
{
    if (!GLEW_INIT())
    {
        return;
    }


    // Cache the Viewport and ViewerDelegate, since they will not change during
    // the lifetime of this object.
    m_viewport = getViewport();
    m_viewerDelegate = m_viewport->getViewerDelegate();

    // Set up the shader program
    std::string fullPath = getenv("KATANA_ROOT");
    fullPath.append("/plugins/Resources/Core/Shaders/GenericViewerPlugins/");
    fullPath.append("uniform_color");
    m_shader.compileShader(fullPath + ".vert", VERTEX);
    m_shader.compileShader(fullPath + ".frag", FRAGMENT);
    m_shader.link();

    m_initialized = true;
    initializeLocators();
}

void FnBaseLocatorViewportLayer::resize(unsigned int width,
                                        unsigned int height)
{
}

void FnBaseLocatorViewportLayer::draw()
{
    std::set<std::string> ignoreLocations;        // ignore nothing
    drawLocators(false, ignoreLocations);
}

void FnBaseLocatorViewportLayer::pickerDraw(
    unsigned int x,
    unsigned int y,
    unsigned int w,
    unsigned int h,
    const FnKat::ViewerAPI::PickedAttrsMap& ignoreAttrs)
{
    // Get the locations to ignore
    std::set<std::string> ignoreLocations;
    FnKat::ViewerAPI::PickedAttrsMap::const_iterator it;
    for (it = ignoreAttrs.begin(); it != ignoreAttrs.end(); ++it)
    {
        FnAttribute::GroupAttribute child = it->second;
        if (child.isValid())
        {
            // The convention for a location will be a GroupAttribute with
            // a child called "location" and the value must be a
            // StringAttribtue containing the location path in its value
            FnAttribute::StringAttribute locationAttr =
                child.getChildByName("location");
            if (locationAttr.isValid())
            {
                ignoreLocations.insert(locationAttr.getValue());
            }
        }
    }

    for (const auto& location : m_excludedLocationsPicking)
    {
        ignoreLocations.insert(location);
    }

    // Y is flipped in relation to the scissor
    y = getViewport()->getHeight() - y - h;
    FnKat::ViewerUtils::GLStateRestore glStateRestore(
        FnKat::ViewerUtils::Scissor);

    glScissor(x, y, w, h);  // Render only the necessary part
    glEnable(GL_SCISSOR_TEST);

    drawLocators(true, ignoreLocations);
}

void FnBaseLocatorViewportLayer::drawLocators(
    bool isPicking,
    std::set<std::string>& ignoreLocations)
{
    if (m_locators.empty())
    {
        return;
    }

    std::shared_ptr<LocatorLocationsMap> locatorLocations =
        m_locatorLocations.lock();
    if (!locatorLocations)
    {
        return;
    }

    auto camera = getViewport()->getActiveCamera();
    if (camera->hasLocationPath())
    {
        // Don't draw locators at looked-through locations
        ignoreLocations.insert(camera->getLocationPath());
    }

    FnKat::ViewerUtils::GLStateRestore glStateRestore(
        FnKat::ViewerUtils::Polygon | FnKat::ViewerUtils::Lighting |
        FnKat::ViewerUtils::Line | FnKat::ViewerUtils::Transform);

    glDisable(GL_CULL_FACE);

    // Example of accessing the camera's view matrix as an Imath data type:
    // IMATH_NAMESPACE::M44d viewMatrix =
    // toImathMatrix44d(camera->getViewMatrix44d());

    // Iterate over the locators
    for (auto& locatorPair : m_locators)
    {

        const std::string& name = locatorPair.first;
        FnBaseLocator::Ptr locator = locatorPair.second;

        std::set<std::string> locations = (*locatorLocations.get())[name];
        if (locations.empty())
        {
            continue;
        }

        locator->onFrameBegin(isPicking);
        for (std::string locationPath : locations)
        {
            // Ignore location if in ignore set
            if (ignoreLocations.find(locationPath) != ignoreLocations.end())
            {
                continue;
            }
            auto rootNode = m_viewerDelegateComponent->getRootSceneNode();
            auto locationNode = rootNode->getChild(locationPath).lock();
            if (!locationNode || !locationNode->isEnabled()
                || locationNode->isHidden())
            {
                continue;
            }

            if (isPicking && !locationNode->isPickable())
            {
                continue;
            }

            locator->locationSetup(locationPath, isPicking);
            // If this is a pickerDraw() call
            if (isPicking)
            {
                locator->pickerDraw(locationPath);
            }
            else
            {
                locator->draw(locationPath);
            }
            locator->locationCleanup(locationPath, isPicking);
        }
        locator->onFrameEnd(isPicking);
    }
}
}  // namespace ViewerUtils
}  // namespace Katana
}  // namespace Foundry
