// Copyright (c) 2016 The Foundry Visionmongers Ltd. All Rights Reserved.

#include <FnViewer/plugin/FnViewportLayer.h>

#include <FnPluginManager/FnPluginManager.h>
#include <FnAttribute/FnAttribute.h>
#include <FnAttribute/FnGroupBuilder.h>
#include <FnAttribute/FnAttribute.h>
#include <FnViewer/plugin/FnOptionIdGenerator.h>

#include <FnLogging/FnLogging.h>

#include <iostream>

////////////////////////////////////////////////////
// C callbacks implementations for the plugin suite
////////////////////////////////////////////////////
void ViewportLayer_destroy(FnViewportLayerPluginHandle handle)
{
    delete handle;
}

void ViewportLayer_setup(FnViewportLayerPluginHandle handle)
{
    handle->getViewportLayer()->setup();
}

void ViewportLayer_cleanup(FnViewportLayerPluginHandle handle)
{
    handle->getViewportLayer()->cleanup();
}

int ViewportLayer_event(FnViewportLayerPluginHandle handle,
                        FnAttributeHandle eventAttrHandle)
{
    // FIXME(DL): Non-standard ref-counting policy. TP 373492
    FnAttribute::GroupAttribute eventAttr =
        FnAttribute::Attribute::CreateAndSteal(eventAttrHandle);
    if (!eventAttr.isValid())
    {
        return 0;
    }

    Foundry::Katana::ViewerAPI::FnEventWrapper eventWrapper(eventAttr);
    return handle->getViewportLayer()->event(eventWrapper)? 1 : 0;
}

void ViewportLayer_draw(FnViewportLayerPluginHandle handle)
{
    handle->getViewportLayer()->draw();
}

void ViewportLayer_resize(FnViewportLayerPluginHandle handle,
    unsigned int width, unsigned int height)
{
    handle->getViewportLayer()->resize(width, height);
}

void ViewportLayer_hover(FnViewportLayerPluginHandle handle, uint8_t isHovering,
    int x, int y)
{
    handle->getViewportLayer()->hover(isHovering, x, y);
}

void ViewportLayer_freeze(FnViewportLayerPluginHandle handle)
{
    handle->getViewportLayer()->freeze();
}

void ViewportLayer_thaw(FnViewportLayerPluginHandle handle)
{
    handle->getViewportLayer()->thaw();
}

void ViewportLayer_setOption(FnViewportLayerPluginHandle handle,
    uint64_t optionId, FnAttributeHandle attrHandle)
{
    // FIXME(DL): Non-standard ref-counting policy. TP 373492
    FnAttribute::Attribute attr =
        FnAttribute::Attribute::CreateAndSteal(attrHandle);

    handle->getViewportLayer()->setOption(optionId, attr);
}

FnAttributeHandle ViewportLayer_getOption(FnViewportLayerPluginHandle handle,
    uint64_t optionId)
{
    return handle->getViewportLayer()->getOption(optionId).getRetainedHandle();
}

void ViewportLayer_pickerDraw(FnViewportLayerPluginHandle handle,
    unsigned int x, unsigned int y,
    unsigned int w, unsigned int h,
    FnAttributeHandle ignoreAttrHandle)
{
    Foundry::Katana::ViewerAPI::PickedAttrsMap ignoreAttrs;
    ignoreAttrs.fromGroupAttribute(
        FnAttribute::Attribute::CreateAndRetain(ignoreAttrHandle));

    handle->getViewportLayer()->pickerDraw(x, y, w, h, ignoreAttrs);
}

int ViewportLayer_customPick(FnViewportLayerPluginHandle handle,
    unsigned int x, unsigned int y,
    unsigned int w, unsigned int h,
    int deepPicking,
    FnAttributeHandle& pickedAttrHandle,
    float* singlePointDepth)
{
    Foundry::Katana::ViewerAPI::PickedAttrsMap pickedAttrs;
    if (!handle->getViewportLayer()->customPick(x, y, w, h, deepPicking,
                                                pickedAttrs,
                                                singlePointDepth))
    {
        return false;
    }

    // Transform the map into a GroupAttribute
    pickedAttrHandle = pickedAttrs.toGroupAttribute().getRetainedHandle();

    return true;
}

int ViewportLayer_usesPickingOnHover(FnViewportLayerPluginHandle handle)
{
    return handle->getViewportLayer()->usesPickingOnHover();
}

///////////////////////////
// FnViewportLayer implementation
///////////////////////////
namespace Foundry
{
namespace Katana
{
namespace ViewerAPI
{

///////////////////////////
// ViewportLayerPluginBase implementation
///////////////////////////

ViewportLayerPluginBase::ViewportLayerPluginBase()
{}

ViewportLayerPluginBase::~ViewportLayerPluginBase()
{}


ViewportWrapperPtr ViewportLayerPluginBase::getViewport()
{
    if (!m_viewportWrapper)
    {
        FnViewportHostHandle viewportHostHandle;
        FnViewportPluginHandle viewportPluginHandle;
        FnViewportPluginSuite_v2* viewportPluginSuite;

        m_hostSuite->getViewport(m_hostHandle,
            &viewportHostHandle, &viewportPluginHandle, &viewportPluginSuite);

        ViewportWrapperPtr ptr(
            new ViewportWrapper(
                getHost(),
                viewportHostHandle,
                viewportPluginHandle,
                viewportPluginSuite));

        m_viewportWrapper = ptr;
    }

    return m_viewportWrapper;
}

void ViewportLayerPluginBase::hover(bool isHovering, int x, int y)
{
    m_hostSuite->hover(m_hostHandle, isHovering, x, y);
}

void ViewportLayerPluginBase::pick(
    unsigned int x, unsigned int y,
    unsigned int w, unsigned int h,
    bool deepPicking,
    PickedAttrsMap& pickedAttrs,
    float* singlePointDepth)
{
    FnAttribute::GroupAttribute attr = FnAttribute::Attribute::CreateAndSteal(
        m_hostSuite->pick(m_hostHandle, x, y, w, h, deepPicking,
                          singlePointDepth));

    // Parse the GroupAttr contents and add it to the map
    pickedAttrs.fromGroupAttribute(attr);
}

FnPlugStatus ViewportLayerPluginBase::setHost(FnPluginHost* host)
{
    m_host = host;

    FnPlugStatus status = FnPluginStatusOK;

    status = FnPluginManager::PluginManager::setHost(host);
    if (status != FnPluginStatusOK)
        return status;

    // We don't verity the return value of OptionIdGenerator::setHost() as the
    // host suite is not guaranteed to be registered yet.
    OptionIdGenerator::setHost(host);

    status = FnLogging::setHost(host);
    if (status != FnPluginStatusOK)
        return status;

    status = FnAttribute::GroupBuilder::setHost(host);
    if (status != FnPluginStatusOK)
        return status;

    status = FnAttribute::Attribute::setHost(host);
    return status;
}

FnPluginHost* ViewportLayerPluginBase::getHost() { return m_host; }

FnPluginHost *ViewportLayerPluginBase::m_host = 0x0;

///////////////////////////
// Viewport implementation
///////////////////////////

ViewportLayer::ViewportLayer(bool usePickingOnHover)
    : m_usesPickingOnHover(usePickingOnHover)
{}

ViewportLayer::~ViewportLayer()
{}

FnAttribute::Attribute ViewportLayer::getOption(
    OptionIdGenerator::value_type optionId)
{
    return FnAttribute::Attribute();
}

void ViewportLayer::setOption(const std::string& name, FnAttribute::Attribute attr)
{
    setOption(OptionIdGenerator::GenerateId(name.c_str()), attr);
}

FnAttribute::Attribute ViewportLayer::getOption(const std::string& name)
{
    return getOption(OptionIdGenerator::GenerateId(name.c_str()));
}

FnPickId ViewportLayer::addPickableObject(FnAttribute::Attribute attr)
{
    return m_hostSuite->addPickableObject(m_hostHandle, attr.getHandle());
}

bool ViewportLayer::customPick(
    unsigned int x, unsigned int y,
    unsigned int w, unsigned int h,
    bool deepPicking,
    PickedAttrsMap& pickedAttrs,
    float* singlePointDepth)
{
    return false;
}

FnViewportLayerPluginSuite_v2 ViewportLayer::createSuite(
    FnViewportLayerPluginHandle (*create)(
        FnViewportLayerHostHandle hostHandle))
{
    FnViewportLayerPluginSuite_v2 suite = {
        create,
        ::ViewportLayer_destroy,
        ::ViewportLayer_setup,
        ::ViewportLayer_cleanup,
        ::ViewportLayer_event,
        ::ViewportLayer_draw,
        ::ViewportLayer_resize,
        ::ViewportLayer_hover,
        ::ViewportLayer_freeze,
        ::ViewportLayer_thaw,
        ::ViewportLayer_setOption,
        ::ViewportLayer_getOption,
        ::ViewportLayer_pickerDraw,
        ::ViewportLayer_usesPickingOnHover,
        ::ViewportLayer_customPick
    };

    return suite;
}

FnViewportLayerPluginHandle ViewportLayer::newViewportLayerHandle(
    ViewportLayer* viewportLayer)
{
    if (!viewportLayer)
    {
        return 0x0;
    }

    FnViewportLayerPluginStruct* pluginHandle =
        new FnViewportLayerPluginStruct(viewportLayer);

    return (FnViewportLayerPluginHandle) pluginHandle;
}

void ViewportLayer::setHostHandle(FnViewportLayerHostHandle hostHandle)
{
    m_hostHandle = hostHandle;
    m_hostSuite = const_cast<FnViewportLayerHostSuite_v2 *>(
        reinterpret_cast<const FnViewportLayerHostSuite_v2 *>(
            m_host->getSuite("ViewportLayerHost", 2)));
}

FnViewportLayerHostHandle ViewportLayer::getHostHandle()
{
    return m_hostHandle;
}

unsigned int ViewportLayer::_apiVersion = 2;
const char* ViewportLayer::_apiName     = "ViewportLayerPlugin";


///////////////////////////
// ViewportWrapper implementation
///////////////////////////

ViewportLayerWrapper::ViewportLayerWrapper(
    FnPluginHost* host,
    FnViewportLayerHostHandle hostHandle,
    FnViewportLayerPluginHandle pluginHandle,
    FnViewportLayerPluginSuite_v2* pluginSuite)
{
    if (!getHost()) { setHost(host); }

    m_hostHandle   = hostHandle;
    m_pluginHandle = pluginHandle;
    m_pluginSuite  = pluginSuite;
    m_hostSuite    = const_cast<FnViewportLayerHostSuite_v2 *>(
        reinterpret_cast<const FnViewportLayerHostSuite_v2 *>(
            m_host->getSuite("ViewportLayerHost", 2)));
}

ViewportLayerWrapper::~ViewportLayerWrapper()
{}

ViewportLayer* ViewportLayerWrapper::getPluginPointer()
{
    return m_pluginHandle->getViewportLayer();
}

void ViewportLayerWrapper::draw()
{
    m_hostSuite->draw(m_hostHandle);
}

bool ViewportLayerWrapper::event(FnEventWrapper eventData)
{
    // FIXME(DL): Non-standard ref-counting policy. TP 373492
    return m_hostSuite->event( m_hostHandle,
        eventData.getRetainedHandle() ) != 0;
}

void ViewportLayerWrapper::setOption(OptionIdGenerator::value_type optionId,
    FnAttribute::Attribute attr)
{
    // FIXME(DL): Non-standard ref-counting policy. TP 373492
    m_pluginSuite->setOption(m_pluginHandle, optionId,
        attr.getRetainedHandle());
}

FnAttribute::Attribute ViewportLayerWrapper::getOption(
    OptionIdGenerator::value_type optionId)
{
    FnAttributeHandle attrHandle =
        m_pluginSuite->getOption(m_pluginHandle, optionId);
    FnAttribute::Attribute attr =
        FnAttribute::Attribute::CreateAndSteal(attrHandle);
    return attr;
}

void ViewportLayerWrapper::setOption(const std::string& name, FnAttribute::Attribute attr)
{
    setOption(OptionIdGenerator::GenerateId(name.c_str()), attr);
}

FnAttribute::Attribute ViewportLayerWrapper::getOption(const std::string& name)
{
    return getOption(OptionIdGenerator::GenerateId(name.c_str()));
}

}
}
}
