// Copyright (c) 2017 The Foundry Visionmongers Ltd. All Rights Reserved.

#include <FnViewer/plugin/FnViewportCamera.h>

#include <FnPluginManager/FnPluginManager.h>
#include <FnAttribute/FnAttribute.h>
#include <FnAttribute/FnGroupBuilder.h>
#include <FnAttribute/FnAttribute.h>

#include <FnLogging/FnLogging.h>

#include <iostream>

using Foundry::Katana::ViewerAPI::Vec3d;
typedef Foundry::Katana::ViewerAPI::ViewportCameraPluginBase FnViewerCameraBase;

////////////////////////////////////////////////////
// C callbacks implementations for the plugin suite
////////////////////////////////////////////////////
void ViewportCamera_destroy(FnViewportCameraPluginHandle handle)
{
    delete handle;
}



/////////////////////////////////////////////////////////////////////////


int ViewportCamera_getCameraTypeID(FnViewportCameraPluginHandle handle)
{
    return handle->getViewportCamera()->getCameraTypeID();
}

void ViewportCamera_setViewportDimensions(FnViewportCameraPluginHandle handle,
    unsigned int width, unsigned int height)
{
    handle->getViewportCamera()->setViewportDimensions(width, height);
}

void ViewportCamera_setOption(FnViewportCameraPluginHandle handle,
    uint64_t optionId, FnAttributeHandle attrHandle)
{
    // FIXME(DL): Non-standard ref-counting policy. TP 373492
    FnAttribute::Attribute attr =
        FnAttribute::Attribute::CreateAndSteal(attrHandle);

    handle->getViewportCamera()->setOption(optionId, attr);
}

FnAttributeHandle ViewportCamera_getOption(
    FnViewportCameraPluginHandle handle, uint64_t optionId)
{
    return handle->getViewportCamera()->getOption(optionId).getRetainedHandle();
}

void ViewportCamera_translate(FnViewportCameraPluginHandle handle, double x, double y, double z)
{
    handle->getViewportCamera()->translate(x, y, z);
}

void ViewportCamera_rotate(FnViewportCameraPluginHandle handle, double x, double y, double z)
{
    handle->getViewportCamera()->rotate(x, y, z);
}

void ViewportCamera_startInteraction(FnViewportCameraPluginHandle handle)
{
    handle->getViewportCamera()->startInteraction();
}

void ViewportCamera_endInteraction(FnViewportCameraPluginHandle handle)
{
    handle->getViewportCamera()->endInteraction();
}

void ViewportCamera_setup(FnViewportCameraPluginHandle handle, FnAttributeHandle attrHandle)
{
    // FIXME(DL): Non-standard ref-counting policy. TP 373492
    FnAttribute::Attribute attr =
        FnAttribute::Attribute::CreateAndSteal(attrHandle);

    handle->getViewportCamera()->setup(attr);
}

FnAttributeHandle ViewportCamera_asAttribute(FnViewportCameraPluginHandle handle)
{
    return handle->getViewportCamera()->asAttribute().getRetainedHandle();
}


bool ViewportCamera_getPointOnPlane(FnViewportCameraPluginHandle handle, int x, int y,
    const double originX, const double originY, const double originZ,
    const double normalX, const double normalY, const double normalZ,
    double* intersectionX, double* intersectionY, double* intersectionZ)
{
    Vec3d intersection;
    bool hit = handle->getViewportCamera()->getPointOnPlane(x, y,
        Vec3d(originX, originY, originZ),
        Vec3d(normalX, normalY, normalZ),
        intersection);

    *intersectionX = intersection.x;
    *intersectionY = intersection.y;
    *intersectionZ = intersection.z;

    return hit;
}

void ViewportCamera_projectObjectIntoWindow(FnViewportCameraPluginHandle handle,
    const double objectX, const double objectY, const double objectZ,
    double* windowX, double* windowY, double* windowZ)
{
    Vec3d window = handle->getViewportCamera()->projectObjectIntoWindow(
        Vec3d(objectX, objectY, objectZ));
    *windowX = window.x;
    *windowY = window.y;
    *windowZ = window.z;
}

void ViewportCamera_projectWindowIntoObject(FnViewportCameraPluginHandle handle,
    const double windowX, const double windowY, const double windowZ,
    double* objectX, double* objectY, double* objectZ)
{
    Vec3d object = handle->getViewportCamera()->projectWindowIntoObject(
        Vec3d(windowX, windowY, windowZ));
    *objectX = object.x;
    *objectY = object.y;
    *objectZ = object.z;
}

void ViewportCamera_setDirty(FnViewportCameraPluginHandle handle,
    int32_t dirtyBits)
{
    handle->getViewportCamera()->setDirty(static_cast<FnViewerCameraBase::CameraDirtyBits>(dirtyBits));
}

void ViewportCamera_getRay(FnViewportCameraPluginHandle handle,
    int x, int y,
    double* posX, double* posY, double* posZ,
    double* dirX, double* dirY, double* dirZ)
{
    Vec3d pos, dir;
    handle->getViewportCamera()->getRay(x, y, pos, dir);

    *posX = pos.x;
    *posY = pos.y;
    *posZ = pos.z;

    *dirX = dir.x;
    *dirY = dir.y;
    *dirZ = dir.z;
}


///////////////////////////
// FnViewportCamera implementation
///////////////////////////
namespace Foundry
{
namespace Katana
{
namespace ViewerAPI
{

///////////////////////////
// ViewportCameraPluginBase implementation
///////////////////////////

ViewportCameraPluginBase::ViewportCameraPluginBase()
{}

ViewportCameraPluginBase::~ViewportCameraPluginBase()
{}

FnPlugStatus ViewportCameraPluginBase::setHost(FnPluginHost* host)
{
    m_host = host;

    FnPlugStatus status = FnPluginStatusOK;

    status = FnPluginManager::PluginManager::setHost(host);
    if (status != FnPluginStatusOK)
        return status;

    // We don't verity the return value of OptionIdGenerator::setHost() as the
    // host suite is not guaranteed to be registered yet.
    OptionIdGenerator::setHost(host);

    status = FnLogging::setHost(host);
    if (status != FnPluginStatusOK)
        return status;

    status = FnAttribute::GroupBuilder::setHost(host);
    if (status != FnPluginStatusOK)
        return status;

    status = FnAttribute::Attribute::setHost(host);
    return status;
}

ViewportWrapperPtr ViewportCameraPluginBase::getViewport()
{
    if (!m_viewportWrapper)
    {
        FnViewportHostHandle viewportHostHandle;
        FnViewportPluginHandle viewportPluginHandle;
        FnViewportPluginSuite_v2* viewportPluginSuite;

        m_hostSuite->getViewport(m_hostHandle,
            &viewportHostHandle, &viewportPluginHandle, &viewportPluginSuite);

        ViewportWrapperPtr ptr(
            new ViewportWrapper(
                getHost(),
                viewportHostHandle,
                viewportPluginHandle,
                viewportPluginSuite));

        m_viewportWrapper = ptr;
    }

    return m_viewportWrapper;
}

const double* ViewportCameraPluginBase::getViewMatrix()
{
    return m_hostSuite->getViewMatrix(m_hostHandle);
}

Matrix44d ViewportCameraPluginBase::getViewMatrix44d()
{
    const double* v = getViewMatrix();

    if (v)
    {
        return Matrix44d(v);
    }
    else
    {
        return Matrix44d();
    }
}

void ViewportCameraPluginBase::setViewMatrix(const double* matrix)
{
    m_hostSuite->setViewMatrix(m_hostHandle, matrix);
}

const double* ViewportCameraPluginBase::getProjectionMatrix()
{
    return m_hostSuite->getProjectionMatrix(m_hostHandle);
}

Matrix44d ViewportCameraPluginBase::getProjectionMatrix44d()
{
    const double* p = getProjectionMatrix();

    if (p)
    {
        return Matrix44d(p);
    }
    else
    {
        return Matrix44d();
    }
}

void ViewportCameraPluginBase::setProjectionMatrix(const double* matrix)
{
    m_hostSuite->setProjectionMatrix(m_hostHandle, matrix);
}

double ViewportCameraPluginBase::getCenterOfInterest()
{
    return m_hostSuite->getCenterOfInterest(m_hostHandle);
}

void ViewportCameraPluginBase::setCenterOfInterest(double coi)
{
    m_hostSuite->setCenterOfInterest(m_hostHandle, coi);
}

double ViewportCameraPluginBase::getFOV()
{
    return m_hostSuite->getFOV(m_hostHandle);
}

void ViewportCameraPluginBase::setFOV(double fov)
{
    m_hostSuite->setFOV(m_hostHandle, fov);
}

double ViewportCameraPluginBase::getOrthographicWidth()
{
    return m_hostSuite->getOrthographicWidth(m_hostHandle);
}

void ViewportCameraPluginBase::setOrthographicWidth(double width)
{
    m_hostSuite->setOrthographicWidth(m_hostHandle, width);
}

void ViewportCameraPluginBase::getNearFar(double &near, double &far)
{
    m_hostSuite->getNearFar(m_hostHandle, &near, &far);
}

void ViewportCameraPluginBase::setNearFar(double near, double far)
{
    m_hostSuite->setNearFar(m_hostHandle, near, far);
}

void ViewportCameraPluginBase::getScreenWindow(double &left, double &right, double &bottom, double &top)
{
    m_hostSuite->getScreenWindow(m_hostHandle, &left, &right, &bottom, &top);
}

void ViewportCameraPluginBase::setScreenWindow(double left, double right, double bottom, double top)
{
    m_hostSuite->setScreenWindow(m_hostHandle, left, right, bottom, top);
}

bool ViewportCameraPluginBase::hasLocationPath()
{
    return m_hostSuite->hasLocationPath(m_hostHandle);
}

std::string ViewportCameraPluginBase::getLocationPath()
{
    const char* result =  m_hostSuite->getLocationPath(m_hostHandle);
    if (!result)
    {
        return std::string(); // empty string
    }

    return result;
}

void ViewportCameraPluginBase::setLocationPath(const std::string& locationPath)
{
    m_hostSuite->setLocationPath(m_hostHandle, locationPath.c_str());
}

Vec3d ViewportCameraPluginBase::getOrigin()
{
    double x, y, z;
    m_hostSuite->getOrigin(m_hostHandle, &x, &y, &z);

    return Vec3d(x, y, z);
}

Vec3d ViewportCameraPluginBase::getDirection()
{
    double x, y, z;
    m_hostSuite->getDirection(m_hostHandle, &x, &y, &z);

    return Vec3d(x, y, z);
}

Vec3d ViewportCameraPluginBase::getUp()
{
    double x, y, z;
    m_hostSuite->getUp(m_hostHandle, &x, &y, &z);

    return Vec3d(x, y, z);
}

Vec3d ViewportCameraPluginBase::getLeft()
{
    double x, y, z;
    m_hostSuite->getLeft(m_hostHandle, &x, &y, &z);

    return Vec3d(x, y, z);
}

bool ViewportCameraPluginBase::isInteractionDisabled()
{
    return m_hostSuite->isInteractionDisabled(m_hostHandle);
}

void ViewportCameraPluginBase::disableInteraction(bool disabled)
{
    m_hostSuite->disableInteraction(m_hostHandle, disabled);
}

FnPluginHost* ViewportCameraPluginBase::getHost() { return m_host; }

FnPluginHost *ViewportCameraPluginBase::m_host = 0x0;

///////////////////////////
// ViewportCamera implementation
///////////////////////////

ViewportCamera::ViewportCamera()
{}

ViewportCamera::~ViewportCamera()
{}

FnViewportCameraPluginSuite_v2 ViewportCamera::createSuite(
    FnViewportCameraPluginHandle (*create)(
        FnViewportCameraHostHandle hostHandle))
{
    FnViewportCameraPluginSuite_v2 suite = {
        create,
        ::ViewportCamera_destroy,
        ::ViewportCamera_getCameraTypeID,
        ::ViewportCamera_setViewportDimensions,
        ::ViewportCamera_setOption,
        ::ViewportCamera_getOption,
        ::ViewportCamera_translate,
        ::ViewportCamera_rotate,
        ::ViewportCamera_setup,
        ::ViewportCamera_asAttribute,
        ::ViewportCamera_getPointOnPlane,
        ::ViewportCamera_projectObjectIntoWindow,
        ::ViewportCamera_projectWindowIntoObject,
        ::ViewportCamera_startInteraction,
        ::ViewportCamera_endInteraction,
        ::ViewportCamera_setDirty,
        ::ViewportCamera_getRay,
    };

    return suite;
}

FnAttribute::Attribute ViewportCamera::getOption(
    OptionIdGenerator::value_type optionId)
{
    return FnAttribute::Attribute();
}

void ViewportCamera::setOption(const std::string& name,
    FnAttribute::Attribute attr)
{
    setOption(OptionIdGenerator::GenerateId(name.c_str()), attr);
}

FnAttribute::Attribute ViewportCamera::getOption(const std::string& name)
{
    return getOption(OptionIdGenerator::GenerateId(name.c_str()));
}

FnViewportCameraPluginHandle ViewportCamera::newViewportCameraHandle(
    ViewportCamera* viewportCamera)
{
    if (!viewportCamera)
    {
        return 0x0;
    }

    FnViewportCameraPluginStruct* pluginHandle =
        new FnViewportCameraPluginStruct(viewportCamera);

    return (FnViewportCameraPluginHandle) pluginHandle;
}

void ViewportCamera::setHostHandle(FnViewportCameraHostHandle hostHandle)
{
    m_hostHandle = hostHandle;
    m_hostSuite = const_cast<FnViewportCameraHostSuite_v2 *>(
        reinterpret_cast<const FnViewportCameraHostSuite_v2 *>(
            m_host->getSuite("ViewportCameraHost", 2)));

    if (!m_hostSuite)
    {
        std::ostringstream os;
        os << "This ViewportCamera plugin requires version 2 of the host API. "
           << "A more recent version of Katana is probably required by this "
           << "plugin.";
        throw std::runtime_error(os.str());
    }
}

FnViewportCameraHostHandle ViewportCamera::getHostHandle()
{
    return m_hostHandle;
}

unsigned int ViewportCamera::_apiVersion = 2;
const char* ViewportCamera::_apiName     = "ViewportCameraPlugin";


///////////////////////////
// ViewportWrapper implementation
///////////////////////////

ViewportCameraWrapper::ViewportCameraWrapper(
    FnPluginHost* host,
    FnViewportCameraHostHandle hostHandle,
    FnViewportCameraPluginHandle pluginHandle,
    FnViewportCameraPluginSuite_v2* pluginSuite)
{
    if (!getHost()) { setHost(host); }

    m_hostHandle   = hostHandle;
    m_pluginHandle = pluginHandle;
    m_pluginSuite  = pluginSuite;
    m_hostSuite    = const_cast<FnViewportCameraHostSuite_v2 *>(
        reinterpret_cast<const FnViewportCameraHostSuite_v2 *>(
            m_host->getSuite("ViewportCameraHost", 2)));

    if (!m_hostSuite)
    {
        std::ostringstream os;
        os << "This ViewportCamera plugin requires version 2 of the host API. "
           << "A more recent version of Katana is probably required by this "
           << "plugin.";
        throw std::runtime_error(os.str());
    }
}

ViewportCameraWrapper::~ViewportCameraWrapper()
{

}

ViewportCamera* ViewportCameraWrapper::getPluginPointer()
{
    return m_pluginHandle->getViewportCamera();
}

int ViewportCameraWrapper::getCameraTypeID()
{
    return m_pluginSuite->getCameraTypeID(m_pluginHandle);
}

void ViewportCameraWrapper::setViewportDimensions(unsigned int width,
    unsigned int height)
{
    m_pluginSuite->setViewportDimensions(m_pluginHandle, width, height);
}

void ViewportCameraWrapper::setOption(OptionIdGenerator::value_type optionId,
    FnAttribute::Attribute attr)
{
    // FIXME(DL): Non-standard ref-counting policy. TP 373492
    m_pluginSuite->setOption(m_pluginHandle, optionId,
        attr.getRetainedHandle());
}

FnAttribute::Attribute ViewportCameraWrapper::getOption(
    OptionIdGenerator::value_type optionId)
{
    FnAttributeHandle attrHandle =
        m_pluginSuite->getOption(m_pluginHandle, optionId);
    FnAttribute::Attribute attr =
        FnAttribute::Attribute::CreateAndSteal(attrHandle);
    return attr;
}

void ViewportCameraWrapper::setOption(const std::string& name,
    FnAttribute::Attribute attr)
{
    setOption(OptionIdGenerator::GenerateId(name.c_str()), attr);
}

FnAttribute::Attribute ViewportCameraWrapper::getOption(const std::string& name)
{
    return getOption(OptionIdGenerator::GenerateId(name.c_str()));
}

void ViewportCameraWrapper::translate(double x, double y, double z)
{
    m_pluginSuite->translate(m_pluginHandle, x, y, z);
}

void ViewportCameraWrapper::rotate(double x, double y, double z)
{
    m_pluginSuite->rotate(m_pluginHandle, x, y, z);
}

void ViewportCameraWrapper::startInteraction()
{
    m_pluginSuite->startInteraction(m_pluginHandle);
}

void ViewportCameraWrapper::endInteraction()
{
    m_pluginSuite->endInteraction(m_pluginHandle);
}

void ViewportCameraWrapper::setup(FnAttribute::GroupAttribute attr)
{
    // FIXME(DL): Non-standard ref-counting policy. TP 373492
    m_pluginSuite->setup(m_pluginHandle, attr.getRetainedHandle());
}

FnAttribute::GroupAttribute ViewportCameraWrapper::asAttribute()
{
    FnAttributeHandle attrHandle =
        m_pluginSuite->asAttribute(m_pluginHandle);
    FnAttribute::Attribute attr =
        FnAttribute::Attribute::CreateAndSteal(attrHandle);
    return attr;
}

bool ViewportCameraWrapper::getPointOnPlane(int x, int y,
    const Vec3d& planeOrigin,
    const Vec3d& planeNormal, Vec3d& intersection)
{
    double resultX, resultY, resultZ;
    bool hit = m_pluginSuite->getPointOnPlane(m_pluginHandle,
        x, y,
        planeOrigin.x, planeOrigin.y, planeOrigin.z,
        planeNormal.x, planeNormal.y, planeNormal.z,
        &resultX, &resultY, &resultZ);

    intersection.x = resultX;
    intersection.y = resultY;
    intersection.z = resultZ;

    return hit;
}

Vec3d ViewportCameraWrapper::projectObjectIntoWindow(Vec3d point)
{
    double x, y, z;
    m_pluginSuite->projectObjectIntoWindow(m_pluginHandle, point.x, point.y, point.z, &x, &y, &z);

    return Vec3d(x, y, z);
}

Vec3d ViewportCameraWrapper::projectWindowIntoObject(Vec3d point)
{
    double x, y, z;
    m_pluginSuite->projectWindowIntoObject(m_pluginHandle, point.x, point.y, point.z, &x, &y, &z);

    return Vec3d(x, y, z);
}

void ViewportCameraWrapper::setDirty(CameraDirtyBits dirtyBits)
{
    m_pluginSuite->setDirty(m_pluginHandle, dirtyBits);
}

void ViewportCameraWrapper::getRay(
    int x, int y, Vec3d& pos, Vec3d& dir)
{
    m_pluginSuite->getRay(m_pluginHandle,
        x, y, &pos.x, &pos.y, &pos.z, &dir.x, &dir.y, &dir.z);
}

}
}
}
