// Copyright (c) 2016 The Foundry Visionmongers Ltd. All Rights Reserved.

#include <FnViewer/plugin/FnViewport.h>
#include <FnViewer/plugin/FnViewportLayer.h>
#include <FnViewer/plugin/FnViewportCamera.h>

#include <FnPluginManager/FnPluginManager.h>
#include <FnAttribute/FnAttribute.h>
#include <FnAttribute/FnGroupBuilder.h>
#include <FnAttribute/FnAttribute.h>

#include <FnLogging/FnLogging.h>

#include "FnViewer/utils/OpenGLTiming.h"

#include <iostream>

////////////////////////////////////////////////////
// C callbacks implementations for the plugin suite
////////////////////////////////////////////////////
void Viewport_destroy(FnViewportPluginHandle handle)
{
    delete handle;
}

void Viewport_setup(FnViewportPluginHandle handle)
{
    handle->getViewport()->setup();
}

void Viewport_cleanup(FnViewportPluginHandle handle)
{
    handle->getViewport()->cleanup();
}

bool Viewport_event(FnViewportPluginHandle handle,
                    FnAttributeHandle eventAttrHandle)
{
    FnAttribute::GroupAttribute eventAttr =
        FnAttribute::Attribute::CreateAndRetain(eventAttrHandle);
    if (eventAttr.isValid())
    {
        Foundry::Katana::ViewerAPI::FnEventWrapper eventWrapper(eventAttr);
        return handle->getViewport()->event(eventWrapper);
    }
    return false;
}

void Viewport_draw(FnViewportPluginHandle handle)
{
    handle->getViewport()->draw();
}

void Viewport_pickerDraw(FnViewportPluginHandle handle,
    unsigned int x, unsigned int y,
    unsigned int w, unsigned int h,
    FnAttributeHandle ignoreAttrHandle)
{
    Foundry::Katana::ViewerAPI::PickedAttrsMap ignoreAttrs;
    ignoreAttrs.fromGroupAttribute(
        FnAttribute::Attribute::CreateAndRetain(ignoreAttrHandle));

    handle->getViewport()->pickerDraw(x, y, w, h, ignoreAttrs);
}

int Viewport_customPick(FnViewportPluginHandle handle,
    unsigned int x, unsigned int y,
    unsigned int w, unsigned int h,
    int deepPicking,
    FnAttributeHandle& pickedAttrHandle,
    float* singlePointDepth)
{
    Foundry::Katana::ViewerAPI::PickedAttrsMap pickedAttrs;
    if (!handle->getViewport()->customPick(x, y, w, h, deepPicking,
                                           pickedAttrs,
                                           singlePointDepth))
    {
        return false;
    }

    // Transform the map into a GroupAttribute
    pickedAttrHandle = pickedAttrs.toGroupAttribute().getRetainedHandle();

    return true;    
}

void Viewport_resize(FnViewportPluginHandle handle, unsigned int width,
    unsigned int height)
{
    handle->getViewport()->resize(width, height);
}

void Viewport_hover(FnViewportPluginHandle handle, uint8_t isHovering,
    int x, int y)
{
    handle->getViewport()->hover(isHovering, x, y);
}

void Viewport_setOption(FnViewportPluginHandle handle, uint64_t optionId,
    FnAttributeHandle attrHandle)
{
    // FIXME(DL): Non-standard ref-counting policy. TP 373492
    FnAttribute::Attribute attr =
        FnAttribute::Attribute::CreateAndSteal(attrHandle);

    handle->getViewport()->setOption(optionId, attr);
}

FnAttributeHandle Viewport_getOption(FnViewportPluginHandle handle,
    uint64_t optionId)
{
    return handle->getViewport()->getOption(optionId).getRetainedHandle();
}

void Viewport_freeze(FnViewportPluginHandle handle)
{
    handle->getViewport()->freeze();
}

void Viewport_thaw(FnViewportPluginHandle handle)
{
    handle->getViewport()->thaw();
}


///////////////////////////
// FnViewport implementation
///////////////////////////
namespace Foundry
{
namespace Katana
{
namespace ViewerAPI
{

///////////////////////////
// ViewportPluginBase implementation
///////////////////////////

ViewportPluginBase::ViewportPluginBase() :
    m_hostSuite(NULL),
    m_hostHandle(NULL)
{}

ViewportPluginBase::~ViewportPluginBase()
{}

std::string ViewportPluginBase::getName()
{
    return std::string( m_hostSuite->getName(m_hostHandle) );
}

ViewerDelegateWrapperPtr ViewportPluginBase::getViewerDelegate()
{
    if (!m_viewerDelegateWrapper)
    {
        // Initialize the shared pointer to the ViewerDelegateWrapper on the
        // first time this is called. This can be cached here because Since a
        // Viewport always has the same ViewerDelegate throughout its lifetime.
        FnViewerDelegateHostHandle viewerDelegateHostHandle;
        FnViewerDelegatePluginHandle viewerDelegatePluginHandle;
        FnViewerDelegatePluginSuite_v3* viewerDelegatePluginSuite = 0x0;

        m_hostSuite->getViewerDelegate(m_hostHandle,
            &viewerDelegateHostHandle,
            &viewerDelegatePluginHandle,
            &viewerDelegatePluginSuite);

        ViewerDelegateWrapperPtr ptr(
            new ViewerDelegateWrapper(
                getHost(),
                viewerDelegateHostHandle,
                viewerDelegatePluginHandle,
                viewerDelegatePluginSuite));

        m_viewerDelegateWrapper = ptr;
    }

    return m_viewerDelegateWrapper;
}

unsigned int ViewportPluginBase::getWidth()
{
    return m_hostSuite->getWidth(m_hostHandle);
}

unsigned int ViewportPluginBase::getHeight()
{
    return m_hostSuite->getHeight(m_hostHandle);
}

ViewportCameraWrapperPtr ViewportPluginBase::getActiveCamera()
{
    FnViewportCameraHostHandle viewportCameraHostHandle;
    FnViewportCameraPluginHandle viewportCameraPluginHandle;
    FnViewportCameraPluginSuite_v2* viewportCameraPluginSuite;

    m_hostSuite->getActiveCamera(m_hostHandle,
        &viewportCameraHostHandle,
        &viewportCameraPluginHandle,
        &viewportCameraPluginSuite);


    ViewportCameraWrapperPtr ptr(
        new ViewportCameraWrapper(
            getHost(),
            viewportCameraHostHandle,
            viewportCameraPluginHandle,
            viewportCameraPluginSuite));

    return ptr;
}

bool ViewportPluginBase::isDirty()
{
    return m_hostSuite->isDirty(m_hostHandle) != 0;
}

void ViewportPluginBase::setDirty(bool dirty)
{
    m_hostSuite->setDirty(m_hostHandle, dirty? 1: 0 );
}


ViewportCameraWrapperPtr ViewportPluginBase::addCamera(
    const std::string& pluginName, const std::string& name)
{
    FnViewportCameraHostHandle viewportCameraHostHandle;
    FnViewportCameraPluginHandle viewportCameraPluginHandle;
    FnViewportCameraPluginSuite_v2* viewportCameraPluginSuite;

    m_hostSuite->addCamera(m_hostHandle,
        pluginName.c_str(),
        name.c_str(),
        &viewportCameraHostHandle,
        &viewportCameraPluginHandle,
        &viewportCameraPluginSuite);

    if (!viewportCameraHostHandle ||
        !viewportCameraPluginHandle ||
        !viewportCameraPluginSuite)
    {
        return Foundry::Katana::ViewerAPI::ViewportCameraWrapperPtr();
    }

    ViewportCameraWrapperPtr ptr(
        new ViewportCameraWrapper(
            getHost(),
            viewportCameraHostHandle,
            viewportCameraPluginHandle,
            viewportCameraPluginSuite));

    return ptr;
}

void ViewportPluginBase::removeCamera(
    ViewportCameraWrapperPtr camera)
{
    m_hostSuite->removeCamera(m_hostHandle,
        camera->getCameraHostHandle());
}

void ViewportPluginBase::setActiveCamera(
    ViewportCameraWrapperPtr camera)
{
    m_hostSuite->setActiveCamera(m_hostHandle,
        camera->getCameraHostHandle());
}

void ViewportPluginBase::setViewFrozen(bool isViewFrozen)
{
    m_hostSuite->setViewFrozen(m_hostHandle, static_cast<int>(isViewFrozen));
}

bool ViewportPluginBase::isViewFrozen() const
{
    return m_hostSuite->isViewFrozen(m_hostHandle) != 0;
}

//=============================================================================

ViewportCameraWrapperPtr ViewportPluginBase::getCamera(
    const std::string& name) const
{
    FnViewportCameraHostHandle viewportCameraHostHandle;
    FnViewportCameraPluginHandle viewportCameraPluginHandle;
    FnViewportCameraPluginSuite_v2* viewportCameraPluginSuite;

    m_hostSuite->getCameraByName(m_hostHandle, name.c_str(),
        &viewportCameraHostHandle,
        &viewportCameraPluginHandle,
        &viewportCameraPluginSuite);

    if (!viewportCameraHostHandle ||
        !viewportCameraPluginHandle ||
        !viewportCameraPluginSuite)
    {
        Foundry::Katana::ViewerAPI::ViewportCameraWrapperPtr nullPtr;
        return nullPtr;
    }

    ViewportCameraWrapperPtr ptr(
        new ViewportCameraWrapper(
            getHost(),
            viewportCameraHostHandle,
            viewportCameraPluginHandle,
            viewportCameraPluginSuite));

    return ptr;
}

ViewportCameraWrapperPtr ViewportPluginBase::getCamera(unsigned int index) const
{
    FnViewportCameraHostHandle viewportCameraHostHandle;
    FnViewportCameraPluginHandle viewportCameraPluginHandle;
    FnViewportCameraPluginSuite_v2* viewportCameraPluginSuite;

    m_hostSuite->getCameraByIndex(m_hostHandle, index,
        &viewportCameraHostHandle,
        &viewportCameraPluginHandle,
        &viewportCameraPluginSuite);

    if (!viewportCameraHostHandle ||
        !viewportCameraPluginHandle ||
        !viewportCameraPluginSuite)
    {
        Foundry::Katana::ViewerAPI::ViewportCameraWrapperPtr nullPtr;
        return nullPtr;
    }

    ViewportCameraWrapperPtr ptr(
        new ViewportCameraWrapper(
            getHost(),
            viewportCameraHostHandle,
            viewportCameraPluginHandle,
            viewportCameraPluginSuite));

    return ptr;
}

void ViewportPluginBase::removeCamera(const std::string& name)
{
    m_hostSuite->removeCameraByName(m_hostHandle, name.c_str());
}

void ViewportPluginBase::removeCamera(unsigned int index)
{
    m_hostSuite->removeCameraByIndex(m_hostHandle, index);
}

unsigned int ViewportPluginBase::getNumberOfCameras() const
{
    return m_hostSuite->getNumberOfCameras(m_hostHandle);
}

std::string ViewportPluginBase::getCameraName(unsigned int index)
{
    return std::string( m_hostSuite->getCameraName(m_hostHandle, index) );
}

std::string ViewportPluginBase::getCameraName(ViewportCameraWrapperPtr camera)
{
    const unsigned int cameraCount = getNumberOfCameras();
    for (unsigned int i = 0; i < cameraCount; ++i)
    {
        ViewportCameraWrapperPtr cam = getCamera(i);
        if (cam->getCameraHostHandle() == camera->getCameraHostHandle())
        {
            return getCameraName(i);
        }
    }
    return std::string();
}

int ViewportPluginBase::getCameraIndex(const std::string& name)
{
    return m_hostSuite->getCameraIndex(m_hostHandle, name.c_str());
}
//=============================================================================
ViewportLayerWrapperPtr ViewportPluginBase::addLayer(
    const std::string& pluginName, const std::string& name)
{
    FnViewportLayerHostHandle viewportLayerHostHandle;
    FnViewportLayerPluginHandle viewportLayerPluginHandle;
    FnViewportLayerPluginSuite_v2* viewportLayerPluginSuite;

    m_hostSuite->addLayer(m_hostHandle, pluginName.c_str(), name.c_str(),
        &viewportLayerHostHandle,
        &viewportLayerPluginHandle,
        &viewportLayerPluginSuite);

    if (!viewportLayerHostHandle ||
        !viewportLayerPluginHandle ||
        !viewportLayerPluginSuite)
    {
        Foundry::Katana::ViewerAPI::ViewportLayerWrapperPtr nullPtr;
        return nullPtr;
    }

    ViewportLayerWrapperPtr ptr(
        new ViewportLayerWrapper(
            getHost(),
            viewportLayerHostHandle,
            viewportLayerPluginHandle,
            viewportLayerPluginSuite));

    return ptr;
}

ViewportLayerWrapperPtr ViewportPluginBase::insertLayer(
    const std::string& pluginName, const std::string& name, unsigned int index)
{
    FnViewportLayerHostHandle viewportLayerHostHandle;
    FnViewportLayerPluginHandle viewportLayerPluginHandle;
    FnViewportLayerPluginSuite_v2* viewportLayerPluginSuite;

    m_hostSuite->insertLayer(m_hostHandle,
        pluginName.c_str(), name.c_str(), index,
        &viewportLayerHostHandle,
        &viewportLayerPluginHandle,
        &viewportLayerPluginSuite);

    if (!viewportLayerHostHandle ||
        !viewportLayerPluginHandle ||
        !viewportLayerPluginSuite)
    {
        Foundry::Katana::ViewerAPI::ViewportLayerWrapperPtr nullPtr;
        return nullPtr;
    }

    ViewportLayerWrapperPtr ptr(
        new ViewportLayerWrapper(
            getHost(),
            viewportLayerHostHandle,
            viewportLayerPluginHandle,
            viewportLayerPluginSuite));

    return ptr;
}

ViewportLayerWrapperPtr ViewportPluginBase::getLayer(
    const std::string& name) const
{
    FnViewportLayerHostHandle viewportLayerHostHandle;
    FnViewportLayerPluginHandle viewportLayerPluginHandle;
    FnViewportLayerPluginSuite_v2* viewportLayerPluginSuite;

    m_hostSuite->getLayerByName(m_hostHandle, name.c_str(),
        &viewportLayerHostHandle,
        &viewportLayerPluginHandle,
        &viewportLayerPluginSuite);

    if (!viewportLayerHostHandle ||
        !viewportLayerPluginHandle ||
        !viewportLayerPluginSuite)
    {
        Foundry::Katana::ViewerAPI::ViewportLayerWrapperPtr nullPtr;
        return nullPtr;
    }

    ViewportLayerWrapperPtr ptr(
        new ViewportLayerWrapper(
            getHost(),
            viewportLayerHostHandle,
            viewportLayerPluginHandle,
            viewportLayerPluginSuite));

    return ptr;
}

ViewportLayerWrapperPtr ViewportPluginBase::getLayer(unsigned int index) const
{
    FnViewportLayerHostHandle viewportLayerHostHandle;
    FnViewportLayerPluginHandle viewportLayerPluginHandle;
    FnViewportLayerPluginSuite_v2* viewportLayerPluginSuite;

    m_hostSuite->getLayerByIndex(m_hostHandle, index,
        &viewportLayerHostHandle,
        &viewportLayerPluginHandle,
        &viewportLayerPluginSuite);

    if (!viewportLayerHostHandle ||
        !viewportLayerPluginHandle ||
        !viewportLayerPluginSuite)
    {
        Foundry::Katana::ViewerAPI::ViewportLayerWrapperPtr nullPtr;
        return nullPtr;
    }

    ViewportLayerWrapperPtr ptr(
        new ViewportLayerWrapper(
            getHost(),
            viewportLayerHostHandle,
            viewportLayerPluginHandle,
            viewportLayerPluginSuite));

    return ptr;
}

void ViewportPluginBase::removeLayer(const std::string& name)
{
    m_hostSuite->removeLayerByName(m_hostHandle, name.c_str());
}

void ViewportPluginBase::removeLayer(unsigned int index)
{
    m_hostSuite->removeLayerByIndex(m_hostHandle, index);
}

unsigned int ViewportPluginBase::getNumberOfLayers() const
{
    return m_hostSuite->getNumberOfLayers(m_hostHandle);
}

std::string ViewportPluginBase::getLayerName(unsigned int index)
{
    return std::string( m_hostSuite->getLayerName(m_hostHandle, index) );
}

int ViewportPluginBase::getLayerIndex(const std::string& name)
{
    return m_hostSuite->getLayerIndex(m_hostHandle, name.c_str());
}

const double* ViewportPluginBase::getViewMatrix()
{
    return m_hostSuite->getViewMatrix(m_hostHandle);
}

Matrix44d ViewportPluginBase::getViewMatrix44d()
{
    const double* v = getViewMatrix();

    if (v)
    {
        return Matrix44d(v);
    }
    else
    {
        return Matrix44d();
    }
}

const double* ViewportPluginBase::getProjectionMatrix()
{
    return m_hostSuite->getProjectionMatrix(m_hostHandle);
}

Matrix44d ViewportPluginBase::getProjectionMatrix44d()
{
    const double* p = getProjectionMatrix();

    if (p)
    {
        return Matrix44d(p);
    }
    else
    {
        return Matrix44d();
    }
}

void ViewportPluginBase::activateManipulator(const std::string& name,
    const std::vector<std::string>& locationPaths)
{
    FnAttribute::StringAttribute locationPathsAttr(locationPaths);

    m_hostSuite->activateManipulator(m_hostHandle, name.c_str(),
        locationPathsAttr.getHandle());
}

void ViewportPluginBase::deactivateManipulator(const std::string& name)
{
    m_hostSuite->deactivateManipulator(m_hostHandle, name.c_str());
}

void ViewportPluginBase::deactivateAllManipulators()
{
    m_hostSuite->deactivateAllManipulators(m_hostHandle);
}

unsigned int ViewportPluginBase::getNumberOfActiveManipulators()
{
    return m_hostSuite->getNumberOfActiveManipulators(m_hostHandle);
}

ManipulatorWrapperPtr ViewportPluginBase::getActiveManipulator(
    unsigned int index)
{
    FnManipulatorHostHandle manipulatorHostHandle;
    FnManipulatorPluginHandle manipulatorPluginHandle;
    FnManipulatorPluginSuite_v1* manipulatorPluginSuite;

    m_hostSuite->getActiveManipulatorByIndex(m_hostHandle, index,
        &manipulatorHostHandle,
        &manipulatorPluginHandle,
        &manipulatorPluginSuite);

    if (!manipulatorHostHandle ||
        !manipulatorPluginHandle ||
        !manipulatorPluginSuite)
    {
        Foundry::Katana::ViewerAPI::ManipulatorWrapperPtr nullPtr;
        return nullPtr;
    }

    ManipulatorWrapperPtr ptr(
        new ManipulatorWrapper(
            getHost(),
            manipulatorHostHandle,
            manipulatorPluginHandle,
            manipulatorPluginSuite));

    return ptr;
}

ManipulatorWrapperPtr ViewportPluginBase::getActiveManipulator(
    const std::string& pluginName)
{
    FnManipulatorHostHandle manipulatorHostHandle;
    FnManipulatorPluginHandle manipulatorPluginHandle;
    FnManipulatorPluginSuite_v1* manipulatorPluginSuite;

    m_hostSuite->getActiveManipulatorByName(m_hostHandle, pluginName.c_str(),
        &manipulatorHostHandle,
        &manipulatorPluginHandle,
        &manipulatorPluginSuite);

    if (!manipulatorHostHandle ||
        !manipulatorPluginHandle ||
        !manipulatorPluginSuite)
    {
        Foundry::Katana::ViewerAPI::ManipulatorWrapperPtr nullPtr;
        return nullPtr;
    }

    ManipulatorWrapperPtr ptr(
        new ManipulatorWrapper(
            getHost(),
            manipulatorHostHandle,
            manipulatorPluginHandle,
            manipulatorPluginSuite));

    return ptr;
}

bool ViewportPluginBase::makeGLContextCurrent()
{
    return m_hostSuite->makeGLContextCurrent(m_hostHandle);
}

bool ViewportPluginBase::doneGLContextCurrent()
{
    return m_hostSuite->doneGLContextCurrent(m_hostHandle);
}

bool ViewportPluginBase::isGLContextCurrent()
{
    return m_hostSuite->isGLContextCurrent(m_hostHandle);
}

unsigned int ViewportPluginBase::getDefaultFramebufferObject()
{
    return m_hostSuite->getDefaultFramebufferObject(m_hostHandle);
}

void ViewportPluginBase::enableCap(uint32_t cap)
{
    return m_hostSuite->enableCap(m_hostHandle, cap);
}

void ViewportPluginBase::disableCap(uint32_t cap)
{
    return m_hostSuite->disableCap(m_hostHandle, cap);
}

void ViewportPluginBase::pick(
    unsigned int x, unsigned int y,
    unsigned int w, unsigned int h,
    bool deepPicking,
    PickedAttrsMap& pickedAttrs,
    float* singlePointDepth)
{
    FnAttribute::GroupAttribute attr =
        FnAttribute::Attribute::CreateAndSteal(
            m_hostSuite->pick(
                m_hostHandle, x, y, w, h, deepPicking, singlePointDepth
            )
        );

    // Parse the GroupAttr contents and add it to the map
    pickedAttrs.fromGroupAttribute(attr);
}

void ViewportPluginBase::setPanAndZoomActive(bool active)
{
    return m_hostSuite->setPanAndZoomActive(m_hostHandle, active);
}

bool ViewportPluginBase::isPanAndZoomActive()
{
    return m_hostSuite->isPanAndZoomActive(m_hostHandle);
}

float ViewportPluginBase::getHorizontalOffset()
{
    return m_hostSuite->getHorizontalOffset(m_hostHandle);
}

float ViewportPluginBase::getVerticalOffset()
{
    return m_hostSuite->getVerticalOffset(m_hostHandle);
}

float ViewportPluginBase::getZoom()
{
    return m_hostSuite->getZoom(m_hostHandle);
}

FnPlugStatus ViewportPluginBase::setHost(FnPluginHost* host)
{
    m_host = host;

    FnPlugStatus status = FnPluginStatusOK;

    status = FnPluginManager::PluginManager::setHost(host);
    if (status != FnPluginStatusOK)
        return status;

    // We don't verity the return value of OptionIdGenerator::setHost() as the
    // host suite is not guaranteed to be registered yet.
    OptionIdGenerator::setHost(host);

    status = FnLogging::setHost(host);
    if (status != FnPluginStatusOK)
        return status;

    status = FnAttribute::GroupBuilder::setHost(host);
    if (status != FnPluginStatusOK)
        return status;

    status = FnAttribute::Attribute::setHost(host);
    return status;
}

FnPluginHost* ViewportPluginBase::getHost() { return m_host; }

FnPluginHost *ViewportPluginBase::m_host = 0x0;


///////////////////////////
// Viewport implementation
///////////////////////////

Viewport::Viewport()
{}

Viewport::~Viewport()
{}

bool Viewport::event(const FnEventWrapper& eventData)
{
    for(int i = getNumberOfLayers() - 1; i >= 0 ; --i)
    {
        ViewportLayerWrapperPtr layer = getLayer(static_cast<unsigned int>(i));
        if(layer->event(eventData))
            return true;
    }
    return false;
}

void Viewport::draw()
{
    GLTIMING_FRAMESCOPE();
    for(unsigned int i = 0; i < getNumberOfLayers(); ++i)
    {
        ViewportLayerWrapperPtr layer = getLayer(i);
        layer->draw();
    }
}

void Viewport::resize(unsigned int width, unsigned int height)
{}

void Viewport::hover(bool isHovering, int x, int y)
{
    for(unsigned int i = 0; i < getNumberOfLayers(); ++i)
    {
        ViewportLayerWrapperPtr layer = getLayer(i);
        layer->hover(isHovering, x, y);
    }
}

FnAttribute::Attribute Viewport::getOption(
    OptionIdGenerator::value_type optionId)
{
    return FnAttribute::Attribute();
}

void Viewport::setOption(const std::string& name, FnAttribute::Attribute attr)
{
    setOption(OptionIdGenerator::GenerateId(name.c_str()), attr);
}

FnAttribute::Attribute Viewport::getOption(const std::string& name)
{
    return getOption(OptionIdGenerator::GenerateId(name.c_str()));
}

FnPickId Viewport::addPickableObject(FnAttribute::Attribute attr)
{
    return m_hostSuite->addPickableObject(m_hostHandle, attr.getHandle());
}

bool Viewport::customPick(
    unsigned int x, unsigned int y,
    unsigned int w, unsigned int h,
    bool deepPicking,
    PickedAttrsMap& pickedAttrs,
    float* singlePointDepth)
{
    return false;
}

FnViewportPluginSuite_v2 Viewport::createSuite(
    FnViewportPluginHandle (*create)(FnViewportHostHandle hostHandle))
{
    FnViewportPluginSuite_v2 suite = {
        create,
        ::Viewport_destroy,
        ::Viewport_setup,
        ::Viewport_cleanup,
        ::Viewport_event,
        ::Viewport_draw,
        ::Viewport_resize,
        ::Viewport_hover,
        ::Viewport_setOption,
        ::Viewport_getOption,
        ::Viewport_freeze,
        ::Viewport_thaw,
        ::Viewport_pickerDraw,
        ::Viewport_customPick
    };

    return suite;
}

FnViewportPluginHandle Viewport::newViewportHandle(Viewport* viewport)
{
    if (!viewport)
    {
        return 0x0;
    }

    FnViewportPluginStruct* pluginHandle =
        new FnViewportPluginStruct(viewport);

    return (FnViewportPluginHandle) pluginHandle;
}

void Viewport::setHostHandle(FnViewportHostHandle hostHandle)
{
    m_hostHandle = hostHandle;
    m_hostSuite = const_cast<FnViewportHostSuite_v3 *>(
        reinterpret_cast<const FnViewportHostSuite_v3 *>(
            m_host->getSuite("ViewportHost", 3)));
}

unsigned int Viewport::_apiVersion = 2;
const char* Viewport::_apiName     = "ViewportPlugin";


///////////////////////////
// ViewportWrapper implementation
///////////////////////////

ViewportWrapper::ViewportWrapper(
    FnPluginHost* host,
    FnViewportHostHandle hostHandle,
    FnViewportPluginHandle pluginHandle,
    FnViewportPluginSuite_v2* pluginSuite)
{
    if (!getHost()) { setHost(host); }

    m_hostHandle = hostHandle;
    m_pluginHandle = pluginHandle;
    m_pluginSuite = pluginSuite;

    m_hostSuite = const_cast<FnViewportHostSuite_v3 *>(
        reinterpret_cast<const FnViewportHostSuite_v3 *>(
            m_host->getSuite("ViewportHost", 3)));
}

ViewportWrapper::~ViewportWrapper()
{}

bool ViewportWrapper::event(const FnEventWrapper& eventData)
{
    return m_pluginSuite->event(m_pluginHandle, eventData.getHandle());
}


void ViewportWrapper::setOption(OptionIdGenerator::value_type optionId,
    FnAttribute::Attribute attr)
{
    // FIXME(DL): Non-standard ref-counting policy. TP 373492
    m_pluginSuite->setOption(m_pluginHandle, optionId,
        attr.getRetainedHandle());
}

FnAttribute::Attribute ViewportWrapper::getOption(
    OptionIdGenerator::value_type optionId)
{
    FnAttributeHandle attrHandle =
        m_pluginSuite->getOption(m_pluginHandle, optionId);
    FnAttribute::Attribute attr =
        FnAttribute::Attribute::CreateAndSteal(attrHandle);
    return attr;
}

void ViewportWrapper::setOption(const std::string& name, FnAttribute::Attribute attr)
{
    setOption(OptionIdGenerator::GenerateId(name.c_str()), attr);
}

FnAttribute::Attribute ViewportWrapper::getOption(const std::string& name)
{
    return getOption(OptionIdGenerator::GenerateId(name.c_str()));
}

Viewport* ViewportWrapper::getPluginPointer()
{
    return m_pluginHandle->getViewport();
}


}
}
}
