// Copyright (c) 2016 The Foundry Visionmongers Ltd. All Rights Reserved.

#include <FnViewer/plugin/FnViewerDelegateComponent.h>
#include <FnViewer/plugin/FnViewerDelegate.h>

#include <FnPluginManager/FnPluginManager.h>
#include <FnAttribute/FnAttribute.h>
#include <FnAttribute/FnGroupBuilder.h>
#include <FnAttribute/FnAttribute.h>
#include <FnViewer/plugin/FnOptionIdGenerator.h>

#include <FnLogging/FnLogging.h>

#include <iostream>

////////////////////////////////////////////////////
// C callbacks implementations for the plugin suite
////////////////////////////////////////////////////
void ViewerDelegateComponent_destroy(
    FnViewerDelegateComponentPluginHandle handle)
{
    delete handle;
}

int ViewerDelegateComponent_locationEvent(
    FnViewerDelegateComponentPluginHandle handle,
    FnViewerLocationEventStruct* eventHandle,
    uint8_t locationHandled)
{
    return static_cast<int>(handle->getViewerDelegateComponent()->locationEvent(
            Foundry::Katana::ViewerAPI::ViewerLocationEvent(*eventHandle),
            locationHandled));
}

void ViewerDelegateComponent_locationsSelected(
    FnViewerDelegateComponentPluginHandle handle, const char** locationPaths,
    int64_t numPaths)
{
    std::vector<std::string> locationPathsVec;
    locationPathsVec.reserve(numPaths);
    for(unsigned int i = 0 ; i < numPaths; ++i)
    {
        std::string locationPath(locationPaths[i]);
        locationPathsVec.push_back(locationPath);
    }

    handle->getViewerDelegateComponent()->locationsSelected(locationPathsVec);
}

int ViewerDelegateComponent_isProcessing(
    FnViewerDelegateComponentPluginHandle handle)
{
    return static_cast<int>(
        handle->getViewerDelegateComponent()->isProcessing());
}


void ViewerDelegateComponent_setOption(
    FnViewerDelegateComponentPluginHandle handle,
    uint64_t optionId, FnAttributeHandle attrHandle)
{
    // FIXME(DL): Non-standard ref-counting policy. TP 373492
    FnAttribute::Attribute attr =
        FnAttribute::Attribute::CreateAndSteal(attrHandle);

    handle->getViewerDelegateComponent()->setOption(optionId, attr);
}

FnAttributeHandle ViewerDelegateComponent_getOption(
    FnViewerDelegateComponentPluginHandle handle, uint64_t optionId)
{
    return handle->getViewerDelegateComponent()->getOption(optionId).getRetainedHandle();
}

void ViewerDelegateComponent_setup(FnViewerDelegateComponentPluginHandle handle)
{
    handle->getViewerDelegateComponent()->setup();
}

void ViewerDelegateComponent_cleanup(
    FnViewerDelegateComponentPluginHandle handle)
{
    handle->getViewerDelegateComponent()->cleanup();
}

FnAttributeHandle ViewerDelegateComponent_getBounds(
    FnViewerDelegateComponentPluginHandle handle, const char* location)
{
    const FnAttribute::DoubleAttribute boundsAttr =
        handle->getViewerDelegateComponent()->getBounds(location);
    return boundsAttr.getRetainedHandle();
}

FnAttributeHandle ViewerDelegateComponent_computeExtent(
    FnViewerDelegateComponentPluginHandle handle, const char* location)
{
    const FnAttribute::DoubleAttribute extentAttr =
        handle->getViewerDelegateComponent()->computeExtent(location);
    return extentAttr.getRetainedHandle();
}

///////////////////////////
// FnViewerDelegateComponent implementation
///////////////////////////
namespace Foundry
{
namespace Katana
{
namespace ViewerAPI
{

///////////////////////////
// ViewerDelegateComponentPluginBase implementation
///////////////////////////

ViewerDelegateComponentPluginBase::ViewerDelegateComponentPluginBase()
    : m_hostSuite(NULL),
      m_hostHandle(NULL)
{
}

ViewerDelegateComponentPluginBase::~ViewerDelegateComponentPluginBase()
{
}

Foundry::Katana::ViewerAPI::ViewerDelegateWrapperPtr
    ViewerDelegateComponentPluginBase::getViewerDelegate()
{
    if (!m_viewerDelegateWrapper)
    {
        FnViewerDelegateHostHandle delegateHostHandle;
        FnViewerDelegatePluginHandle delegatePluginHandle;
        FnViewerDelegatePluginSuite_v3* delegatePluginSuite;

        m_hostSuite->getViewerDelegate(m_hostHandle,
            &delegateHostHandle,
            &delegatePluginHandle,
            &delegatePluginSuite);

        ViewerDelegateWrapperPtr ptr(
            new ViewerDelegateWrapper(
                getHost(),
                delegateHostHandle,
                delegatePluginHandle,
                delegatePluginSuite));

        m_viewerDelegateWrapper = ptr;
    }

    return m_viewerDelegateWrapper;
}

FnAttribute::GroupAttribute ViewerDelegateComponentPluginBase::getAttributes(
    const std::string& locationPath)
{
    return getViewerDelegate()->getAttributes(locationPath);
}

FnPlugStatus ViewerDelegateComponentPluginBase::setHost(FnPluginHost* host)
{
    m_host = host;

    FnPlugStatus status = FnPluginStatusOK;

    status = FnPluginManager::PluginManager::setHost(host);
    if (status != FnPluginStatusOK)
        return status;

    // We don't verity the return value of OptionIdGenerator::setHost() as the
    // host suite is not guaranteed to be registered yet.
    OptionIdGenerator::setHost(host);

    status = FnLogging::setHost(host);
    if (status != FnPluginStatusOK)
        return status;

    status = FnAttribute::GroupBuilder::setHost(host);
    if (status != FnPluginStatusOK)
        return status;

    status = FnAttribute::Attribute::setHost(host);
    return status;
}

FnPluginHost* ViewerDelegateComponentPluginBase::getHost() { return m_host; }

FnPluginHost *ViewerDelegateComponentPluginBase::m_host = 0x0;


///////////////////////////
// ViewerDelegateComponent implementation
///////////////////////////

ViewerDelegateComponent::ViewerDelegateComponent()
{}

ViewerDelegateComponent::~ViewerDelegateComponent()
{}

FnAttribute::Attribute ViewerDelegateComponent::getOption(
    OptionIdGenerator::value_type optionId)
{
    return FnAttribute::Attribute();
}

void ViewerDelegateComponent::setOption(const std::string& name, FnAttribute::Attribute attr)
{
    setOption(OptionIdGenerator::GenerateId(name.c_str()), attr);
}

FnAttribute::Attribute ViewerDelegateComponent::getOption(const std::string& name)
{
    return getOption(OptionIdGenerator::GenerateId(name.c_str()));
}

FnAttribute::DoubleAttribute ViewerDelegateComponent::getBounds(
    const std::string& location)
{
    return FnAttribute::DoubleAttribute();
}

FnAttribute::DoubleAttribute ViewerDelegateComponent::computeExtent(
    const std::string& location)
{
    return FnAttribute::DoubleAttribute();
}

FnViewerDelegateComponentPluginSuite_v2 ViewerDelegateComponent::createSuite(
    FnViewerDelegateComponentPluginHandle (*create)(
        FnViewerDelegateComponentHostHandle hostHandle))
{
    FnViewerDelegateComponentPluginSuite_v2 suite = {
        create,
        ::ViewerDelegateComponent_destroy,
        ::ViewerDelegateComponent_locationEvent,
        ::ViewerDelegateComponent_locationsSelected,
        ::ViewerDelegateComponent_setOption,
        ::ViewerDelegateComponent_getOption,
        ::ViewerDelegateComponent_setup,
        ::ViewerDelegateComponent_cleanup,
        ::ViewerDelegateComponent_isProcessing,
        ::ViewerDelegateComponent_getBounds,
        ::ViewerDelegateComponent_computeExtent,
    };

    return suite;
}

FnViewerDelegateComponentPluginHandle
ViewerDelegateComponent::newViewerDelegateComponentHandle(
    ViewerDelegateComponent* viewerDelegate)
{
    if (!viewerDelegate)
    {
        return 0x0;
    }

    FnViewerDelegateComponentPluginStruct* pluginHandle =
        new FnViewerDelegateComponentPluginStruct(viewerDelegate);

    return (FnViewerDelegateComponentPluginHandle) pluginHandle;
}

void ViewerDelegateComponent::setHostHandle(
    FnViewerDelegateComponentHostHandle hostHandle)
{
    m_hostHandle = hostHandle;
    m_hostSuite = const_cast<FnViewerDelegateComponentHostSuite_v2 *>(
        reinterpret_cast<const FnViewerDelegateComponentHostSuite_v2 *>(
            m_host->getSuite("ViewerDelegateComponentHost", 2)));
}

unsigned int ViewerDelegateComponent::_apiVersion = 2;
const char* ViewerDelegateComponent::_apiName =
    "ViewerDelegateComponentPlugin";


///////////////////////////
// ViewerDelegateComponentWrapper implementation
///////////////////////////

ViewerDelegateComponentWrapper::ViewerDelegateComponentWrapper(
    FnPluginHost* host,
    FnViewerDelegateComponentHostHandle hostHandle,
    FnViewerDelegateComponentPluginHandle pluginHandle,
    FnViewerDelegateComponentPluginSuite_v2* pluginSuite)
{
    if (!getHost()) { setHost(host); }

    m_hostHandle   = hostHandle;
    m_pluginHandle = pluginHandle;
    m_pluginSuite  = pluginSuite;
    m_hostSuite    = const_cast<FnViewerDelegateComponentHostSuite_v2 *>(
        reinterpret_cast<const FnViewerDelegateComponentHostSuite_v2 *>(
            m_host->getSuite("ViewerDelegateComponentHost", 2)));
}

ViewerDelegateComponentWrapper::~ViewerDelegateComponentWrapper()
{}

ViewerDelegateComponent* ViewerDelegateComponentWrapper::getPluginPointer()
{
    return m_pluginHandle->getViewerDelegateComponent();
}

void ViewerDelegateComponentWrapper::setOption(OptionIdGenerator::value_type optionId,
    FnAttribute::Attribute attr)
{
    // FIXME(DL): Non-standard ref-counting policy. TP 373492
    m_pluginSuite->setOption(m_pluginHandle, optionId,
        attr.getRetainedHandle());
}

FnAttribute::Attribute ViewerDelegateComponentWrapper::getOption(
    OptionIdGenerator::value_type optionId)
{
    FnAttributeHandle attrHandle =
        m_pluginSuite->getOption(m_pluginHandle, optionId);
    FnAttribute::Attribute attr =
        FnAttribute::Attribute::CreateAndSteal(attrHandle);
    return attr;
}

void ViewerDelegateComponentWrapper::setOption(const std::string& name, FnAttribute::Attribute attr)
{
    setOption(OptionIdGenerator::GenerateId(name.c_str()), attr);
}

FnAttribute::Attribute ViewerDelegateComponentWrapper::getOption(const std::string& name)
{
    return getOption(OptionIdGenerator::GenerateId(name.c_str()));
}

FnAttribute::DoubleAttribute ViewerDelegateComponentWrapper::getBounds(
    const std::string& location)
{
    const FnAttributeHandle attrHandle =
        m_pluginSuite->getBounds(m_pluginHandle, location.c_str());
    return FnAttribute::Attribute::CreateAndSteal(attrHandle);
}

FnAttribute::DoubleAttribute ViewerDelegateComponentWrapper::computeExtent(
    const std::string& location)
{
    const FnAttributeHandle attrHandle =
        m_pluginSuite->computeExtent(m_pluginHandle, location.c_str());
    return FnAttribute::Attribute::CreateAndSteal(attrHandle);
}

}
}
}
