// Copyright (c) 2016 The Foundry Visionmongers Ltd. All Rights Reserved.

#include <FnViewer/plugin/FnViewerDelegate.h>
#include <FnViewer/plugin/FnViewerDelegateComponent.h>
#include <FnViewer/plugin/FnViewport.h>
#include <FnViewer/plugin/FnOptionIdGenerator.h>
#include <FnViewer/plugin/FnViewerLocationEvent.h>

#include <FnPluginManager/FnPluginManager.h>
#include <FnAttribute/FnAttribute.h>
#include <FnAttribute/FnGroupBuilder.h>
#include <FnAttribute/FnAttribute.h>
#include <FnLogging/FnLogging.h>



#include <iostream>
#include <vector>


////////////////////////////////////////////////////
// C callbacks implementations for the plugin suite
////////////////////////////////////////////////////
void ViewerDelegate_destroy(FnViewerDelegatePluginHandle handle)
{
    delete handle;
}

void ViewerDelegate_locationEvent(
    FnViewerDelegatePluginHandle handle,
    FnViewerLocationEventStruct* eventHandle,
    uint8_t locationHandled)
{
    handle->getViewerDelegate()->locationEvent(
            Foundry::Katana::ViewerAPI::ViewerLocationEvent(*eventHandle),
            static_cast<bool>(locationHandled));
}

void ViewerDelegate_sourceLocationEvent(
    FnViewerDelegatePluginHandle handle,
    FnViewerLocationEventStruct* eventHandle)
{
    handle->getViewerDelegate()->sourceLocationEvent(
            Foundry::Katana::ViewerAPI::ViewerLocationEvent(*eventHandle));
}

void ViewerDelegate_locationsSelected(FnViewerDelegatePluginHandle handle,
    const char** locationPaths, int64_t numPaths)
{
    std::vector<std::string> locationPathsVec;
    locationPathsVec.reserve(numPaths);
    for(unsigned int i = 0 ; i < numPaths; ++i)
    {
        std::string locationPath(locationPaths[i]);
        locationPathsVec.push_back(locationPath);
    }

    handle->getViewerDelegate()->locationsSelected(locationPathsVec);
}

int ViewerDelegate_isProcessing(FnViewerDelegatePluginHandle handle)
{
    return static_cast<int>(handle->getViewerDelegate()->isProcessing());
}

void ViewerDelegate_setOption(FnViewerDelegatePluginHandle handle,
    uint64_t optionId, FnAttributeHandle attrHandle)
{
    // FIXME(DL): Non-standard ref-counting policy. TP 373492
    FnAttribute::Attribute attr =
        FnAttribute::Attribute::CreateAndSteal(attrHandle);

    handle->getViewerDelegate()->setOption(optionId, attr);
}

FnAttributeHandle ViewerDelegate_getOption(FnViewerDelegatePluginHandle handle,
    uint64_t optionId)
{
    return handle->getViewerDelegate()->getOption(optionId).getRetainedHandle();
}

void ViewerDelegate_freeze(FnViewerDelegatePluginHandle handle)
{
    handle->getViewerDelegate()->freeze();
}

void ViewerDelegate_thaw(FnViewerDelegatePluginHandle handle)
{
    handle->getViewerDelegate()->thaw();
}

void ViewerDelegate_setup(FnViewerDelegatePluginHandle handle)
{
    handle->getViewerDelegate()->setup();
}

void ViewerDelegate_cleanup(FnViewerDelegatePluginHandle handle)
{
    handle->getViewerDelegate()->cleanup();
}

FnAttributeHandle ViewerDelegate_getBounds(FnViewerDelegatePluginHandle handle,
    const char* location)
{
    return handle->getViewerDelegate()->getBounds(location).getRetainedHandle();
}

FnAttributeHandle ViewerDelegate_computeExtent(
    FnViewerDelegatePluginHandle handle, const char* location)
{
    const FnAttribute::DoubleAttribute extentAttr =
        handle->getViewerDelegate()->computeExtent(location);
    return extentAttr.getRetainedHandle();
}

void ViewerDelegate_setBoundingBoxesEnabled(
    FnViewerDelegatePluginHandle handle,
    uint8_t enabled)
{
    handle->getViewerDelegate()->setBoundingBoxesEnabled(
        static_cast<bool>(enabled));
}

void ViewerDelegate_setProxyGeometryEnabled(
    FnViewerDelegatePluginHandle handle,
    uint8_t enabled)
{
    handle->getViewerDelegate()->setProxyGeometryEnabled(
        static_cast<bool>(enabled));
}


///////////////////////////
// FnViewerDelegate classes implementation
///////////////////////////
namespace Foundry
{
namespace Katana
{
namespace ViewerAPI
{

///////////////////////////
// ViewerDelegatePluginBase implementation
///////////////////////////

ViewerDelegatePluginBase::ViewerDelegatePluginBase() :
    m_hostSuite(NULL),
    m_hostHandle(NULL)
{
}

ViewerDelegatePluginBase::~ViewerDelegatePluginBase()
{
}

void ViewerDelegatePluginBase::openLocation(const std::string& locationPath)
{
    const char* locationPathStr = locationPath.c_str();
    m_hostSuite->openLocations(m_hostHandle, &locationPathStr, 1);
}

void ViewerDelegatePluginBase::openLocations(
    const std::vector<std::string>& locationPaths)
{
    size_t numLocationPaths = locationPaths.size();
    if (numLocationPaths == 0)
    {
        return;
    }

    std::vector<const char*> locationPathsVec;
    locationPathsVec.reserve(numLocationPaths);
    for(size_t i = 0 ; i < numLocationPaths; ++i)
    {
        locationPathsVec[i] = locationPaths[i].c_str();
    }

    m_hostSuite->openLocations(m_hostHandle, locationPathsVec.data(),
        static_cast<uint64_t>(numLocationPaths));
}

void ViewerDelegatePluginBase::closeLocation(const std::string& locationPath)
{
    const char* locationPathStr = locationPath.c_str();
    m_hostSuite->closeLocations(m_hostHandle, &locationPathStr, 1);
}

void ViewerDelegatePluginBase::closeLocations(
    const std::vector<std::string>& locationPaths)
{
    size_t numLocationPaths = locationPaths.size();
    if (numLocationPaths == 0)
    {
        return;
    }

    std::vector<const char*> locationPathsVec;
    locationPathsVec.reserve(numLocationPaths);
    for(size_t i = 0 ; i < numLocationPaths; ++i)
    {
        locationPathsVec[i] = locationPaths[i].c_str();
    }

    m_hostSuite->closeLocations(m_hostHandle, locationPathsVec.data(),
        static_cast<uint64_t>(numLocationPaths));
}

void ViewerDelegatePluginBase::activateSourceLocation(
    const std::string& locationPath)
{
    const char* locationPathStr = locationPath.c_str();
    m_hostSuite->activateSourceLocations(m_hostHandle, &locationPathStr, 1);
}

void ViewerDelegatePluginBase::activateSourceLocations(
   const std::vector<std::string>& locations)
{
   size_t numPaths = locations.size();
   std::vector<const char*> locationPathsVec;
   locationPathsVec.reserve(numPaths);
   for (size_t i = 0; i < numPaths; ++i)
       locationPathsVec.push_back(locations[i].c_str());

   m_hostSuite->activateSourceLocations(m_hostHandle, locationPathsVec.data(),
       static_cast<int64_t>(numPaths));
}

void ViewerDelegatePluginBase::deactivateSourceLocation(
    const std::string& locationPath)
{
    const char* locationPathStr = locationPath.c_str();
    m_hostSuite->deactivateSourceLocations(m_hostHandle, &locationPathStr, 1);
}

void ViewerDelegatePluginBase::deactivateSourceLocations(
   const std::vector<std::string>& locations)
{
   size_t numPaths = locations.size();
   std::vector<const char*> locationPathsVec;
   locationPathsVec.reserve(numPaths);
   for (size_t i = 0; i < numPaths; ++i)
       locationPathsVec.push_back(locations[i].c_str());

   m_hostSuite->deactivateSourceLocations(m_hostHandle, locationPathsVec.data(),
       static_cast<int64_t>(numPaths));
}

FnAttribute::GroupAttribute ViewerDelegatePluginBase::getAttributes(
    const std::string& locationPath, FnViewerDelegateSceneView sceneView)
{
    FnAttributeHandle attrHandle =
        m_hostSuite->getAttributes(m_hostHandle, locationPath.c_str(),
            sceneView);

    FnAttribute::GroupAttribute attr =
        FnAttribute::Attribute::CreateAndSteal(attrHandle);

    return attr;
}

Matrix44d ViewerDelegatePluginBase::getLocalXform(
    const std::string& locationPath, bool& isAbsolute,
    FnViewerDelegateSceneView sceneView)
{
    uint8_t isAbsoluteInt;
    FnAttributeHandle attrHandle =
        m_hostSuite->getLocalXform(m_hostHandle, locationPath.c_str(),
                &isAbsoluteInt, sceneView);
    isAbsolute = static_cast<bool>(isAbsoluteInt);

    FnAttribute::DoubleAttribute attr =
        FnAttribute::Attribute::CreateAndSteal(attrHandle);

    if (!attr.isValid())
    {
        return Matrix44d(); // identity matrix
    }

    FnAttribute::DoubleConstVector m = attr.getNearestSample(0);
    return Matrix44d(m.data());
}

Matrix44d ViewerDelegatePluginBase::getWorldXform(
    const std::string& locationPath, FnViewerDelegateSceneView sceneView)
{
    FnAttributeHandle attrHandle =
        m_hostSuite->getWorldXform(m_hostHandle, locationPath.c_str(),
            sceneView);

    FnAttribute::DoubleAttribute attr =
        FnAttribute::Attribute::CreateAndSteal(attrHandle);

    if (!attr.isValid())
    {
        return Matrix44d(); // identity matrix
    }

    FnAttribute::DoubleConstVector m = attr.getNearestSample(0);
    return Matrix44d(m.data());
}

Matrix44d ViewerDelegatePluginBase::getParentXform(
    const std::string& locationPath, FnViewerDelegateSceneView sceneView)
{
    FnAttributeHandle attrHandle =
        m_hostSuite->getParentXform(m_hostHandle, locationPath.c_str(),
            sceneView);

    FnAttribute::DoubleAttribute attr =
        FnAttribute::Attribute::CreateAndSteal(attrHandle);

    if (!attr.isValid())
    {
        return Matrix44d(); // identity matrix
    }

    FnAttribute::DoubleConstVector m = attr.getNearestSample(0);
    return Matrix44d(m.data());
}

Matrix44d ViewerDelegatePluginBase::getPartialXform(
    const std::string& locationPath, const std::string& groupName,
    const std::string& componentName, bool includeComponent,
    FnViewerDelegateSceneView sceneView,
    bool includeParentXform,
    bool includeBeforeGroup,
    bool includeAfterGroup)
{
    FnAttributeHandle attrHandle =
        m_hostSuite->getPartialXform(m_hostHandle, locationPath.c_str(),
            groupName.c_str(), componentName.c_str(), includeComponent,
            sceneView, includeParentXform, includeBeforeGroup,
            includeAfterGroup);

    FnAttribute::DoubleAttribute attr =
        FnAttribute::Attribute::CreateAndSteal(attrHandle);

    if (!attr.isValid())
    {
        return Matrix44d(); // identity matrix
    }

    FnAttribute::DoubleConstVector m = attr.getNearestSample(0);
    return Matrix44d(m.data());
}

FnAttribute::GroupAttribute
    ViewerDelegatePluginBase::getCompatibleManipulatorsInfo(
        const std::vector<std::string>& locationPaths)
{
    if(locationPaths.empty())
        return FnAttribute::GroupAttribute();

    // Convert to a C-string vector
    std::vector<const char *> cstrLocationPaths;
    cstrLocationPaths.reserve(locationPaths.size());
    for(unsigned int i = 0; i < locationPaths.size(); ++i)
    {
        cstrLocationPaths.push_back(locationPaths[i].c_str());
    }

    // Get the manipulator data
    FnAttributeHandle attrHandle =
        m_hostSuite->getCompatibleManipulatorsInfo(m_hostHandle,
                                               &cstrLocationPaths.front(),
                                               cstrLocationPaths.size());

    FnAttribute::GroupAttribute attr =
        FnAttribute::Attribute::CreateAndSteal(attrHandle);

    return attr;
}

FnAttribute::Attribute ViewerDelegatePluginBase::callPythonCallback(
    const std::string& name, FnAttribute::Attribute message)
{
    FnAttributeHandle attrHandle = m_hostSuite->callPythonCallback(
        m_hostHandle, name.c_str(), message.getHandle());
    return FnAttribute::Attribute::CreateAndSteal(attrHandle);
}

void ViewerDelegatePluginBase::selectLocation(const std::string& locationPath)
{
    const char* locationPathStr = locationPath.c_str();
    m_hostSuite->selectLocation(m_hostHandle, locationPathStr);
}

void ViewerDelegatePluginBase::selectLocations(
    const std::vector<std::string>& locationPaths)
{
    size_t numPaths = locationPaths.size();
    std::vector<const char*> locationPathsVec;
    locationPathsVec.reserve(numPaths);
    for(size_t i = 0 ; i < numPaths; ++i)
    {
        locationPathsVec.push_back(locationPaths[i].c_str());
    }

    m_hostSuite->selectLocations(m_hostHandle, locationPathsVec.data(),
        static_cast<int64_t>(numPaths));
}

void ViewerDelegatePluginBase::getSelectedLocations(
    std::vector<std::string>& locationPaths)
{
    FnAttribute::StringAttribute attr =
        FnAttribute::Attribute::CreateAndSteal(
            m_hostSuite->getSelectedLocations(m_hostHandle));

    if (attr.isValid())
    {
        // Copy the contents of the string attr to the output vector
        FnAttribute::StringConstVector attrValues = attr.getNearestSample(0);
        locationPaths.reserve(attrValues.size());
        for (unsigned int i  = 0; i < attrValues.size(); ++i)
        {
            locationPaths.push_back(attrValues[i]);
        }
    }
}

ViewerDelegateComponentWrapperPtr ViewerDelegatePluginBase::addComponent(
    const std::string& pluginName, const std::string& name)
{
    FnViewerDelegateComponentHostHandle viewerDelegateComponentHostHandle;
    FnViewerDelegateComponentPluginHandle viewerDelegateComponentPluginHandle;
    FnViewerDelegateComponentPluginSuite_v2* viewerDelegateComponentPluginSuite;

    m_hostSuite->addComponent(m_hostHandle,
        pluginName.c_str(), name.c_str(),
        &viewerDelegateComponentHostHandle,
        &viewerDelegateComponentPluginHandle,
        &viewerDelegateComponentPluginSuite);

    if (!viewerDelegateComponentHostHandle ||
        !viewerDelegateComponentPluginHandle ||
        !viewerDelegateComponentPluginSuite)
    {
        Foundry::Katana::ViewerAPI::ViewerDelegateComponentWrapperPtr nullPtr;
        return nullPtr;
    }

    ViewerDelegateComponentWrapperPtr ptr(
        new ViewerDelegateComponentWrapper(
            getHost(),
            viewerDelegateComponentHostHandle,
            viewerDelegateComponentPluginHandle,
            viewerDelegateComponentPluginSuite));

    return ptr;
}

ViewerDelegateComponentWrapperPtr ViewerDelegatePluginBase::insertComponent(
    const std::string& pluginName, const std::string& name,
    unsigned int index)
{
    FnViewerDelegateComponentHostHandle viewerDelegateComponentHostHandle;
    FnViewerDelegateComponentPluginHandle viewerDelegateComponentPluginHandle;
    FnViewerDelegateComponentPluginSuite_v2* viewerDelegateComponentPluginSuite;

    m_hostSuite->insertComponent(m_hostHandle,
        pluginName.c_str(), name.c_str(), index,
        &viewerDelegateComponentHostHandle,
        &viewerDelegateComponentPluginHandle,
        &viewerDelegateComponentPluginSuite);

    if (!viewerDelegateComponentHostHandle ||
        !viewerDelegateComponentPluginHandle ||
        !viewerDelegateComponentPluginSuite)
    {
        return Foundry::Katana::ViewerAPI::ViewerDelegateComponentWrapperPtr();
    }

    ViewerDelegateComponentWrapperPtr ptr(
        new ViewerDelegateComponentWrapper(
            getHost(),
            viewerDelegateComponentHostHandle,
            viewerDelegateComponentPluginHandle,
            viewerDelegateComponentPluginSuite));

    return ptr;
}

ViewerDelegateComponentWrapperPtr ViewerDelegatePluginBase::getComponent(
    const std::string& name) const
{
    FnViewerDelegateComponentHostHandle viewerDelegateComponentHostHandle;
    FnViewerDelegateComponentPluginHandle viewerDelegateComponentPluginHandle;
    FnViewerDelegateComponentPluginSuite_v2* viewerDelegateComponentPluginSuite;

    m_hostSuite->getComponentByName(m_hostHandle, name.c_str(),
        &viewerDelegateComponentHostHandle,
        &viewerDelegateComponentPluginHandle,
        &viewerDelegateComponentPluginSuite);

    if (!viewerDelegateComponentHostHandle ||
        !viewerDelegateComponentPluginHandle ||
        !viewerDelegateComponentPluginSuite)
    {
        Foundry::Katana::ViewerAPI::ViewerDelegateComponentWrapperPtr nullPtr;
        return nullPtr;
    }

    ViewerDelegateComponentWrapperPtr ptr(
        new ViewerDelegateComponentWrapper(
            getHost(),
            viewerDelegateComponentHostHandle,
            viewerDelegateComponentPluginHandle,
            viewerDelegateComponentPluginSuite));

    return ptr;
}

ViewerDelegateComponentWrapperPtr ViewerDelegatePluginBase::getComponent(
    unsigned int index) const
{
    FnViewerDelegateComponentHostHandle viewerDelegateComponentHostHandle;
    FnViewerDelegateComponentPluginHandle viewerDelegateComponentPluginHandle;
    FnViewerDelegateComponentPluginSuite_v2* viewerDelegateComponentPluginSuite;

    m_hostSuite->getComponentByIndex(m_hostHandle, index,
        &viewerDelegateComponentHostHandle,
        &viewerDelegateComponentPluginHandle,
        &viewerDelegateComponentPluginSuite);

    if (!viewerDelegateComponentHostHandle ||
        !viewerDelegateComponentPluginHandle ||
        !viewerDelegateComponentPluginSuite)
    {
        Foundry::Katana::ViewerAPI::ViewerDelegateComponentWrapperPtr nullPtr;
        return nullPtr;
    }

    ViewerDelegateComponentWrapperPtr ptr(
        new ViewerDelegateComponentWrapper(
            getHost(),
            viewerDelegateComponentHostHandle,
            viewerDelegateComponentPluginHandle,
            viewerDelegateComponentPluginSuite));

    return ptr;
}

void ViewerDelegatePluginBase::removeComponent(const std::string& name)
{
    m_hostSuite->removeComponentByName(m_hostHandle, name.c_str());
}

void ViewerDelegatePluginBase::removeComponent(unsigned int index)
{
    m_hostSuite->removeComponentByIndex(m_hostHandle, index);
}

unsigned int ViewerDelegatePluginBase::getNumberOfComponents() const
{
    return m_hostSuite->getNumberOfComponents(m_hostHandle);
}

std::string ViewerDelegatePluginBase::getComponentName(unsigned int index)
{
    return std::string( m_hostSuite->getComponentName(m_hostHandle, index) );
}

int ViewerDelegatePluginBase::getComponentIndex(
    const std::string& name)
{
    return m_hostSuite->getComponentIndex(m_hostHandle, name.c_str());
}

Foundry::Katana::ViewerAPI::ViewportWrapperPtr
    ViewerDelegatePluginBase::getViewport(
        const std::string& name)
{
    FnViewportHostHandle viewportHostHandle;
    FnViewportPluginHandle viewportPluginHandle;
    FnViewportPluginSuite_v2* viewportPluginSuite;

    m_hostSuite->getViewportByName(m_hostHandle, name.c_str(),
        &viewportHostHandle, &viewportPluginHandle, &viewportPluginSuite);

    if (!viewportHostHandle || !viewportPluginHandle || !viewportPluginSuite)
    {
        Foundry::Katana::ViewerAPI::ViewportWrapperPtr nullPtr;
        return nullPtr;
    }

    ViewportWrapperPtr ptr(
        new ViewportWrapper(
            getHost(),
            viewportHostHandle,
            viewportPluginHandle,
            viewportPluginSuite));

    return ptr;
}

Foundry::Katana::ViewerAPI::ViewportWrapperPtr
    ViewerDelegatePluginBase::getViewport(
        unsigned int index)
{
    FnViewportHostHandle viewportHostHandle;
    FnViewportPluginHandle viewportPluginHandle;
    FnViewportPluginSuite_v2* viewportPluginSuite;

    m_hostSuite->getViewportByIndex(m_hostHandle, index,
        &viewportHostHandle, &viewportPluginHandle, &viewportPluginSuite);

    if (!viewportHostHandle || !viewportPluginHandle || !viewportPluginSuite)
    {
        Foundry::Katana::ViewerAPI::ViewportWrapperPtr nullPtr;
        return nullPtr;
    }

    ViewportWrapperPtr ptr(
        new ViewportWrapper(
            getHost(),
            viewportHostHandle,
            viewportPluginHandle,
            viewportPluginSuite));

    return ptr;
}

unsigned int ViewerDelegatePluginBase::getNumberOfViewports()
{
    return (unsigned int)m_hostSuite->getNumberOfViewports(m_hostHandle);
}

std::string ViewerDelegatePluginBase::getViewportName(unsigned int index)
{
    return m_hostSuite->getViewportName(m_hostHandle, index);
}

bool ViewerDelegatePluginBase::setManipulatedAttribute(
    const std::string& locationPath, const std::string& attrName,
    FnAttribute::Attribute valueAttr, bool isFinal)
{
    return m_hostSuite->setManipulatedAttribute(m_hostHandle,
        locationPath.c_str(), attrName.c_str(), valueAttr.getHandle(),
        isFinal);
}

FnAttribute::GroupAttribute ViewerDelegatePluginBase::getLiveAttributes(
    const std::string& locationPath, FnViewerDelegateSceneView sceneView)
{
    const FnAttributeHandle attrHandle =
        m_hostSuite->getLiveAttributes(m_hostHandle,
                                       locationPath.c_str(),
                                       sceneView);

    return FnAttribute::GroupAttribute::CreateAndSteal(attrHandle);
}

FnAttribute::GroupAttribute
ViewerDelegatePluginBase::getCookedAttributes(
    const std::string& locationPath, FnViewerDelegateSceneView sceneView)
{
    const FnAttributeHandle attrHandle =
        m_hostSuite->getCookedAttributes(m_hostHandle,
                                                locationPath.c_str(),
                                                sceneView);

    return FnAttribute::GroupAttribute::CreateAndSteal(attrHandle);
}

void ViewerDelegatePluginBase::openManipulationGroup(const std::string& locationPath)
{
    m_hostSuite->openManipulationGroup(m_hostHandle, locationPath.c_str());
}

void ViewerDelegatePluginBase::closeManipulationGroup(const std::string& locationPath)
{
    m_hostSuite->closeManipulationGroup(m_hostHandle, locationPath.c_str());
}

FnPlugStatus ViewerDelegatePluginBase::setHost(FnPluginHost* host)
{
    m_host = host;

    FnPlugStatus status = FnPluginStatusOK;

    status = FnPluginManager::PluginManager::setHost(host);
    if (status != FnPluginStatusOK)
        return status;

    // We don't verity the return value of OptionIdGenerator::setHost() as the
    // host suite is not guaranteed to be registered yet.
    OptionIdGenerator::setHost(host);

    status = FnLogging::setHost(host);
    if (status != FnPluginStatusOK)
        return status;

    status = FnAttribute::GroupBuilder::setHost(host);
    if (status != FnPluginStatusOK)
        return status;

    status = FnAttribute::Attribute::setHost(host);
    return status;
}

void ViewerDelegatePluginBase::setBoundingBoxesEnabled(bool enabled)
{
    m_hostSuite->setBoundingBoxesEnabled(
        m_hostHandle, static_cast<uint8_t>(enabled));
}

void ViewerDelegatePluginBase::setProxyGeometryEnabled(bool enabled)
{
    m_hostSuite->setProxyGeometryEnabled(
        m_hostHandle, static_cast<uint8_t>(enabled));
}

FnAttribute::StringAttribute ViewerDelegatePluginBase::getDescendantLocations(
    const std::string& location)
{
    FnAttributeHandle attrHandle =
        m_hostSuite->getDescendantLocations(m_hostHandle, location.c_str());
    return FnAttribute::StringAttribute::CreateAndSteal(attrHandle);
}

FnAttribute::DoubleAttribute ViewerDelegatePluginBase::computeMergedExtent(
    const std::vector<std::string>& locations,
    const std::string& excludeLocation)
{
    std::vector<const char*> locationsStr;
    locationsStr.reserve(locations.size());
    for (const auto& location : locations)
    {
        locationsStr.push_back(location.c_str());
    }

    const FnAttributeHandle attrHandle =
        m_hostSuite->computeMergedExtent(m_hostHandle, locationsStr.data(),
            locations.size(), excludeLocation.c_str());
    return FnAttribute::StringAttribute::CreateAndSteal(attrHandle);
}


FnPluginHost* ViewerDelegatePluginBase::getHost() { return m_host; }

FnPluginHost *ViewerDelegatePluginBase::m_host = 0x0;


///////////////////////////
// ViewerDelegate implementation
///////////////////////////

ViewerDelegate::ViewerDelegate()
{
}

ViewerDelegate::~ViewerDelegate()
{
}

FnAttribute::DoubleAttribute ViewerDelegate::getBounds(
    const std::string& location)
{
    return FnAttribute::DoubleAttribute();
}

FnAttribute::DoubleAttribute ViewerDelegate::computeExtent(
    const std::string& location)
{
    return FnAttribute::DoubleAttribute();
}

FnAttribute::Attribute ViewerDelegate::getOption(
    OptionIdGenerator::value_type optionId)
{
    return FnAttribute::Attribute();
}

void ViewerDelegate::setOption(const std::string& name, FnAttribute::Attribute attr)
{
    setOption(OptionIdGenerator::GenerateId(name.c_str()), attr);
}

FnAttribute::Attribute ViewerDelegate::getOption(const std::string& name)
{
    return getOption(OptionIdGenerator::GenerateId(name.c_str()));
}

FnViewerDelegatePluginSuite_v3 ViewerDelegate::createSuite(
    FnViewerDelegatePluginHandle (*create)(
        FnViewerDelegateHostHandle hostHandle))
{
    FnViewerDelegatePluginSuite_v3 suite = {
        create,
        ::ViewerDelegate_destroy,
        ::ViewerDelegate_locationEvent,
        ::ViewerDelegate_sourceLocationEvent,
        ::ViewerDelegate_locationsSelected,
        ::ViewerDelegate_setOption,
        ::ViewerDelegate_getOption,
        ::ViewerDelegate_freeze,
        ::ViewerDelegate_thaw,
        ::ViewerDelegate_setup,
        ::ViewerDelegate_cleanup,
        ::ViewerDelegate_isProcessing,
        ::ViewerDelegate_getBounds,
        ::ViewerDelegate_computeExtent,
    };

    return suite;
}

FnViewerDelegatePluginHandle ViewerDelegate::newViewerDelegateHandle(
    ViewerDelegate* viewerDelegate)
{
    if (!viewerDelegate)
    {
        return 0x0;
    }

    FnViewerDelegatePluginStruct* pluginHandle =
        new FnViewerDelegatePluginStruct(viewerDelegate);

    return (FnViewerDelegatePluginHandle) pluginHandle;
}

void ViewerDelegate::setHostHandle(FnViewerDelegateHostHandle hostHandle)
{
    m_hostHandle = hostHandle;
    m_hostSuite = const_cast<FnViewerDelegateHostSuite_v4 *>(
        reinterpret_cast<const FnViewerDelegateHostSuite_v4 *>(
            m_host->getSuite("ViewerDelegateHost",
                FnViewerDelegateHostSuite_version)));
}

unsigned int ViewerDelegate::_apiVersion = 3;
const char* ViewerDelegate::_apiName     = "ViewerDelegatePlugin";

///////////////////////////
// ViewerDelegateWrapper implementation
///////////////////////////

ViewerDelegateWrapper::ViewerDelegateWrapper(
    FnPluginHost* host,
    FnViewerDelegateHostHandle hostHandle,
    FnViewerDelegatePluginHandle pluginHandle,
    FnViewerDelegatePluginSuite_v3* pluginSuite)
{
    if (!getHost()) { setHost(host); }

    m_hostHandle = hostHandle;
    m_pluginHandle = pluginHandle;
    m_pluginSuite = pluginSuite;

    m_hostSuite = const_cast<FnViewerDelegateHostSuite_v4 *>(
        reinterpret_cast<const FnViewerDelegateHostSuite_v4 *>(
            m_host->getSuite("ViewerDelegateHost",
                FnViewerDelegateHostSuite_version)));
}

ViewerDelegateWrapper::~ViewerDelegateWrapper()
{}

void ViewerDelegateWrapper::setOption(OptionIdGenerator::value_type optionId,
    FnAttribute::Attribute attr)
{
    // FIXME(DL): Non-standard ref-counting policy. TP 373492
    FnAttributeHandle attrHandle = attr.getRetainedHandle();
    m_pluginSuite->setOption(m_pluginHandle, optionId, attrHandle);
}

FnAttribute::Attribute ViewerDelegateWrapper::getOption(
    OptionIdGenerator::value_type optionId)
{
    FnAttributeHandle attrHandle =
        m_pluginSuite->getOption(m_pluginHandle, optionId);
    FnAttribute::Attribute attr =
        FnAttribute::Attribute::CreateAndSteal(attrHandle);
    return attr;
}

void ViewerDelegateWrapper::setOption(const std::string& name, FnAttribute::Attribute attr)
{
    setOption(OptionIdGenerator::GenerateId(name.c_str()), attr);
}

FnAttribute::Attribute ViewerDelegateWrapper::getOption(const std::string& name)
{
    return getOption(OptionIdGenerator::GenerateId(name.c_str()));
}

ViewerDelegate* ViewerDelegateWrapper::getPluginPointer()
{
    return m_pluginHandle->getViewerDelegate();
}

}
}
}
