// Copyright (c) 2016 The Foundry Visionmongers Ltd. All Rights Reserved.

#include <FnViewer/plugin/FnManipulatorHandle.h>
#include <FnViewer/plugin/FnOptionIdGenerator.h>

#include <FnPluginManager/FnPluginManager.h>
#include <FnAttribute/FnGroupBuilder.h>
#include <FnLogging/FnLogging.h>

#include <iostream>

#ifndef FNKAT_UNUSED
    #define FNKAT_UNUSED(x) (void)x
#endif

////////////////////////////////////////////////////
// C callbacks implementations for the plugin suite
////////////////////////////////////////////////////
void ManipulatorHandle_destroy(FnManipulatorHandlePluginHandle handle)
{
    delete handle;
}

int ManipulatorHandle_event(FnManipulatorHandlePluginHandle handle,
                            FnAttributeHandle eventAttrHandle)
{
    // FIXME(DL): Non-standard ref-counting policy. TP 373492
    FnAttribute::GroupAttribute eventAttr =
        FnAttribute::Attribute::CreateAndSteal(eventAttrHandle);
    if (!eventAttr.isValid())
    {
        return 0;
    }

    Foundry::Katana::ViewerAPI::FnEventWrapper eventWrapper(eventAttr);
    return handle->getManipulatorHandle()->event(eventWrapper)? 1 : 0;
}

void ManipulatorHandle_cancelManipulation(
    FnManipulatorHandlePluginHandle handle)
{
    handle->getManipulatorHandle()->cancelManipulation();
}

void ManipulatorHandle_draw(FnManipulatorHandlePluginHandle handle)
{
    handle->getManipulatorHandle()->draw();
}

void ManipulatorHandle_pickerDraw(FnManipulatorHandlePluginHandle handle,
    int64_t pickerId)
{
    handle->getManipulatorHandle()->pickerDraw(pickerId);
}


void ManipulatorHandle_setOption(FnManipulatorHandlePluginHandle handle,
    uint64_t optionId, FnAttributeHandle attrHandle)
{
    // FIXME(DL): Non-standard ref-counting policy. TP 373492
    FnAttribute::Attribute attr =
        FnAttribute::Attribute::CreateAndSteal(attrHandle);

    handle->getManipulatorHandle()->setOption(optionId, attr);
}

FnAttributeHandle ManipulatorHandle_getOption(FnManipulatorHandlePluginHandle handle,
    uint64_t optionId)
{
    return handle->getManipulatorHandle()->getOption(optionId).getRetainedHandle();
}


///////////////////////////
// FnManipulatorHandle implementation
///////////////////////////
namespace Foundry
{
namespace Katana
{
namespace ViewerAPI
{

///////////////////////////
// ManipulatorHandlePluginBase implementation
///////////////////////////


ManipulatorHandlePluginBase::ManipulatorHandlePluginBase()
{}

ManipulatorHandlePluginBase::~ManipulatorHandlePluginBase()
{}


ManipulatorWrapperPtr ManipulatorHandlePluginBase::getManipulator()
{
    if (!m_manipulatorWrapper)
    {
        // Initialize the shared pointer to the ManipulatorWrapper on the
        // first time this is called. This can be cached here because Since a
        // Viewport always has the same ViewerDelegate throughout its lifetime.
        FnManipulatorHostHandle manipulatorHostHandle;
        FnManipulatorPluginHandle manipulatorPluginHandle;
        FnManipulatorPluginSuite_v1* manipulatorPluginSuite;

        m_hostSuite->getManipulator(m_hostHandle,
            &manipulatorHostHandle,
            &manipulatorPluginHandle,
            &manipulatorPluginSuite);

        ManipulatorWrapperPtr ptr(
            new ManipulatorWrapper(
                getHost(),
                manipulatorHostHandle,
                manipulatorPluginHandle,
                manipulatorPluginSuite));

        m_manipulatorWrapper = ptr;
    }

    return m_manipulatorWrapper;
}

Matrix44d ManipulatorHandlePluginBase::getXform()
{
    FnAttributeHandle attrHandle =
        m_hostSuite->getXform(m_hostHandle);

    FnAttribute::DoubleAttribute attr =
        FnAttribute::Attribute::CreateAndSteal(attrHandle);

    if (!attr.isValid())
    {
        return Matrix44d(); // identity matrix
    }

    FnAttribute::DoubleConstVector m = attr.getNearestSample(0);
    return Matrix44d(m.data());
}

void ManipulatorHandlePluginBase::setLocalXform(const Matrix44d& xform)
{
    FnAttribute::DoubleAttribute attr(xform.data, 16, 4);
    m_hostSuite->setLocalXform(m_hostHandle, attr.getHandle());
}

Matrix44d ManipulatorHandlePluginBase::getLocalXform()
{
    FnAttributeHandle attrHandle =
        m_hostSuite->getLocalXform(m_hostHandle);

    FnAttribute::DoubleAttribute attr =
        FnAttribute::Attribute::CreateAndSteal(attrHandle);

    if (!attr.isValid())
    {
        return Matrix44d(); // identity matrix
    }

    FnAttribute::DoubleConstVector m = attr.getNearestSample(0);
    return Matrix44d(m.data());
}

bool ManipulatorHandlePluginBase::isActive()
{
    return m_hostSuite->isActive(m_hostHandle);
}

bool ManipulatorHandlePluginBase::isHovered()
{
    return m_hostSuite->isHovered(m_hostHandle);
}

FnPlugStatus ManipulatorHandlePluginBase::setHost(FnPluginHost* host)
{
    m_host = host;

    FnPlugStatus status = FnPluginStatusOK;

    status = FnPluginManager::PluginManager::setHost(host);
    if (status != FnPluginStatusOK)
        return status;

    // We don't verity the return value of OptionIdGenerator::setHost() as the
    // host suite is not guaranteed to be registered yet.
    OptionIdGenerator::setHost(host);

    status = FnLogging::setHost(host);
    if (status != FnPluginStatusOK)
        return status;

    status = FnAttribute::GroupBuilder::setHost(host);
    if (status != FnPluginStatusOK)
        return status;

    status = FnAttribute::Attribute::setHost(host);
    return status;
}

FnPluginHost* ManipulatorHandlePluginBase::getHost() { return m_host; }

FnPluginHost *ManipulatorHandlePluginBase::m_host = 0x0;


///////////////////////////
// ManipulatorHandle implementation
///////////////////////////

ManipulatorHandle::ManipulatorHandle()
{}

ManipulatorHandle::~ManipulatorHandle()
{}

bool ManipulatorHandle::event(const FnEventWrapper& eventData)
{
    FNKAT_UNUSED(eventData);
    return false;
}

FnAttribute::Attribute ManipulatorHandle::getOption(
    OptionIdGenerator::value_type optionId)
{
    return FnAttribute::Attribute();
}

void ManipulatorHandle::setOption(const std::string& name, FnAttribute::Attribute attr)
{
    setOption(OptionIdGenerator::GenerateId(name.c_str()), attr);
}

FnAttribute::Attribute ManipulatorHandle::getOption(const std::string& name)
{
    return getOption(OptionIdGenerator::GenerateId(name.c_str()));
}

FnManipulatorHandlePluginSuite_v1 ManipulatorHandle::createSuite(
    FnManipulatorHandlePluginHandle (*create)(
        FnManipulatorHandleHostHandle hostHandle))
{
    FnManipulatorHandlePluginSuite_v1 suite = {
        create,
        ::ManipulatorHandle_destroy,
        ::ManipulatorHandle_event,
        ::ManipulatorHandle_draw,
        ::ManipulatorHandle_pickerDraw,
        ::ManipulatorHandle_setOption,
        ::ManipulatorHandle_getOption,
        ::ManipulatorHandle_cancelManipulation,
    };

    return suite;
}

FnManipulatorHandlePluginHandle ManipulatorHandle::newManipulatorHandleHandle(
    ManipulatorHandle* manipulatorHandle)
{
    if (!manipulatorHandle)
    {
        return 0x0;
    }

    FnManipulatorHandlePluginStruct* pluginHandle =
        new FnManipulatorHandlePluginStruct(manipulatorHandle);

    return (FnManipulatorHandlePluginHandle) pluginHandle;
}

void ManipulatorHandle::setHostHandle(FnManipulatorHandleHostHandle hostHandle)
{
    m_hostHandle = hostHandle;
    m_hostSuite = const_cast<FnManipulatorHandleHostSuite_v1 *>(
        reinterpret_cast<const FnManipulatorHandleHostSuite_v1 *>(
            m_host->getSuite("ManipulatorHandleHost", 1)));
}

unsigned int ManipulatorHandle::_apiVersion = 1;
const char* ManipulatorHandle::_apiName     = "ManipulatorHandlePlugin";


///////////////////////////
// ManipulatorHandleWrapper implementation
///////////////////////////

ManipulatorHandleWrapper::ManipulatorHandleWrapper(
    FnPluginHost* host,
    FnManipulatorHandleHostHandle hostHandle,
    FnManipulatorHandlePluginHandle pluginHandle,
    FnManipulatorHandlePluginSuite_v1* pluginSuite)
{
    if (!getHost()) { setHost(host); }

    m_hostHandle   = hostHandle;
    m_pluginHandle = pluginHandle;
    m_pluginSuite  = pluginSuite;
    m_hostSuite    = const_cast<FnManipulatorHandleHostSuite_v1 *>(
        reinterpret_cast<const FnManipulatorHandleHostSuite_v1 *>(
            m_host->getSuite("ManipulatorHandleHost", 1)));
}

ManipulatorHandleWrapper::~ManipulatorHandleWrapper()
{}

ManipulatorHandle* ManipulatorHandleWrapper::getPluginPointer()
{
    return m_pluginHandle->getManipulatorHandle();
}

void ManipulatorHandleWrapper::setActive(bool active)
{
    m_hostSuite->setActive(m_hostHandle, active);
}

void ManipulatorHandleWrapper::setHovered(bool hovered)
{
    m_hostSuite->setHovered(m_hostHandle, hovered);
}

void ManipulatorHandleWrapper::cancelManipulation()
{
    m_hostSuite->cancelManipulation(m_hostHandle);
}

void ManipulatorHandleWrapper::draw()
{
    m_hostSuite->draw(m_hostHandle);
}

bool ManipulatorHandleWrapper::event(FnEventWrapper eventData)
{
    // FIXME(DL): Non-standard ref-counting policy. TP 373492
    return m_hostSuite->event( m_hostHandle,
        eventData.getRetainedHandle() ) != 0;
}

void ManipulatorHandleWrapper::pickerDraw(int64_t pickerId)
{
    m_hostSuite->pickerDraw(m_hostHandle, pickerId);
}

void ManipulatorHandleWrapper::setOption(OptionIdGenerator::value_type optionId,
    FnAttribute::Attribute attr)
{
    // FIXME(DL): Non-standard ref-counting policy. TP 373492
    m_pluginSuite->setOption(m_pluginHandle, optionId,
        attr.getRetainedHandle());
}

FnAttribute::Attribute ManipulatorHandleWrapper::getOption(
    OptionIdGenerator::value_type optionId)
{
    FnAttributeHandle attrHandle =
        m_pluginSuite->getOption(m_pluginHandle, optionId);
    FnAttribute::Attribute attr =
        FnAttribute::Attribute::CreateAndSteal(attrHandle);
    return attr;
}

void ManipulatorHandleWrapper::setOption(const std::string& name, FnAttribute::Attribute attr)
{
    setOption(OptionIdGenerator::GenerateId(name.c_str()), attr);
}

FnAttribute::Attribute ManipulatorHandleWrapper::getOption(const std::string& name)
{
    return getOption(OptionIdGenerator::GenerateId(name.c_str()));
}

}
}
}
