// Copyright (c) 2016 The Foundry Visionmongers Ltd. All Rights Reserved.

#include <FnViewer/plugin/FnManipulator.h>
#include <FnViewer/plugin/FnManipulatorHandle.h>
#include <FnViewer/plugin/FnOptionIdGenerator.h>

#include <FnPluginManager/FnPluginManager.h>
#include <FnAttribute/FnGroupBuilder.h>

#include <FnLogging/FnLogging.h>

#include <iostream>

#ifndef _UNUSED
    #define _UNUSED(x) (void)x
#endif

////////////////////////////////////////////////////
// C callbacks implementations for the plugin suite
////////////////////////////////////////////////////
void Manipulator_destroy(FnManipulatorPluginHandle handle)
{
    delete handle;
}

void Manipulator_setup(FnManipulatorPluginHandle handle)
{
    handle->getManipulator()->setup();
}

int Manipulator_event(FnManipulatorPluginHandle handle,
                      FnAttributeHandle eventAttrHandle)
{
    FnAttribute::GroupAttribute eventAttr =
        FnAttribute::Attribute::CreateAndRetain(eventAttrHandle);
    if (!eventAttr.isValid())
    {
        return false;
    }

    Foundry::Katana::ViewerAPI::FnEventWrapper eventWrapper(eventAttr);

    return handle->getManipulator()->event(eventWrapper) ? 1 : 0;
}

void Manipulator_draw(FnManipulatorPluginHandle handle)
{
    handle->getManipulator()->draw();
}

void Manipulator_pickerDraw(FnManipulatorPluginHandle handle, int64_t pickerId)
{
    handle->getManipulator()->pickerDraw(pickerId);
}

void Manipulator_setOption(FnManipulatorPluginHandle handle,
    uint64_t optionId, FnAttributeHandle attrHandle)
{
    // FIXME(DL): Non-standard ref-counting policy. TP 373492
    FnAttribute::Attribute attr =
        FnAttribute::Attribute::CreateAndSteal(attrHandle);

    handle->getManipulator()->setOption(optionId, attr);
}

FnAttributeHandle Manipulator_getOption(FnManipulatorPluginHandle handle,
    uint64_t optionId)
{
    return handle->getManipulator()->getOption(optionId).getRetainedHandle();
}


///////////////////////////
// FnManipulator classes implementation
///////////////////////////
namespace Foundry
{
namespace Katana
{
namespace ViewerAPI
{

///////////////////////////
// ManipulatorPluginBase implementation
///////////////////////////

const FnManipulatorHostSuite_v2* ManipulatorPluginBase::_manipulatorSuite = 0x0;

ManipulatorPluginBase::ManipulatorPluginBase()
    : m_hostSuite(nullptr),
      m_hostHandle(nullptr)
{}

ManipulatorPluginBase::~ManipulatorPluginBase()
{}

// Tag names
const char ManipulatorPluginBase::kTagName[]             = "name";
const char ManipulatorPluginBase::kTagShortcut[]         = "shortcut";
const char ManipulatorPluginBase::kTagGroup[]            = "group";
const char ManipulatorPluginBase::kTagTechnology[]       = "technology";
const char ManipulatorPluginBase::kTagAlwaysAvailable[]  = "alwaysAvailable";
const char ManipulatorPluginBase::kTagExclusiveInGroup[] = "exclusiveInGroup";
const char ManipulatorPluginBase::kTagPriorityInGroup[]  = "priorityInGroup";

// Group Tag values
const char ManipulatorPluginBase::kTagGroup_TRANSFORM[] = "Transform";
const char ManipulatorPluginBase::kTagGroup_LIGHT[]     = "Light";

std::string ManipulatorPluginBase::getPluginName()
{
    return m_hostSuite->getPluginName(m_hostHandle);
}

Matrix44d ManipulatorPluginBase::getXform()
{
    FnAttribute::DoubleAttribute attr =
        FnAttribute::Attribute::CreateAndSteal(
            m_hostSuite->getXform(m_hostHandle));

    if (!attr.isValid())
    {
        return Matrix44d(); // identity matrix
    }

    FnAttribute::DoubleConstVector m = attr.getNearestSample(0);
    return Matrix44d(m.data());
}

void ManipulatorPluginBase::setXform(const Matrix44d& xformMatrix)
{
    FnAttribute::DoubleAttribute attr(&xformMatrix.data[0], 16, 4);
    return m_hostSuite->setXform(m_hostHandle, attr.getHandle());
}

Matrix44d ManipulatorPluginBase::calculateAveragePositionXform()
{

    FnAttribute::DoubleAttribute attr =
        FnAttribute::Attribute::CreateAndRetain(
            m_hostSuite->calculateAveragePositionXform(m_hostHandle));

    FnAttribute::DoubleConstVector m = attr.getNearestSample(0);
    return Matrix44d(m.data());
}

bool ManipulatorPluginBase::setValue(const std::string& locationPath,
    const std::string& attrName, FnAttribute::Attribute valueAttr,
    bool isFinal)
{
    return m_hostSuite->setValue(m_hostHandle, locationPath.c_str(),
        attrName.c_str(), valueAttr.getHandle(), isFinal);
}

void ManipulatorPluginBase::openManipulationGroup(const std::string& locationPath)
{
    m_hostSuite->openManipulationGroup(m_hostHandle, locationPath.c_str());
}

void ManipulatorPluginBase::closeManipulationGroup(const std::string& locationPath)
{
    m_hostSuite->closeManipulationGroup(m_hostHandle, locationPath.c_str());
}

FnAttribute::Attribute  ManipulatorPluginBase::getValue(
    const std::string& locationPath, const std::string& attrName)
{
    FnAttributeHandle attrHandle =
        m_hostSuite->getValue(m_hostHandle, locationPath.c_str(),
            attrName.c_str());

    return FnAttribute::Attribute::CreateAndRetain(attrHandle);
}

ViewportWrapperPtr ManipulatorPluginBase::getViewport()
{
    FnViewportHostHandle viewportHostHandle;
    FnViewportPluginHandle viewportPluginHandle;
    FnViewportPluginSuite_v2* viewportPluginSuite;

    m_hostSuite->getViewport(m_hostHandle,
        &viewportHostHandle, &viewportPluginHandle, &viewportPluginSuite);

    ViewportWrapperPtr ptr(
        new ViewportWrapper(
            getHost(),
            viewportHostHandle,
            viewportPluginHandle,
            viewportPluginSuite));

    return ptr;
}

void ManipulatorPluginBase::getLocationPaths(std::vector<std::string>& paths)
{
    paths.clear();

    FnAttributeHandle attrHandle = m_hostSuite->getLocationPaths(m_hostHandle);
    FnAttribute::StringAttribute attr =
        FnAttribute::Attribute::CreateAndSteal(attrHandle);

    if (!attr.isValid())
    {
        return;
    }

    FnAttribute::StringConstVector stringVec = attr.getNearestSample(0.0f);
    paths.assign(stringVec.begin(), stringVec.end());
}

ManipulatorHandleWrapperPtr ManipulatorPluginBase::addManipulatorHandle(
    const std::string& pluginName, const std::string& name)
{
    FnManipulatorHandleHostHandle manipulatorHandleHostHandle;
    FnManipulatorHandlePluginHandle manipulatorHandlePluginHandle;
    FnManipulatorHandlePluginSuite_v1* manipulatorHandlePluginSuite;

    m_hostSuite->addManipulatorHandle(m_hostHandle,
        pluginName.c_str(), name.c_str(),
        &manipulatorHandleHostHandle,
        &manipulatorHandlePluginHandle,
        &manipulatorHandlePluginSuite);

    if (!manipulatorHandleHostHandle ||
        !manipulatorHandlePluginHandle ||
        !manipulatorHandlePluginSuite)
    {
        Foundry::Katana::ViewerAPI::ManipulatorHandleWrapperPtr nullPtr;
        return nullPtr;
    }

    ManipulatorHandleWrapperPtr ptr(
        new ManipulatorHandleWrapper(
            getHost(),
            manipulatorHandleHostHandle,
            manipulatorHandlePluginHandle,
            manipulatorHandlePluginSuite));

    return ptr;
}

ManipulatorHandleWrapperPtr ManipulatorPluginBase::getManipulatorHandle(
    const std::string& name)
{
    FnManipulatorHandleHostHandle manipulatorHandleHostHandle;
    FnManipulatorHandlePluginHandle manipulatorHandlePluginHandle;
    FnManipulatorHandlePluginSuite_v1* manipulatorHandlePluginSuite;

    m_hostSuite->getManipulatorHandleByName(m_hostHandle, name.c_str(),
        &manipulatorHandleHostHandle,
        &manipulatorHandlePluginHandle,
        &manipulatorHandlePluginSuite);

    if (!manipulatorHandleHostHandle ||
        !manipulatorHandlePluginHandle ||
        !manipulatorHandlePluginSuite)
    {
        Foundry::Katana::ViewerAPI::ManipulatorHandleWrapperPtr nullPtr;
        return nullPtr;
    }

    ManipulatorHandleWrapperPtr ptr(
        new ManipulatorHandleWrapper(
            getHost(),
            manipulatorHandleHostHandle,
            manipulatorHandlePluginHandle,
            manipulatorHandlePluginSuite));

    return ptr;
}

ManipulatorHandleWrapperPtr ManipulatorPluginBase::getManipulatorHandle(
    unsigned int index)
{
    FnManipulatorHandleHostHandle manipulatorHandleHostHandle;
    FnManipulatorHandlePluginHandle manipulatorHandlePluginHandle;
    FnManipulatorHandlePluginSuite_v1* manipulatorHandlePluginSuite;

    m_hostSuite->getManipulatorHandleByIndex(m_hostHandle, index,
        &manipulatorHandleHostHandle,
        &manipulatorHandlePluginHandle,
        &manipulatorHandlePluginSuite);

    if (!manipulatorHandleHostHandle ||
        !manipulatorHandlePluginHandle ||
        !manipulatorHandlePluginSuite)
    {
        Foundry::Katana::ViewerAPI::ManipulatorHandleWrapperPtr nullPtr;
        return nullPtr;
    }

    ManipulatorHandleWrapperPtr ptr(
        new ManipulatorHandleWrapper(
            getHost(),
            manipulatorHandleHostHandle,
            manipulatorHandlePluginHandle,
            manipulatorHandlePluginSuite));

    return ptr;
}

void ManipulatorPluginBase::removeManipulatorHandle(const std::string& name)
{
    m_hostSuite->removeManipulatorHandleByName(m_hostHandle, name.c_str());
}

void ManipulatorPluginBase::removeManipulatorHandle(unsigned int index)
{
    m_hostSuite->removeManipulatorHandleByIndex(m_hostHandle, index);
}

unsigned int ManipulatorPluginBase::getNumberOfManipulatorHandles() const
{
    return m_hostSuite->getNumberOfManipulatorHandles(m_hostHandle);
}

std::string ManipulatorPluginBase::getManipulatorHandleName(unsigned int index)
{
    return std::string(m_hostSuite->getManipulatorHandleName(
        m_hostHandle, index));
}

FnAttribute::GroupAttribute
ManipulatorPluginBase::GetRegisteredManipulatorsInfo()
{
    const FnManipulatorHostSuite_v2* manipulatorSuite =
        reinterpret_cast<const FnManipulatorHostSuite_v2*>(
            m_host->getSuite("ManipulatorHost", 2));

    FnAttributeHandle attrHandle =
        manipulatorSuite->GetRegisteredManipulatorsInfo();

    return FnAttribute::Attribute::CreateAndRetain(attrHandle);
}

bool ManipulatorPluginBase::isInteractive() const
{
    return m_hostSuite->isInteractive(m_hostHandle);
}

FnPlugStatus ManipulatorPluginBase::setHost(FnPluginHost* host)
{
    m_host = host;

    // We don't verify that _manipulatorSuite is non-NULL, as the host suite
    // isn't guaranteed to be registered yet.
    _manipulatorSuite = reinterpret_cast<const FnManipulatorHostSuite_v2*>(
        host->getSuite("ManipulatorHost", 2));

    FnPlugStatus status = FnPluginStatusOK;

    status = FnPluginManager::PluginManager::setHost(host);
    if (status != FnPluginStatusOK)
        return status;

    // We don't verity the return value of OptionIdGenerator::setHost() as the
    // host suite is not guaranteed to be registered yet.
    OptionIdGenerator::setHost(host);

    status = FnLogging::setHost(host);
    if (status != FnPluginStatusOK)
        return status;

    status = FnAttribute::GroupBuilder::setHost(host);
    if (status != FnPluginStatusOK)
        return status;

    status = FnAttribute::Attribute::setHost(host);
    return status;
}

FnPluginHost* ManipulatorPluginBase::getHost() { return m_host; }

FnPluginHost *ManipulatorPluginBase::m_host = 0x0;


///////////////////////////
// Manipulator implementation
///////////////////////////

Manipulator::Manipulator()
{}

Manipulator::~Manipulator()
{}

bool Manipulator::event(const FnEventWrapper& eventData)
{
    for (unsigned int i = 0; i < getNumberOfManipulatorHandles(); ++i)
    {
        ManipulatorHandleWrapperPtr manipHandle = getManipulatorHandle(i);
        manipHandle->event(eventData);
    }

    return false;
}

void Manipulator::draw()
{
    for (unsigned int i = 0; i < getNumberOfManipulatorHandles(); ++i)
    {
        ManipulatorHandleWrapperPtr manipHandle = getManipulatorHandle(i);
        manipHandle->draw();
    }
}

void Manipulator::pickerDraw(int64_t pickerId)
{
    for (unsigned int i = 0; i < getNumberOfManipulatorHandles(); ++i)
    {
        ManipulatorHandleWrapperPtr manipHandle = getManipulatorHandle(i);
        manipHandle->pickerDraw(pickerId + i);
    }
}

FnAttribute::Attribute Manipulator::getOption(
    OptionIdGenerator::value_type optionId)
{
    return FnAttribute::Attribute();
}

void Manipulator::setOption(const std::string& name, FnAttribute::Attribute attr)
{
    setOption(OptionIdGenerator::GenerateId(name.c_str()), attr);
}

FnAttribute::Attribute Manipulator::getOption(const std::string& name)
{
    return getOption(OptionIdGenerator::GenerateId(name.c_str()));
}

FnManipulatorPluginSuite_v1 Manipulator::createSuite(
    FnManipulatorPluginHandle (*create)(FnManipulatorHostHandle hostHandle),
    int (*matches)(FnAttributeHandle locationAttributes),
    FnAttributeHandle (*getTags)() )
{
    FnManipulatorPluginSuite_v1 manipulatorSuite = {
        create,
        ::Manipulator_destroy,
        matches,
        getTags,
        ::Manipulator_setup,
        ::Manipulator_event,
        ::Manipulator_draw,
        ::Manipulator_pickerDraw,
        ::Manipulator_setOption,
        ::Manipulator_getOption,
    };

    return manipulatorSuite;
}

FnManipulatorPluginHandle Manipulator::newManipulatorPluginHandle(
    Manipulator* manipulatorHandle)
{
    if (!manipulatorHandle)
    {
        return 0x0;
    }

    FnManipulatorPluginStruct* pluginHandle =
        new FnManipulatorPluginStruct(manipulatorHandle);

    return (FnManipulatorPluginHandle) pluginHandle;
}

void Manipulator::setHostHandle(FnManipulatorHostHandle hostHandle)
{
    m_hostHandle = hostHandle;
    m_hostSuite = const_cast<FnManipulatorHostSuite_v2 *>(
        reinterpret_cast<const FnManipulatorHostSuite_v2 *>(
            m_host->getSuite("ManipulatorHost", 2)));
}

unsigned int Manipulator::_apiVersion = 2;
const char* Manipulator::_apiName     = "ManipulatorPlugin";

///////////////////////////
// ManipulatorWrapper implementation
///////////////////////////

ManipulatorWrapper::ManipulatorWrapper(
    FnPluginHost* host,
    FnManipulatorHostHandle hostHandle,
    FnManipulatorPluginHandle pluginHandle,
    FnManipulatorPluginSuite_v1* pluginSuite)
{
    if (!getHost()) { setHost(host); }

    m_hostHandle = hostHandle;
    m_pluginHandle = pluginHandle;
    m_pluginSuite = pluginSuite;

    m_hostSuite = const_cast<FnManipulatorHostSuite_v2 *>(
        reinterpret_cast<const FnManipulatorHostSuite_v2 *>(
            m_host->getSuite("ManipulatorHost", 2)));
}

ManipulatorWrapper::~ManipulatorWrapper()
{}

bool ManipulatorWrapper::matches(
    const FnAttribute::GroupAttribute& locationAttrs)
{
    return m_pluginSuite->matches(locationAttrs.getHandle());
}

FnAttribute::GroupAttribute ManipulatorWrapper::getTags()
{
    return FnAttribute::Attribute::CreateAndSteal(m_pluginSuite->getTags());
}

Manipulator* ManipulatorWrapper::getPluginPointer()
{
    return m_pluginHandle->getManipulator();
}

void ManipulatorWrapper::draw()
{
    m_hostSuite->draw(m_hostHandle);
}


bool ManipulatorWrapper::event(FnEventWrapper eventData)
{
    return m_hostSuite->event( m_hostHandle,
        eventData.getHandle() ) != 0;
}


void ManipulatorWrapper::pickerDraw(int64_t pickerId)
{
    m_hostSuite->pickerDraw(m_hostHandle, pickerId);
}

void ManipulatorWrapper::setOption(OptionIdGenerator::value_type optionId,
    FnAttribute::Attribute attr)
{
    // FIXME(DL): Non-standard ref-counting policy. TP 373492
    m_pluginSuite->setOption(m_pluginHandle, optionId,
        attr.getRetainedHandle());
}

FnAttribute::Attribute ManipulatorWrapper::getOption(
    OptionIdGenerator::value_type optionId)
{
    FnAttributeHandle attrHandle =
        m_pluginSuite->getOption(m_pluginHandle, optionId);
    FnAttribute::Attribute attr =
        FnAttribute::Attribute::CreateAndSteal(attrHandle);
    return attr;
}

void ManipulatorWrapper::setOption(const std::string& name, FnAttribute::Attribute attr)
{
    setOption(OptionIdGenerator::GenerateId(name.c_str()), attr);
}

FnAttribute::Attribute ManipulatorWrapper::getOption(const std::string& name)
{
    return getOption(OptionIdGenerator::GenerateId(name.c_str()));
}

}
}
}
