// Copyright (c) 2013 The Foundry Visionmongers Ltd. All Rights Reserved.

#include "FnRender/plugin/RenderBase.h"
#include "FnAttribute/FnGroupBuilder.h"

#include "FnAsset/FnDefaultAssetPlugin.h"
#include "FnAsset/FnDefaultFileSequencePlugin.h"

#include <FnPluginManager/FnPluginManager.h>
#include <FnLogging/FnLogging.h>

#include <FnPlatform/internal/UniquePtr.h>

#include <cstdlib>
#include <memory>

struct FnRenderPluginStruct
{
public:
    explicit FnRenderPluginStruct(FnKat::Render::RenderBase* renderBase)
        : _renderBase(renderBase)
    {
    }
    ~FnRenderPluginStruct() {}

    FnKat::Render::RenderBase& getRenderPlugin() { return *_renderBase; }

private:
    FnPlatform::internal::UniquePtr<FnKat::Render::RenderBase>::type
        _renderBase;
};


////////////////////////////////////////////////////
// C callbacks implementations for the plugin suite
////////////////////////////////////////////////////
void destroy(FnRenderPluginHandle renderPluginHandle)
{
    delete renderPluginHandle;
}

void setRootIterator(FnRenderPluginHandle renderPluginHandle,
                     FnSgIteratorHandle rootIteratorHandle)
{
    renderPluginHandle->getRenderPlugin().setRootIterator(
        FNSCENEGRAPHITERATOR_NAMESPACE::FnScenegraphIterator(
            rootIteratorHandle));
}

int setup(FnRenderPluginHandle renderPluginHandle)
{
    return renderPluginHandle->getRenderPlugin().setup();
}

int cleanup(FnRenderPluginHandle renderPluginHandle)
{
    return renderPluginHandle->getRenderPlugin().cleanup();
}

int start(FnRenderPluginHandle renderPluginHandle)
{
    return renderPluginHandle->getRenderPlugin().start();
}

int pause(FnRenderPluginHandle renderPluginHandle)
{
    return renderPluginHandle->getRenderPlugin().pause();
}

int resume(FnRenderPluginHandle renderPluginHandle)
{
    return renderPluginHandle->getRenderPlugin().resume();
}

int stop(FnRenderPluginHandle renderPluginHandle)
{
    return renderPluginHandle->getRenderPlugin().stop();
}

int startLiveEditing(FnRenderPluginHandle renderPluginHandle)
{
    return renderPluginHandle->getRenderPlugin().startLiveEditing();
}

int stopLiveEditing(FnRenderPluginHandle renderPluginHandle)
{
    return renderPluginHandle->getRenderPlugin().stopLiveEditing();
}

int processControlCommand(FnRenderPluginHandle renderPluginHandle,
                          const char* command)
{
    return renderPluginHandle->getRenderPlugin().processControlCommand(
        command);
}

int queueDataUpdates(FnRenderPluginHandle renderPluginHandle,
                     FnAttributeHandle commandAttributeHandle)
{
    return renderPluginHandle->getRenderPlugin().queueDataUpdates(
        FnAttribute::Attribute::CreateAndRetain(commandAttributeHandle));
}

bool hasPendingDataUpdates(FnRenderPluginHandle renderPluginHandle)
{
    return renderPluginHandle->getRenderPlugin().hasPendingDataUpdates();
}

int applyPendingDataUpdates(FnRenderPluginHandle renderPluginHandle)
{
    return renderPluginHandle->getRenderPlugin().applyPendingDataUpdates();
}

FnAttributeHandle configureDiskRenderOutputProcess(
    FnRenderPluginHandle renderPluginHandle, const char* outputName,
    const char* outputPath, const char* renderMethodName,
    const float* frameTime)
{
    return renderPluginHandle->getRenderPlugin()\
        ._configureDiskRenderOutputProcess(outputName, outputPath,
                                           renderMethodName, *frameTime)\
        .getRetainedHandle();
}


///////////////////////////
// FnRender implementation
///////////////////////////
namespace Foundry
{
namespace Katana
{
namespace Render
{

std::string RenderBase::getRenderMethodName() const
{
    return findArgument("renderMethodName");
}

std::string RenderBase::findArgument(const std::string& argumentName,
                                     const std::string& defaultValue) const
{
    return FnAttribute::StringAttribute(
        _arguments.getChildByName(argumentName)).getValue(defaultValue, false);
}

bool RenderBase::applyRenderThreadsOverride(int& numberOfRenderThreads) const
{
    std::string threadArg = findArgument("threads");
    if(threadArg == "")
        return false;

    numberOfRenderThreads = atoi(threadArg.c_str());
    return true;
}

float RenderBase::getRenderTime() const
{
    FnScenegraphIterator rootIterator = getRootIterator();
    if (rootIterator.isValid())
    {
        FnAttribute::FloatAttribute frameNumberAttr = rootIterator.getAttribute(
        "renderSettings.currentTime");
        if (frameNumberAttr.isValid())
        {
            return frameNumberAttr.getValue();
        }
    }

    // if we can't get a valid value from the scenegraph, use the supplied
    // command-line argumnet instead.
    std::string renderTimeArgument = findArgument("renderTime");
    if (renderTimeArgument != "")
    {
        try
        {
            float converted = std::stof(renderTimeArgument);
            return converted;
        }
        catch (const std::invalid_argument & error)
        {
            return 0.0f;
        }
    }
    return 0.0f;
}

bool RenderBase::useRenderPassID() const
{
    std::string useID = findArgument("useID");
    if(useID == "True")
        return true;

    return false;
}

bool RenderBase::isExpandProceduralActive(bool defaultValue) const
{
    std::string expandProcedural = findArgument(
        "expandProcedural", (defaultValue ? "True" : "False"));
    if(expandProcedural == "True")
        return true;
    return false;
}

std::string RenderBase::getEnvironmentVariable(const std::string& name) const
{
#if !defined(_WIN32)
    char const* env = ::getenv(name.c_str());
    if(env == NULL)
#else
    size_t returnValue = 0;
    char env[4096];
    ::getenv_s(&returnValue, env, name.c_str());
    if (returnValue == 0)
#endif  // _WIN32
        return "";
    return std::string(env);
}

FnAttribute::Attribute RenderBase::_configureDiskRenderOutputProcess(
    const std::string& outputName, const std::string& outputPath,
    const std::string& renderMethodName, const float& frameTime) const
{
    DiskRenderOutputProcess diskRenderOutputProcess;
    configureDiskRenderOutputProcess(diskRenderOutputProcess, outputName,
                                     outputPath, renderMethodName, frameTime);
    return diskRenderOutputProcess.buildRenderOutputAttribute();
}

FnRenderPluginHandle RenderBase::newRenderHandle(RenderBase* renderBase)
{
    if (!renderBase)
    {
        return 0x0;
    }

    FnRenderPluginHandle renderPluginHandle = new FnRenderPluginStruct(
        renderBase);
    return renderPluginHandle;
}

FnPlugStatus RenderBase::setHost(FnPluginHost* host)
{
    _host = host;
    FnPluginManager::PluginManager::setHost(host);
    RenderOutputUtils::setHost(host);
    FnScenegraphIterator::setHost(host);
    FnAttribute::GroupBuilder::setHost(host);
    FnAsset::DefaultAssetPlugin::setHost(host);
    FnAsset::DefaultFileSequencePlugin::setHost(host);
    FnLogging::setHost(host);
    return FnAttribute::Attribute::setHost(host);
}

FnPluginHost* RenderBase::getHost() { return _host; }

FnRenderPluginSuite RenderBase::createSuite(
    FnRenderPluginHandle (*create)(FnSgIteratorHandle, FnAttributeHandle))
{
    FnRenderPluginSuite renderPluginSuite = {
        create,
        ::destroy,
        ::start,
        ::pause,
        ::resume,
        ::stop,
        ::startLiveEditing,
        ::stopLiveEditing,
        ::processControlCommand,
        ::queueDataUpdates,
        ::hasPendingDataUpdates,
        ::applyPendingDataUpdates,
        ::configureDiskRenderOutputProcess,
        ::setRootIterator,
        ::setup,
        ::cleanup
    };

    return renderPluginSuite;
}

unsigned int RenderBase::_apiVersion = FnRenderPluginSuite_version;
const char* RenderBase::_apiName     = "RenderPlugin";
FnPluginHost *RenderBase::_host      = 0x0;

}
}
}
