#include "FnGeolib/util/Path.h"

#include <algorithm>
#include <cctype>
#include <cstring>
#include <iostream>
#include <sstream>
#include <stdexcept>

#include "FnGeolibServices/suite/FnGeolibCookInterfaceUtilsSuite.h"
#include "FnPluginManager/FnPluginManager.h"
#include "pystring/pystring.h"

FNGEOLIBUTIL_NAMESPACE_ENTER
{

namespace Path
{
static std::string StripTrailingSlash(FnPlatform::StringView locationPath)
{
    if (locationPath.empty()) return std::string(locationPath);
    if (locationPath[locationPath.size() - 1] == '/')
    {
        return std::string(locationPath.substr(0, locationPath.size() - 1));
    }
    else
    {
        return std::string(locationPath);
    }
}

std::string NormalizeAbsPath(const std::string & locationPath)
{
    return StripTrailingSlash(
        pystring::os::path::normpath_posix(locationPath));
}

std::string GetLocationParent(FnPlatform::StringView locationPath)
{
    size_t index = locationPath.find_last_of("/");
    if(index == std::string::npos) return "";
    return std::string(locationPath.substr(0, index));

    /*
    std::vector<std::string> result;
    pystring::rsplit(locationPath, result, "/", 1);
    if (result.size() <= 1) return ""; // workaround for pystring bug
    std::string parent = result[0];
    */
}

std::string GetLeafName(FnPlatform::StringView locationPath)
{
    size_t index = locationPath.find_last_of("/");
    if(index == std::string::npos) return "";
    return std::string(locationPath.substr(index + 1, locationPath.size() - 1));

    /*
    std::vector<std::string> result;
    pystring::rsplit(locationPath, result, "/", 1);
    // workaround for pystring bug
    if (result.size() == 0) return "";
    if (result.size() == 1) return pystring::slice(locationPath, 1);
    return result[result.size()-1];
    */
}

void GetLeafAndParent(std::string& parent,
                      std::string& leaf,
                      FnPlatform::StringView locationPath)
{
    size_t index = locationPath.find_last_of("/");
    if (index == FnPlatform::StringView::npos)
    {
        parent = "";
        leaf = "";
    }
    else
    {
        parent.assign(locationPath.begin(), locationPath.begin() + index);
        leaf.assign(locationPath.begin() + index + 1, locationPath.end());
    }
}

// TODO: This can be hugely optimized

void GetLocationStack(std::vector<std::string> & returnStack,
    const std::string & locationPath, const std::string & rootPath)
{
    returnStack.clear();

    if (rootPath.empty())
    {
        std::vector<std::string> parts;
        pystring::split(pystring::strip(locationPath, "/"), parts, "/");
        for (unsigned int i=0; i<parts.size(); ++i)
        {
            returnStack.push_back(
                "/" + pystring::join("/", std::vector<std::string>(parts.begin(), parts.begin()+i+1)));
        }
    }
    else
    {
        if (locationPath == rootPath)
        {
            returnStack.push_back(locationPath);
        }
        else
        {
            returnStack.push_back(rootPath);
            std::string normRootPath = "/"+pystring::strip(rootPath, "/"); //ensure normalized
            if (Path::IsAncestorOrEqual(normRootPath, locationPath))
            {
                std::string rel = pystring::slice(locationPath, (int)normRootPath.size());
                std::vector<std::string> parts;
                pystring::split(pystring::strip(rel, "/"), parts, "/");
                for (unsigned int i=0; i<parts.size(); ++i)
                {
                    returnStack.push_back(
                        normRootPath + "/" + pystring::join("/", std::vector<std::string>(parts.begin(), parts.begin()+i+1)));
                }
            }
        }
    }
}

bool IsAncestorOrEqual(FnPlatform::StringView locA, FnPlatform::StringView locB)
{
    size_t n1 = locA.size(), n2 = locB.size();

    while (n1 > 1 && locA[n1 - 1] == '/')
        --n1;

    while (n2 > 1 && locB[n2 - 1] == '/')
        --n2;

    return (n1 || !n2) && n1 <= n2 && locA.compare(0, n1, locB, 0, n1) == 0 &&
           (n1 == n2 || locB[n1] == '/' || locA[n1 - 1] == '/');
}

bool IsAncestor(FnPlatform::StringView locA, FnPlatform::StringView locB)
{
    size_t n1 = locA.size(), n2 = locB.size();

    while (n1 > 1 && locA[n1 - 1] == '/')
        --n1;

    while (n2 > 1 && locB[n2 - 1] == '/')
        --n2;

    return n1 && n1 < n2 && locA.compare(0, n1, locB, 0, n1) == 0 &&
           (locB[n1] == '/' || locA[n1 - 1] == '/');
}

std::string Join(FnPlatform::StringView locA, FnPlatform::StringView locB)
{
    return StripTrailingSlash(locA).append("/").append(locB.begin(),
                                                       locB.end());
}

bool IsRelativePath(const std::string& path)
{
    return path.empty() || (path[0] != '/');
}

namespace
{
/// Makes a number of assumptions to reduce processing time of pystring::split.
void quickSplit(FnPlatform::StringView str,
                std::vector<FnPlatform::StringView>& components,
                char sep)
{
    FnPlatform::StringView::size_type i = 0, j = 0, length = str.size();

    while (i + 1 <= length)
    {
        if (str[i] == sep)
        {
            components.emplace_back(str.substr(j, i - j));
            i = j = i + 1;
        }
        else
        {
            ++i;
        }
    }
    components.emplace_back(str.substr(j, length - j));
}
}  // namespace

std::string RelativeToAbsPath(const std::string& rootPath,
                              const std::string& path)
{
    static constexpr char kDot[] = ".";
    static constexpr char kParent[] = "..";

    // StringView views of the above.
    static const FnPlatform::StringView kDotView(kDot);
    static const FnPlatform::StringView kParentView(kParent);

    if (!IsRelativePath(path))
    {
        return path;
    }

    // Join the two paths...
    // Note (DH): This implementation is simply paraphrased and slightly
    // optimised from the pystring::join_posix
    const std::string unNormalisedPathStr = [&]() -> std::string {
        if (rootPath.empty() || rootPath.back() == '/')
        {
            return rootPath + path;
        }
        else
        {
            return rootPath + '/' + path;
        }
    }();

    // Optimised version of pystring::..::normpath_posix()
    FnPlatform::StringView unNormalisedPath(unNormalisedPathStr);
    if (unNormalisedPath.empty())
        return kDot;

    int numberOfInitialSlashes = unNormalisedPath[0] == '/' ? 1 : 0;

    // POSIX allows one or two initial slashes, but treats three or more
    // as single slash.
    if (numberOfInitialSlashes && unNormalisedPath.size() > 3 &&
        unNormalisedPath[1] == '/' && unNormalisedPath[2] == '/')
    {
        numberOfInitialSlashes = 2;
    }

    // Tokenize on '/' character
    std::vector<FnPlatform::StringView> components;
    components.reserve(32);
    quickSplit(unNormalisedPath, components, '/');

    std::vector<FnPlatform::StringView> newComponents;
    // At least as big as the original path
    newComponents.reserve(components.size());

    for (const auto& component : components)
    {
        // Skip past empty or '.', as they don't move us up or down the path
        if (component.empty() || component == kDotView)
        {
            continue;
        }

        if (component != kParentView)
        {
            newComponents.push_back(component);
        }
        else if (numberOfInitialSlashes == 0 && newComponents.empty())
        {
            newComponents.push_back(component);
        }
        else if (!newComponents.empty() &&
                 newComponents[newComponents.size() - 1] == "..")
        {
            newComponents.push_back(component);
        }
        else if (!newComponents.empty())
        {
            newComponents.pop_back();
        }
    }

    // Join all the strings together.
    // First determine string length.
    size_t pathLength =
        newComponents.size() + static_cast<size_t>(numberOfInitialSlashes);
    for (const auto& component : newComponents)
        pathLength += component.length();

    std::string result;
    result.reserve(pathLength);

    // First fill up the numberOfInitialSlashes
    for (auto x = 0; x < numberOfInitialSlashes - 1; ++x)
    {
        result.push_back('/');
    }

    // Then join the remaining strings.
    for (const auto& component : newComponents)
    {
        result.push_back('/');
        // Copy the characters in our return string.
        for (size_t j = 0; j < component.length(); ++j)
            result.push_back(component[j]);
    }

    return result;
}

// TODO: This can be hugely optimized

std::string NormalizedRelativePath(const std::string & rootpath_,
    const std::string & path)
{
    std::string rootpath = StripTrailingSlash(rootpath_);
    std::string abspath = pystring::rstrip(RelativeToAbsPath(rootpath, path));

    if(!IsAncestorOrEqual(rootpath, abspath))
    {
        std::ostringstream err;
        err << "The rootpath " << rootpath_;
        err << " is not an ancestor or equal to " << abspath;
        throw std::runtime_error(err.str().c_str());
    }

    std::string result = pystring::replace(abspath, rootpath, "", 1);
    return pystring::lstrip(result, "/");
}

std::string RelativePath(const std::string & rootPath,
        const std::string & path)
{
    std::string absRootPath = NormalizeAbsPath(rootPath);
    std::string absPath = NormalizeAbsPath(path);

    std::vector<std::string> rootPathSplit;
    std::vector<std::string> pathSplit;
    pystring::split(absRootPath, rootPathSplit, "/");
    pystring::split(absPath, pathSplit, "/");

    // find common prefix
    int commonIndex = -1;
    for (std::vector<std::string>::const_iterator
        I=pathSplit.begin(), E=pathSplit.end(),
        rI=rootPathSplit.begin(), rE=rootPathSplit.end();
        I!=E && rI!=rE;
        ++I, ++rI)
    {
        if (*I == *rI)
        {
            ++commonIndex;
        }
        else
        {
            break;
        }
    }

    // build relative path
    std::vector<std::string> relPathSplit;
    relPathSplit.reserve(((int)rootPathSplit.size()-commonIndex-1) +
                         ((int)pathSplit.size()-commonIndex+1));
    for (int i=0; i<(int)rootPathSplit.size()-commonIndex-1; ++i)
    {
        relPathSplit.push_back("..");
    }
    for (int i=commonIndex+1; i<(int)pathSplit.size(); ++i)
    {
        relPathSplit.push_back(pathSplit[i]);
    }
    return pystring::join("/", relPathSplit);
}

void ExactMatch(FnMatchInfo& matchInfo,
                FnPlatform::StringView testpath,
                FnPlatform::StringView pattern)
{
    if(testpath == pattern)
    {
        matchInfo.match = true;
        matchInfo.canMatchChildren = false;
    }
    else
    {
        matchInfo.match = false;
        matchInfo.canMatchChildren = IsAncestor(testpath, pattern);
    }
}

void FnMatch(FnMatchInfo& matchInfo,
             FnPlatform::StringView testpath,
             FnPlatform::StringView pattern)
{
    static FnPluginManager::LazyHostSuite<
        FnGeolibCookInterfaceUtilsHostSuite_v2>
        suite = { "GeolibCookInterfaceUtilsHost", 2 };

    FnGeolibCookInterfaceUtilsFnMatchInfo result =
        suite->FnMatch(testpath.data(), static_cast<int>(testpath.size()),
                       pattern.data(), static_cast<int>(pattern.size()));
    matchInfo.match = result.matches != 0;
    matchInfo.canMatchChildren = result.canMatchChildren != 0;
}

void FnMatchIncludingAncestors(FnMatchInfo& matchInfo,
                               FnPlatform::StringView testpath,
                               FnPlatform::StringView pattern)
{
    static FnPluginManager::LazyHostSuite<FnGeolibCookInterfaceUtilsHostSuite_v2> suite = {
        "GeolibCookInterfaceUtilsHost", 2};

    FnGeolibCookInterfaceUtilsFnMatchInfo result =
        suite->FnMatchIncludingAncestors(testpath.data(), static_cast<int>(testpath.size()),
                                         pattern.data(), static_cast<int>(pattern.size()));
    matchInfo.match = result.matches != 0;
    matchInfo.canMatchChildren = result.canMatchChildren != 0;
}

std::string MakeUniqueName(FnPlatform::StringView baseName,
                           const std::set<std::string>& existingNames)
{
    std::string name(baseName);
    int count = 1;
    while (existingNames.find(name) != existingNames.end())
    {
        std::ostringstream nameStr;
        nameStr << baseName << count;
        name = nameStr.str();
        ++count;
    }

    return name;
}

std::string MakeSafeIdentifier(FnPlatform::StringView identifier)
{
    if (identifier.empty()) return std::string("_");

    std::string result(identifier);
    for (std::string::iterator i=result.begin(), e=result.end();
         i!=e; ++i)
    {
        if (!std::isalnum(*i))
        {
            *i = '_';
        }
    }

    if (!std::isalpha(result[0]) && result[0] != '_')
    {
        return std::string("_") + result;
    }

    return result;
}

int Compare(FnPlatform::StringView path1, FnPlatform::StringView path2)
{
    size_t n1 = path1.size(), n2 = path2.size();

    // the only normalization we do is to strip trailing slashes.
    while (n1 > 1 && path1[n1 - 1] == '/')
        --n1;

    while (n2 > 1 && path2[n2 - 1] == '/')
        --n2;

    for (size_t i = 0; i < std::min(n1, n2); ++i)
    {
        if (path1[i] != path2[i])
        {
            // map '/' to NUL, so that "/a/b/c/d" sorts before "/a/b/c.d"
            char ch1 = path1[i] == '/' ? '\0' : path1[i];
            char ch2 = path2[i] == '/' ? '\0' : path2[i];
            return static_cast<unsigned char>(ch1) -
                   static_cast<unsigned char>(ch2);
        }
    }
    return static_cast<int>(n1 - n2);
}

}  // namespace Path
}
FNGEOLIBUTIL_NAMESPACE_EXIT
