// Copyright (c) 2012 The Foundry Visionmongers Ltd. All Rights Reserved.

#include <cstring>

#include <string>

#include <FnDisplayDriver/suite/FnDisplayDriverSuite.h>
#include "FnDisplayDriver/NewChannelMessage.h"
#include "FnDisplayDriver/NewFrameMessage.h"

namespace Foundry
{
namespace Katana
{
NewChannelMessage::NewChannelMessage(const uint8_t* frameID,
                                     uint16_t channelID,
                                     uint32_t height,
                                     uint32_t width,
                                     int xorigin,
                                     int yorigin,
                                     float sampleRate1,
                                     float sampleRate2,
                                     uint32_t dataSize)
    : Message(getSuite()->createNewChannelMessage(frameID,
                                                  channelID,
                                                  height,
                                                  width,
                                                  xorigin,
                                                  yorigin,
                                                  sampleRate1,
                                                  sampleRate2,
                                                  dataSize)),
      _channelName()
{
}

NewChannelMessage::NewChannelMessage(const NewFrameMessage& frame,
                                     uint16_t channelID,
                                     uint32_t height,
                                     uint32_t width,
                                     int xorigin,
                                     int yorigin,
                                     float sampleRate1,
                                     float sampleRate2,
                                     uint32_t dataSize)
    : Message(getSuite()->createNewChannelMessage(frame.frameUUID(),
                                                  channelID,
                                                  height,
                                                  width,
                                                  xorigin,
                                                  yorigin,
                                                  sampleRate1,
                                                  sampleRate2,
                                                  dataSize)),
      _channelName()
{
}

int NewChannelMessage::setChannelName(const std::string& name)
{
    return getSuite()->setChannelName(_handle, name.c_str());
}

void NewChannelMessage::setChannelDimensions(uint32_t width, uint32_t height)
{
    getSuite()->setChannelDimensions(_handle, width, height);
}

void NewChannelMessage::setChannelOrigin(int ox, int oy)
{
    getSuite()->setChannelOrigin(_handle, ox, oy);
}

void NewChannelMessage::setDataSize(uint32_t dataSize)
{
    getSuite()->setDataSize(_handle, dataSize);
}

void NewChannelMessage::setChannelID(uint16_t channelID)
{
    getSuite()->setChannelID(_handle, channelID);
}

void NewChannelMessage::setSampleRate(const float* sampleRate)
{
    getSuite()->setSampleRate(_handle, sampleRate);
}

const uint8_t* NewChannelMessage::frameUUID() const
{
    return getSuite()->channelFrameUUID(_handle);
}

uint16_t NewChannelMessage::channelID() const
{
    return getSuite()->channelID(_handle);
}

uint32_t NewChannelMessage::channelHeight() const
{
    return getSuite()->channelHeight(_handle);
}

uint32_t NewChannelMessage::channelWidth() const
{
    return getSuite()->channelWidth(_handle);
}

int NewChannelMessage::channelXOrigin() const
{
    return getSuite()->channelXOrigin(_handle);
}

int NewChannelMessage::channelYOrigin() const
{
    return getSuite()->channelYOrigin(_handle);
}

uint32_t NewChannelMessage::channelDataSize() const
{
    return getSuite()->channelDataSize(_handle);
}

const std::string& NewChannelMessage::channelName() const
{
    const char* channelName = getSuite()->channelName(_handle);
    if (channelName)
    {
        _channelName = channelName;
    }
    else
    {
        _channelName.clear();
    }
    return _channelName;
}

uint32_t NewChannelMessage::channelNameLength() const
{
    return getSuite()->channelNameLength(_handle);
}

const float* NewChannelMessage::sampleRate() const
{
    return getSuite()->sampleRate(_handle);
}

bool NewChannelMessage::isIdPass() const
{
    // Get raw string to avoid string allocation
    const char* channelName = getSuite()->channelName(_handle);
    return strstr(channelName, "__id") != nullptr;
}

////////////////////////////////////////////////////////////////////////////////

NewChannelMessage_v2::NewChannelMessage_v2(const uint8_t* frameID,
                                           uint16_t channelID,
                                           PixelLayout pixelLayout,
                                           uint32_t height,
                                           uint32_t width,
                                           int xorigin,
                                           int yorigin,
                                           float sampleRate1,
                                           float sampleRate2,
                                           uint32_t dataSize)
    : NewChannelMessage(getSuite()->createNewChannelMessage_v2(
          frameID,
          channelID,
          static_cast<FnKatPixelLayout>(pixelLayout),
          height,
          width,
          xorigin,
          yorigin,
          sampleRate1,
          sampleRate2,
          dataSize))
{
}

NewChannelMessage_v2::NewChannelMessage_v2(const NewFrameMessage& frame,
                                           uint16_t channelID,
                                           PixelLayout pixelLayout,
                                           uint32_t height,
                                           uint32_t width,
                                           int xorigin,
                                           int yorigin,
                                           float sampleRate1,
                                           float sampleRate2,
                                           uint32_t dataSize)
    : NewChannelMessage(getSuite()->createNewChannelMessage_v2(
          frame.frameUUID(),
          channelID,
          static_cast<FnKatPixelLayout>(pixelLayout),
          height,
          width,
          xorigin,
          yorigin,
          sampleRate1,
          sampleRate2,
          dataSize))
{
}

void NewChannelMessage_v2::setPixelLayout(PixelLayout pixelLayout)
{
    getSuite()->setPixelLayout_v2(_handle,
                                  static_cast<FnKatPixelLayout>(pixelLayout));
}

NewChannelMessage_v2::PixelLayout NewChannelMessage_v2::pixelLayout() const
{
    return static_cast<NewChannelMessage_v2::PixelLayout>(
        getSuite()->pixelLayout_v2(_handle));
}

}  // namespace Katana
}  // namespace Foundry
