# Copyright (c) 2017 The Foundry Visionmongers Ltd. All Rights Reserved.
"""
Module defining the L{BaseViewerPluginExtension} class.
"""


class BaseViewerPluginExtension(object):
    """
    Base class for extending Viewer API-based Viewer plug-ins.

    Classes derived from this class should be registered as Katana plug-ins of
    type C{'ViewerPluginExtension'}.

    One instance of a class derived from this class is created for each Viewer
    API-based Viewer tab, which means that tab-specific state can be stored in
    instance variables per viewer plug-in extension/viewer tab.
    """

    def onTabCreated(self, tab):
        """
        Callback triggered when an instance of a C{BaseViewerTab} has been
        created.

        @type tab: C{BaseViewerTab}
        @param tab: The viewer tab that was created.
        """
        pass

    def onDelegateCreated(self, viewerDelegate, pluginName):
        """
        Callback triggered when a ViewerDelegate has been created via the
        C{BaseViewerTab} interface.

        @type viewerDelegate: C{ViewerDelegate}
        @type pluginName: C{str}
        @param viewerDelegate: The C{ViewerDelegate} that was created.
        @param pluginName: The registered plug-in name of the viewer delegate.
        """
        pass

    def onViewportCreated(self, viewportWidget, pluginName, viewportName):
        """
        Callback triggered when a ViewportWidget has been created via the
        C{BaseViewerTab} interface.

        @type viewportWidget: C{ViewportWidget}
        @type pluginName: C{str}
        @type viewportName: C{str}
        @param viewportWidget: The C{ViewportWidget} that was created.
        @param pluginName: The registered plug-in name of the C{ViewportWidget}.
        @param viewportName: The easily identifiable name of the viewport
            which can be used to look-up the viewport in the viewer delegate.
        """
        pass

    def onApplyTerminalOps(self, txn, inputOp, viewerDelegate):
        """
        Callback triggered when the chain of terminal Ops is being created for
        a viewer delegate, for example after setting the view flag on a node.

        @type txn: C{FnGeolib.GeolibRuntime.Transaction}
        @type inputOp: C{FnGeolib.GeolibRuntimeOp} or C{None}
        @type viewerDelegate: C{ViewerAPI.ViewerDelegate}
        @rtype: C{FnGeolib.GeolibRuntimeOp}
        @param txn: The current Geolib3 tansaction.
        @param inputOp: The last Op in the current Op chain.
        @param viewerDelegate: The viewer delegate for which terminal Ops are
            applied.
        @return: The terminal Op to use in the Op chain for the viewer. Should
            be the given input Op, if no changes to the Op chain are required.
        """
        return inputOp
