// Copyright (c) 2017 The Foundry Visionmongers Ltd. All Rights Reserved.

#ifndef INCLUDED_UTIL_OPENGLTIMING_H
#define INCLUDED_UTIL_OPENGLTIMING_H

#ifdef USEGLTIMING

#include "GL/glew.h"

#include <string>
#include <cstdio>
#include <cstdlib>
#include <cstring>

// requires trailing semi-colon
#define GLTIMING_FRAMESCOPE() \
    static int _gpuTimingFrameCount = 0; printf("GL GPU timing: frame %i\n", _gpuTimingFrameCount++)
#define GLTIMING_STATICVAR() \
    static UTIL::OpenGL::GPUTimingStats _gpuTimingStats
#define GLTIMING_SCOPED(_autovar,_layerName) \
    UTIL::OpenGL::GPUTiming _autovar(&_gpuTimingStats,_layerName)

namespace UTIL
{
    namespace OpenGL
    {
        struct GPUTimingStats
        {
            GPUTimingStats()
            {
                _queries[0]      = _queries[1] = 0;
                _writeQueryIndex = 0;
                _lastFrameTimeMS = 0.0f;
                _minFrameTimeMS  = 0.0f;
                _maxFrameTimeMS  = 0.0f;
                _avgFrameTimeMS  = 0.0f;
                _numFrameTimes =  0;

                glGenQueries(2, _queries);
                glBeginQuery(GL_TIME_ELAPSED, _queries[_writeQueryIndex]);
                glEndQuery(GL_TIME_ELAPSED);
                ++_writeQueryIndex;
            }

            GLuint _queries[2];
            int    _writeQueryIndex;
            float  _lastFrameTimeMS;
            float  _minFrameTimeMS;
            float  _maxFrameTimeMS;
            float  _avgFrameTimeMS;
            int    _numFrameTimes;
        };

        class GPUTiming
        {
        public:
            explicit GPUTiming(GPUTimingStats *context, const std::string &layerName)
                :
                _layerName(layerName)
            {
                if (!GLEW_ARB_timer_query)
                {
                    return;
                }
                _context = context;
                glBeginQuery(GL_TIME_ELAPSED, _context->_queries[_context->_writeQueryIndex]);
            }

            ~GPUTiming()
            {
                if (!GLEW_ARB_timer_query)
                {
                    return;
                }
                glEndQuery(GL_TIME_ELAPSED);
                // wait
                GLint available = 0;
                const int readQueryIndex = (_context->_writeQueryIndex + 1) % 2;
                while (!available) {
                    glGetQueryObjectiv(_context->_queries[readQueryIndex], GL_QUERY_RESULT_AVAILABLE, &available);
                }

                GLuint64 timeElapsedNS = 0;
                glGetQueryObjectui64v(_context->_queries[readQueryIndex], GL_QUERY_RESULT, &timeElapsedNS);

                _context->_lastFrameTimeMS = timeElapsedNS / 1024.f / 1024.f;
                if (_context->_lastFrameTimeMS < _context->_minFrameTimeMS) {
                    _context->_minFrameTimeMS = _context->_lastFrameTimeMS;
                }
                if (_context->_lastFrameTimeMS > _context->_maxFrameTimeMS) {
                    _context->_maxFrameTimeMS = _context->_lastFrameTimeMS;
                }
                _context->_avgFrameTimeMS = (_context->_lastFrameTimeMS + _context->_numFrameTimes * _context->_avgFrameTimeMS) / (_context->_numFrameTimes + 1);
                _context->_writeQueryIndex = readQueryIndex;
                ++_context->_numFrameTimes;
                printf("%s: Last %.2fms: Min %.2fms: Max %.2fms: Avg %.2fms\n",
                    _layerName.c_str(),
                    _context->_lastFrameTimeMS,
                    _context->_minFrameTimeMS,
                    _context->_maxFrameTimeMS,
                    _context->_avgFrameTimeMS);
                do_csv_export();
            }

        private:
            void do_csv_export()
            {
                const char *csvPath = getenv("GL_TIMING_CSV");
                if (0 == csvPath) {
                    return;
                }
                const char *csvLayerFilter = getenv("GL_TIMING_CSV_FILTER");
                if (0 != csvLayerFilter) {
                    // only accept layers that are an exact match
                    if (0 != strcmp(csvLayerFilter, _layerName.c_str())) {
                        return;
                    }
                }
                FILE *csv = fopen(csvPath, "a");
                if (0 == csv) {
                    return;
                }
                fprintf(csv, "%s,%i,%f,%f,%f,%f\n",
                    _layerName.c_str(),
                    _context->_numFrameTimes,
                    _context->_avgFrameTimeMS,
                    _context->_lastFrameTimeMS,
                    _context->_minFrameTimeMS,
                    _context->_maxFrameTimeMS);
                fclose(csv);
            }

        private:
            GPUTimingStats *_context;
            std::string     _layerName;
        };

    }  // namespace OpenGL
}  // namespace UTIL

#else  // USEGLTRACEMARKERS

// requires trailing semi-colon, boils away both arguments
#define GLTIMING_FRAMESCOPE() ((void)0)
#define GLTIMING_STATICVAR() ((void)0)
#define GLTIMING_SCOPED(_autovar,_msg) ((void)0)

#endif  // USEGLTRACEMARKERS

#endif  // INCLUDED_UTIL_OPENGLTIMING_H
