// Copyright (c) 2020 The Foundry Visionmongers Ltd. All Rights Reserved.

#ifndef MANIPULATORSHELPERS_H_
#define MANIPULATORSHELPERS_H_

#include <ImathMatrix.h>
#include <ImathVec.h>

#include "FnViewer/plugin/FnViewport.h"
#include "FnViewer/utils/FnImathHelpers.h"

namespace Foundry
{
namespace Katana
{
namespace ViewerUtils
{
/// Calculates a scale that will compensate for the perspective.
///
/// This guarantees that the size of an object at the given point will be
/// constant from any camera position, fov, etc. by compensating the effect
/// of the projection and view matrices.
///
/// \param viewport The pointer to the viewport where the object will be drawn.
/// \param point The position where the object will be drawn.
/// \param factor A optional factor size. Default is 100.0.
/// \return The scale that needs to be applied to the object in order to keep
///     its size fixed.
inline double calculateFixedSizeScale(
    const ViewerAPI::ViewportWrapperPtr& viewport,
    const IMATH_NAMESPACE::V3d& point,
    const double factor = 100.0)
{
    const int width = viewport->getWidth();
    const int height = viewport->getHeight();

    const IMATH_NAMESPACE::M44d viewMatrix(
        toImathMatrix44d(viewport->getViewMatrix44d()));
    const IMATH_NAMESPACE::M44d projMatrix(
        toImathMatrix44d(viewport->getProjectionMatrix()));

    const IMATH_NAMESPACE::M44d toScreenXform(viewMatrix * projMatrix);
    const IMATH_NAMESPACE::M44d screenToManip(toScreenXform.inverse());

    const IMATH_NAMESPACE::V3d a(point * toScreenXform);
    IMATH_NAMESPACE::V3d b(a.x, a.y, a.z);

    if (width < height)
    {
        b.x += factor / width;
    }
    else
    {
        b.y += factor / height;
    }

    return (a * screenToManip - b * screenToManip).length();
}
}  // namespace ViewerUtils
}  // namespace Katana
}  // namespace Foundry

#endif  // MANIPULATORSHELPERS_H_
