// Copyright (c) 2017 The Foundry Visionmongers Ltd. All Rights Reserved.

#ifndef FNIMATHHELPERS_H_
#define FNIMATHHELPERS_H_

#include <FnViewer/plugin/FnMathTypes.h>
#include <FnAttribute/FnAttribute.h>

#include <ImathVec.h>
#include <ImathMatrix.h>


namespace Foundry
{
namespace Katana
{
namespace ViewerUtils
{

/**
 *  This optional header contains helper functions for the manipulation and
 *  conversion of Imath Vectors and Matrices. If this header is used, then the
 *  OpenEXR library or its Imath component needs to be linked too.
 */

static inline IMATH_NAMESPACE::V3d toRadians(const IMATH_NAMESPACE::V3d& vec)
{
    return vec * (M_PI / 180.0);
}

static inline IMATH_NAMESPACE::V3d toDegrees(const IMATH_NAMESPACE::V3d& vec)
{
    return vec * (180.0 / M_PI);
}

inline IMATH_NAMESPACE::Matrix44<double> toImathMatrix44d(const double* m)
{
    return IMATH_NAMESPACE::Matrix44<double>(m[0], m[1], m[2], m[3], m[4], m[5],
                                             m[6], m[7], m[8], m[9], m[10],
                                             m[11], m[12], m[13], m[14], m[15]);
}

inline IMATH_NAMESPACE::Matrix44<double> toImathMatrix44d(
    const Foundry::Katana::ViewerAPI::Matrix44d& m)
{
    return IMATH_NAMESPACE::Matrix44<double>(
        m.data[0], m.data[1], m.data[2], m.data[3], m.data[4], m.data[5],
        m.data[6], m.data[7], m.data[8], m.data[9], m.data[10], m.data[11],
        m.data[12], m.data[13], m.data[14], m.data[15]);
}

inline IMATH_NAMESPACE::Matrix44<double> toImathMatrix44d(
    FnAttribute::DoubleAttribute a)
{
    if (!a.isValid())
    {
        return IMATH_NAMESPACE::M44d();
    }

    FnAttribute::DoubleConstVector m = a.getNearestSample(0);
    return IMATH_NAMESPACE::M44d(m[0], m[1], m[2], m[3], m[4], m[5], m[6], m[7],
                                 m[8], m[9], m[10], m[11], m[12], m[13], m[14],
                                 m[15]);
}

inline Foundry::Katana::ViewerAPI::Matrix44d toMatrix44d(
    const IMATH_NAMESPACE::Matrix44<double>& m)
{
    return Foundry::Katana::ViewerAPI::Matrix44d(
        m[0][0], m[0][1], m[0][2], m[0][3],
        m[1][0], m[1][1], m[1][2], m[1][3],
        m[2][0], m[2][1], m[2][2], m[2][3],
        m[3][0], m[3][1], m[3][2], m[3][3]);
}

inline FnAttribute::DoubleAttribute toDoubleAttribute(
    const IMATH_NAMESPACE::Matrix44<double>& m)
{
    double matrixArray[16] = {
        m[0][0], m[0][1], m[0][2], m[0][3],
        m[1][0], m[1][1], m[1][2], m[1][3],
        m[2][0], m[2][1], m[2][2], m[2][3],
        m[3][0], m[3][1], m[3][2], m[3][3] };
    return FnAttribute::DoubleAttribute(matrixArray, 16, 4);
}

inline IMATH_NAMESPACE::V3d toImathV3d(
    const Foundry::Katana::ViewerAPI::Vec3d& v)
{
    return IMATH_NAMESPACE::V3d(v.x, v.y, v.z);
}

inline Foundry::Katana::ViewerAPI::Vec3d toVec3d(const IMATH_NAMESPACE::V3d& v)
{
    return Foundry::Katana::ViewerAPI::Vec3d(v.x, v.y, v.z);
}

inline Foundry::Katana::ViewerAPI::Vec3f toVec3f(const IMATH_NAMESPACE::V3d& v)
{
    return Foundry::Katana::ViewerAPI::Vec3f(
            static_cast<float>(v.x),
            static_cast<float>(v.y),
            static_cast<float>(v.z));
}

inline IMATH_NAMESPACE::V2i toImathV2i(
    const Foundry::Katana::ViewerAPI::Vec2i& v)
{
    return IMATH_NAMESPACE::V2i(v.x, v.y);
}

inline Foundry::Katana::ViewerAPI::Vec2i toVec2i(const IMATH_NAMESPACE::V2i& v)
{
    return Foundry::Katana::ViewerAPI::Vec2i(v.x, v.y);
}

} // ViewerUtils
} // Katana
} // Foundry

#endif // FNIMATHHELPERS_H_
