// Copyright (c) 2016 The Foundry Visionmongers Ltd. All Rights Reserved.

#ifndef GLSHADERPROGRAM_H_
#define GLSHADERPROGRAM_H_

#ifdef _WIN32
#include <Windows.h>
#endif

#include <FnAttribute/FnAttribute.h>
#include <FnViewer/plugin/FnMathTypes.h>

#include <GL/glew.h>
#include <string>
#include <map>

namespace Foundry
{
namespace Katana
{
namespace ViewerUtils
{

using Foundry::Katana::ViewerAPI::Matrix44d;

/// An enumeration of the different type of GLSL shaders.
enum ShaderType
{
    VERTEX = GL_VERTEX_SHADER,
    FRAGMENT = GL_FRAGMENT_SHADER,
    GEOMETRY = GL_GEOMETRY_SHADER,
    TESS_CONTROL = GL_TESS_CONTROL_SHADER,
    TESS_EVALUATION = GL_TESS_EVALUATION_SHADER,
    COMPUTE = GL_COMPUTE_SHADER
};

/**
 * This is a class that encapsulates the process of compiling and using GLSL
 * shader programs.
 */
class GLShaderProgram
{
public:
    GLShaderProgram();
    ~GLShaderProgram();

    /** Cleans up GL resources and other cached data. */
    void cleanup();

    /**
     * Reads a GLSL shader from a file, compiles it and attaches it to
     * the current program.
     */
    void compileShader(const std::string& filename, ShaderType type);

    /**
     * Compiles a GLSL shader from the passed shader source string, and
     * attaches it to the current handle.
     */
    void compileShader(const std::string& name, ShaderType type, const std::string& shaderSource);

    /// Gets the location of named uniform in the current shader program.
    GLuint getUniformLocation(const std::string& name);

    /// Links the currently attached shaders.
    void link();

    /// Activates the shader program.
    void use();

    /// Returns the raw shader handle;
    GLuint getHandle() const { return m_handle; }

    /// Returns true if the shader program has been linked.
    bool isLinked() const { return m_isLinked; }

    /// Binds the named attribute location to the specified index.
    void bindAttribLocation(GLuint location, const std::string& name);

    /// Binds the named fragment data location to the specified index.
    void bindFragDataLocation(GLuint location, const std::string& name);

    /// Sets the value of the named uniform variable
    void setUniform(const std::string& name, float val);

    /// Sets the value of the named uniform variable
    void setUniform(const std::string& name, float x, float y);

    /// Sets the value of the named uniform variable
    void setUniform(const std::string& name, float x, float y, float z);

    /// Sets the value of the named uniform variable
    void setUniform(const std::string& name, float x, float y, float z, float w);

    /// Sets the value of the named uniform variable
    void setUniform(const std::string& name, int val);

    /// Sets the value of the named uniform variable
    void setUniform(const std::string& name, bool val);

    /// Sets the value of the named uniform variable
    /// Can accept multiple sizes of FloatAttribute values.
    void setUniform(const std::string& name, FnAttribute::FloatAttribute val);

    /// Sets the value of the named uniform variable
    /// Can accept multiple sizes of DoubleAttribute values.
    void setUniform(const std::string& name, FnAttribute::DoubleAttribute val);

    /// Sets the value of the named uniform variable
    void setUniform(const std::string& name, const Matrix44d& matrix);

    /// Sets the value of the named uniform variable
    void setUniformMatrix4d(const std::string& name, const double* matrix);

private:
    /// Loads a GLSL shader file from disk.
    const std::string readFile(const std::string& filename);
    /// The shader program handle.
    GLuint m_handle;
    /// Is the program linked.
    bool m_isLinked;
    /// A map of uniform variable names and their location handles.
    std::map<std::string, GLuint> m_uniformLocations;
};

} // ViewerUtils
} // Katana
} // Foundry
#endif /* GLSHADERPROGRAM_H_ */
