
// Copyright (c) 2017 The Foundry Visionmongers Ltd. All Rights Reserved.

#ifndef GLMANIPULATOR_H_
#define GLMANIPULATOR_H_

#include <FnViewer/plugin/FnManipulator.h>
#include <FnViewer/plugin/FnManipulatorHandle.h>
#include <FnViewer/plugin/FnMathTypes.h>
#include <FnViewer/plugin/FnEventWrapper.h>
#include <FnViewer/utils/FnGLShaderProgram.h>

#include <stdint.h>
#include <string>

namespace Foundry
{
namespace Katana
{
namespace ViewerUtils
{

/// Render mode to use by the manipulators shader
enum RenderMode
{
    kPicking = 0,
    kShadingFlat = 1,
    kShading3D = 2,
    kLineStipple = 3,
    kTextured = 4,
};

// A lot of entities from the ViewerAPI namespace will be accessed.
using namespace Foundry::Katana::ViewerAPI;

/**
 * @brief A base class for a GL based Manipulator.
 *
 * This is a base class implementation of an OpenGL powered manipulator
 * that is managed by the GLManipulatorLayer layer. It specializes the
 * functionality of the generic Manipulator API, that is meant to be used by
 * any renderers, including non GL ones. The Manipulators provided by Katana
 * also extend this class.
 *
 * The handles of a GLManipulator should extend GLManipulatorHandle.
 */
class GLManipulator : public Manipulator
{
public:

    /** @brief Constructor. */
    GLManipulator();

    /** @brief Destructor. */
    virtual ~GLManipulator();

    /**
     * @brief The technology tag to be added to the manipulator's tags.
     *
     *     tag key: "technology"
     *     tag value: StringAttribute(kTechnology)
     */
    static const std::string kTechnology;
};

/**
 * @brief A base class for a GL based ManipulatorHandle.
 *
 * This is a base class implementation of an OpenGL powered manipulator handle
 * that is managed by the GLManipulatorLayer layer. It specializes the
 * functionality of the generic ManipulatorHandle API, that is meant to be used
 * by any renderers, including non GL ones. The ManipulatorHandles provided by
 * Katana extend this class.
 *
 * This also implements the detection of dragging using a mouse or pointer.
 *
 * The manipulator class of a GLManipulatorHandle should extend GLManipulator.
 */
class GLManipulatorHandle : public ManipulatorHandle
{
public:

    /** @brief Constructor. */
    GLManipulatorHandle();

    /** @brief Destructor. */
    virtual ~GLManipulatorHandle();

    /**
    * @brief Cancels the manipulation.
    *
    * Typically called before the handle is destroyed to discard any pending
    * transform that hasn't been made persistent while dragging the handle.
    */
   virtual void cancelManipulation();

     /**
     * @brief Draws the handle using the standard shaders.
     *
     * This should be called by the implementation on child classes in order to
     * bind the standard shader.
     *
     * The vertex shader makes use of the Manipulator's transform and also the
     * handle local transform that can be set using setLocalXform() in order to
     * render the handle with the correct transform.
     */
    virtual void draw();

    /**
     * @brief Draws the handle for picking using the standard shaders.
     *
     * This should be called by the implementation on child classes in order to
     * bind the standard picking shader, unless if a custom shader is meant to
     * be used (not recommended).
     *
     * The vertex shader makes use of the Manipulator's transform and also a
     * local handle transform that can be set using setLocalXform() in order to
     * render the handle with the correct transform.
     *
     * The fragment shader will render the handles using the picker ID, which
     * is then decoded by GLManipulatorLayer during picking.
     */
    virtual void pickerDraw(int64_t pickerID);

    /**
     * @brief Handles events once it is activated by the GLManipulatorLayer.
     *
     * Detects if the user is dragging the handle. It will call the dragging
     * functions: startDrag(), drag() and endDrag()
     *
     * This should be called by the implementation on child classes in order to
     * detect the handle dragging, unless if dragging is not required.
     */
    virtual bool event(const FnEventWrapper& eventData);

    /**
     * @brief Gets the option given the option ID.
     *
     * It has been overridden to support the "NewMousePosition" option.
     */
    virtual FnAttribute::Attribute getOption(
            Foundry::Katana::ViewerAPI::OptionIdGenerator::value_type);

protected:
    /// @brief Utility that returns a pointer to the GLManipulator instance.
    GLManipulator* getGLManipulator();

    /// @brief Utility that returns the Viewport.
    ViewportWrapperPtr getViewport();

    /// @brief Utility that returns the ViewerDelegate.
    ViewerDelegateWrapperPtr getViewerDelegate();

    /// @brief Specifies if the user is currently dragging the handle
    bool isDragging();

    /// @brief Specified if the user has been dragging the handle
    bool hasBeenDragged();

    /**
     * @brief Gets the dragging plane's origin and normal.
     *
     * When dragging a manipulator the mouse pointer position will be projected
     * on a virtual plane in world space. This point on the plane will then be
     * used as the dragging values during the interaction.
     */
    virtual bool getDraggingPlane(Vec3d& origin, Vec3d& normal);

    /**
     * @brief Called when a mouse dragging starts.
     *
     * This can be used to initialize any dragging related data.
     *
     * @param initialPointOnPlane The initial projection of the mouse pointer
     *     on the virtual manipulation plane defined by getDraggingPlane().
     * @param initialMousePosition The initial mouse position in pixels.
     */
    virtual void startDrag(const Vec3d& initialPointOnPlane,
                           const Vec2i& initialMousePosition);

    /**
     * @brief Called when a mouse dragging occurs.
     *
     * This is the function where the Manipulator::setValue() should be called
     * to set any node graph parameters / scene graph attributes.
     *
     * @param initialPointOnPlane The initial projection of the mouse pointer
     *     on the virtual manipulation plane defined by getDraggingPlane().
     * @param previousPointOnPlane The previous projection of the mouse
     *     pointer on the virtual manipulation plane defined by
     *     getDraggingPlane().
     * @param currentPointOnPlane The current projection of the mouse pointer
     *     on the virtual manipulation plane defined by getDraggingPlane().
     * @param initialMousePosition The initial mouse position in pixels.
     * @param previousMousePosition The previous mouse position in pixels.
     * @param currentMousePosition The current mouse position in pixels.
     * @param isFinal True if this is a final value.
     */
    virtual void drag(const Vec3d& initialPointOnPlane,
                      const Vec3d& previousPointOnPlane,
                      const Vec3d& currentPointOnPlane,
                      const Vec2i& initialMousePosition,
                      const Vec2i& previousMousePosition,
                      const Vec2i& currentMousePosition,
                      bool isFinal);

    /**
     * @brief Called when a mouse dragging ends.
     *
     * This can be used to tear down any dragging related data.
     */
    virtual void endDrag();

    /**
     * @brief Gets whether the manipulator handle can process the given
     *     keyboard modifiers. The default behavior is to ignore all events
     *     with modifiers.
     *
     * This protected, virtual function can be overridden to allow manipulator
     * handle implementations to process certain (or all) modifiers.
     */
    virtual bool canProcessKeyboardModifiers(int modifiers) const
    {
        return modifiers == FnEventWrapper::kNoModifier;
    }

    /// @brief Activates the default manipulator handle shader for drawing.
    void useDrawingShader(const Matrix44d& xform, const Vec4f& color,
                          bool isFlat);

    void useSnappingShader(const Matrix44d& xform);

    /** @brief Activates the manipulator handle shader for drawing line
     *     stipples.
     *
     * The `pattern` and the `factor` parameters are equivalent to the ones in
     * the deprecated `glLineStipple` function.
     */
    void useLineStippleDrawingShader(const Matrix44d& xform,
                                     const Vec4f& color, int pattern,
                                     float factor);

    /// @brief Activates the default manipulator handle shader for drawing.
    void usePickingShader(const Matrix44d& xform, int handleId,
                          int handlePriority);

    /// @return The latest keyboard modifiers mask (i.e. the modifiers received
    ///     in the last handled event() call).
    int getLatestKeyboardModifiers() const
    {
        return m_latestKeyboardModifiers;
    }

    /// @return The latest mouse position (i.e. the mouse position received in
    ///     the last handled event() call).
    const Vec2i& getLatestMousePosition() const
    {
        return m_latestMousePosition;
    }

    const Matrix44d& getInitialHandleXform() const {
        return m_initialHandleXform;
    }
    const Vec3d& getInitialPointOnPlane() const {
        return m_initialPointOnPlane;
    }
    const Vec3d& getLastPointOnPlane() const {
        return m_previousPointOnPlane;
    }
    const Vec2i& getInitialMousePosition() const {
        return m_initialMousePosition;
    }
    const Vec2i& getLastMousePosition() const {
        return m_previousMousePosition;
    }

private:
    /// Default shader initialization. For internal use only
    static void useShader(ViewportWrapperPtr viewport, const Matrix44d& xform,
                          RenderMode renderMode,
                          GLShaderProgram& shaderProgram);

    /// Determines whether the manipulator has captured a click event. It will
    /// be switched to false when the release event is received.
    bool m_processingManipulation;

    /// @brief Set by setLocalXform().
    Matrix44d m_localXform;

    /// The shader programs.
    GLShaderProgram m_shaderProgram;
    GLShaderProgram m_shaderProgramLineStipple;

    /// The last mouse position received in a handled event() call.
    Vec2i m_latestMousePosition;

    /// The last keyboard modifiers mask received in a handled event() call.
    int m_latestKeyboardModifiers;

    /// The initial handle xform when the dragging started
    Matrix44d m_initialHandleXform;

    /// The initial and previous point on dragging plane
    Vec3d m_initialPointOnPlane;
    Vec3d m_previousPointOnPlane;

    /// The initial and previous mouse positions in pixels
    Vec2i m_initialMousePosition;
    Vec2i m_previousMousePosition;

    /// Specifies if the user is currently dragging the handle
    bool m_isDragging;

    /// Specifies if the user has been dragging the handle
    bool m_hasBeenDragged;
};


}
}
}

#endif /* GLMANIPULATOR_H_ */
