// Copyright (c) 2017 The Foundry Visionmongers Ltd. All Rights Reserved.

#ifndef FNDRAWABLE_H_
#define FNDRAWABLE_H_

#include <vector>
#include <FnViewer/plugin/FnGLStateHelper.h>
#include <FnViewer/plugin/FnMathTypes.h>
#include <FnViewer/utils/FnDrawingHelpers.h>
#include <FnViewer/utils/FnImathHelpers.h>
#include <GL/glew.h>

using Foundry::Katana::ViewerAPI::Vec3f;
using Foundry::Katana::ViewerAPI::Vec2f;

namespace Foundry
{
namespace Katana
{
namespace ViewerUtils
{


/**
 * @brief An object that can be drawn on the Viewport.
 *
 * This can be used to draw Manipulator handle elements, such as lines, points
 * or meshes. It maintains a VAO and VBOs internally.
 */
class Drawable
{
public:

    /// Buffer types
    enum BUFFER {
        VERTEX_BUFFER = 0,
        NORMAL_BUFFER,
        UV_BUFFER,
        INDEX_BUFFER,
        NUM_BUFFERS
    };

    /// Constructor
    Drawable()
    : m_isReady(false),
      m_vao(0),
      m_numVertices(0),
      m_numIndices(0),
      m_lineWidth(1),
      m_pointSize(1)
    {
        m_vbo[VERTEX_BUFFER] = 0;
        m_vbo[NORMAL_BUFFER] = 0;
        m_vbo[INDEX_BUFFER] = 0;
        m_vbo[UV_BUFFER] = 0;
    }

    /// Destructor
    ~Drawable() { cleanup(); }

    /// Tells if the Drawable is ready to be drawn (if setup has been called).
    bool isReady()
    {
        return m_isReady;
    }

    /// Sets up the vertex buffers
    void setup(
        const std::vector<Vec3f>& vertices,
        const std::vector<Vec3f>& normals,
        const std::vector<unsigned int>& indices,
        const std::vector<Vec2f>& uvs = std::vector<Vec2f>())
    {
        if (isReady())
        {
            // Do not load it again
            return;
        }

        // No existing VAO found for this context
        glGenVertexArrays(1, &m_vao);
        glBindVertexArray(m_vao);

        glGenBuffers(NUM_BUFFERS, &m_vbo[0]);

        // Fill the vertex buffer
        glBindBuffer(GL_ARRAY_BUFFER, m_vbo[VERTEX_BUFFER]);
        glBufferData(
            GL_ARRAY_BUFFER,
            vertices.size() * sizeof(Vec3f),
            vertices.data(),
            GL_STATIC_DRAW);
        glVertexAttribPointer(VERTEX_BUFFER, 3, GL_FLOAT, GL_FALSE, 0, 0);
        glEnableVertexAttribArray(VERTEX_BUFFER);

        // Fill the normal buffer
        glBindBuffer(GL_ARRAY_BUFFER, m_vbo[NORMAL_BUFFER]);
        glBufferData(
            GL_ARRAY_BUFFER,
            normals.size() * sizeof(Vec3f),
            normals.data(),
            GL_STATIC_DRAW);
        glVertexAttribPointer(NORMAL_BUFFER, 3, GL_FLOAT, GL_FALSE, 0, 0);
        glEnableVertexAttribArray(NORMAL_BUFFER);

        // Fill the index buffer
        glBindBuffer(GL_ELEMENT_ARRAY_BUFFER, m_vbo[INDEX_BUFFER]);
        glBufferData(
            GL_ELEMENT_ARRAY_BUFFER,
            indices.size() * sizeof(unsigned int),
            indices.data(),
            GL_STATIC_DRAW);

        // Fill the uv buffer
        if (!uvs.empty())
        {
            glBindBuffer(GL_ARRAY_BUFFER, m_vbo[UV_BUFFER]);
            glBufferData(
                GL_ARRAY_BUFFER,
                uvs.size() * sizeof(Vec2f),
                uvs.data(),
                GL_STATIC_DRAW);
            glVertexAttribPointer(UV_BUFFER, 2, GL_FLOAT, GL_FALSE, 0, 0);
            glEnableVertexAttribArray(UV_BUFFER);
        }

        // Unbind buffers
        glBindVertexArray(0);
        glBindBuffer(GL_ELEMENT_ARRAY_BUFFER, 0);
        glBindBuffer(GL_ARRAY_BUFFER, 0);

        m_numVertices = static_cast<unsigned int>(vertices.size());
        m_numIndices = static_cast<unsigned int>(indices.size());
        m_isReady = true;
    }

    void cleanup()
    {
        if (m_vbo[VERTEX_BUFFER])
        {
            glDeleteBuffers(1, &m_vbo[VERTEX_BUFFER]);
            m_vbo[VERTEX_BUFFER] = 0;
        }

        if (m_vbo[NORMAL_BUFFER])
        {
            glDeleteBuffers(1, &m_vbo[NORMAL_BUFFER]);
            m_vbo[NORMAL_BUFFER] = 0;
        }

        if (m_vbo[INDEX_BUFFER])
        {
            glDeleteBuffers(1, &m_vbo[INDEX_BUFFER]);
            m_vbo[INDEX_BUFFER] = 0;
        }

        if (m_vbo[UV_BUFFER])
        {
            glDeleteBuffers(1, &m_vbo[UV_BUFFER]);
            m_vbo[UV_BUFFER] = 0;
        }

        if (glIsVertexArray(m_vao))
        {
            glDeleteVertexArrays(1, &m_vao);
            m_vao = 0;
        }

        m_isReady = false;
    }

    /// Update mesh vertices positions
    void updateVertices(const std::vector<Vec3f>& vertices)
    {
        updateVertices(vertices.data(), vertices.size());
    }

    /// Update mesh vertices positions
    void updateVertices(const Vec3f* const vertices,
                        const std::size_t numberOfVertices)
    {
        if (!isReady())
        {
            // Do not update if it hasn't been created yet
            return;
        }

        // Update the vertex data
        glBindBuffer(GL_ARRAY_BUFFER, m_vbo[VERTEX_BUFFER]);
        glBufferData(
            GL_ARRAY_BUFFER,
            numberOfVertices * sizeof(Vec3f),
            vertices,
            GL_STATIC_DRAW);
        glBindBuffer(GL_ARRAY_BUFFER, 0);
    }

    /// Sets the width of the lines to be drawn.
    void setLineWidth(unsigned int width)
    {
        m_lineWidth = width;
    }

    /// Sets the width of the points to be drawn.
    void setPointWidth(unsigned int size)
    {
        m_pointSize = size;
    }

    /// Draws as a mesh
    void draw(bool withTransparency = true,
              bool cull = true)
    {
        if (!isReady())
        {
            // Not loaded yet, don't render
            return;
        }

        ViewerUtils::GLStateRestore glStateRestore(ViewerUtils::ColorBuffer |
                                                   ViewerUtils::Polygon);

        if (withTransparency)
        {
            // Enable alpha
            glEnable(GL_BLEND);             // ViewerUtils::ColorBuffer
            glBlendFunc(GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);
        }

        // Update culling
        if (cull)
            glEnable(GL_CULL_FACE);         // ViewerUtils::Polygon
        else
            glDisable(GL_CULL_FACE);        // ViewerUtils::Polygon


        // Draw the VAO
        glBindVertexArray(m_vao);
        glDrawElements(GL_TRIANGLES, m_numIndices, GL_UNSIGNED_INT, 0);
        glBindVertexArray(0);
    }

    /// Draws as lines
    void drawLines(bool withTransparency=true)
    {
        if (!isReady())
        {
            // Not loaded yet, don't render
            return;
        }

        ViewerUtils::GLStateRestore glStateRestore(ViewerUtils::ColorBuffer |
                                                   ViewerUtils::Line);

        if (withTransparency)
        {
            // Enable alpha
            glEnable(GL_BLEND);             // // ViewerUtils::ColorBuffer
            glBlendFunc(GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);
        }

        glLineWidth(static_cast<GLfloat>(m_lineWidth));

        glBindVertexArray(m_vao);
        glDrawElements(GL_LINES, m_numIndices, GL_UNSIGNED_INT, 0);
        glBindVertexArray(0);
    }

    /// Draws as a points
    void drawPoints(bool withTransparency=true)
    {
        if (!isReady())
        {
            // Not loaded yet, don't render
            return;
        }

        ViewerUtils::GLStateRestore glStateRestore(ViewerUtils::ColorBuffer |
                                                   ViewerUtils::Point);

        if (withTransparency)
        {
            // Enable alpha
            glEnable(GL_BLEND);             // ViewerUtils::ColorBuffer
            glBlendFunc(GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);
        }

        glPointSize(static_cast<GLfloat>(m_pointSize));

        glBindVertexArray(m_vao);
        glDrawElements(GL_POINTS, m_numIndices, GL_UNSIGNED_INT, 0);
        glBindVertexArray(0);
    }


private:
    bool m_isReady;
    GLuint m_vao;
    GLuint m_vbo[NUM_BUFFERS];
    unsigned int m_numVertices;
    unsigned int m_numIndices;
    unsigned int m_lineWidth;
    unsigned int m_pointSize;
};

}
}
}

#endif /* FNDRAWABLE_H_ */
