// Copyright (c) 2018 The Foundry Visionmongers Ltd. All Rights Reserved.

#ifndef _FNBASELOCATOR_H_
#define _FNBASELOCATOR_H_

#include <FnViewer/plugin/FnViewerDelegate.h>
#include <FnViewer/plugin/FnViewerDelegateComponent.h>
#include <FnViewer/plugin/FnViewport.h>
#include <FnViewer/plugin/FnViewportLayer.h>
#include <FnViewer/utils/FnDrawingHelpers.h>
#include <FnViewer/utils/FnGLShaderProgram.h>
#include <FnViewer/utils/FnImathHelpers.h>

#include <GL/glew.h>

#include <map>
#include <memory>
#include <set>
#include <string>
#include <unordered_set>
#include <vector>

namespace Foundry
{
namespace Katana
{
namespace ViewerUtils
{
class FnBaseLocatorVDC;
class FnBaseLocatorViewportLayer;

typedef std::map<std::string, std::set<std::string> > LocatorLocationsMap;

/**
 * \defgroup FnBaseLocator Locator Plugins
 * @{
 */

/***** Locator *****/

/**
 * Base class implementing a locator plug-in that is used to customize how an
 * object that's represented by a scene graph location is drawn.
 * Each locator plug-in will be instantiated once per viewport, and is used to
 * draw the geometry for multiple matching locations.
 */
class FnBaseLocator
{
public:
    /// Shared pointer type
    typedef std::shared_ptr<FnBaseLocator> Ptr;
    /// Weak pointer type
    typedef std::weak_ptr<FnBaseLocator> WeakPtr;

    FnBaseLocator();
    virtual ~FnBaseLocator();

    /// Called after the object has been created to initialize resources
    /// (called with a valid OpenGL context).
    virtual void setup() = 0;

    /// Allows the object to clean up resources
    /// (called with a valid OpenGL context).
    virtual void cleanup() = 0;

    /// Called immediately before a location is drawn
    virtual void locationSetup(const std::string& locationPath,
                               bool isPicking);

    /// Called immediately after a location is drawn
    virtual void locationCleanup(const std::string& locationPath,
                                 bool isPicking);

    /// Draws the object that's represented by the scene graph location with
    /// the given path using the given shader program.
    virtual void draw(const std::string& locationPath) = 0;

    /// Draws the object that's represented by the scene graph location with
    /// the given path using the given shader program as part of a picking
    /// pass.
    virtual void pickerDraw(const std::string& locationPath) = 0;

    /// Called at the beginning of each frame to allow batching of draws
    virtual void onFrameBegin(bool isPicking);

    /// Called at the end of each frame to allow batching of draws
    virtual void onFrameEnd(bool isPicking);

public:
    /// Returns the ViewerDelegateComponent for querying scene data.
    FnBaseLocatorVDC* getViewerDelegateComponent() const;

    /// Sets the ViewerDelegateComponent for querying scene data.
    void setViewerDelegateComponent(FnBaseLocatorVDC* vdc);

    /// Returns the ViewportLayer that owns this object
    FnBaseLocatorViewportLayer* getViewportLayer() const;

    /// Sets the ViewportLayer that owns this object
    void setViewportLayer(FnBaseLocatorViewportLayer* vdc);

    /// Returns the named attribute of the scene graph location with the given
    /// path, or all attributes if an attribute name is not specified.
    FnAttribute::Attribute getAttribute(const std::string& locationPath,
                                        const std::string& attrName = "");

    /// Returns true if the scene graph location with the given path is
    /// currently selected.
    bool isLocationSelected(const std::string& locationPath);

    /// Returns the default shading shader program
    GLShaderProgram* getDefaultShaderProgram() const;

    /// Sets the default shading shader program
    void setDefaultShaderProgram(GLShaderProgram* shaderProgram);
private:
    FnBaseLocatorVDC* m_viewerDelegateComponent;
    FnBaseLocatorViewportLayer* m_viewportLayer;
    GLShaderProgram* m_shader;
};


/***** Viewer Delegate Component *****/

/// @typedef LocatorCreateCallback
/// A callback function pointer type for \c FnBaseLocator::create() with the
/// following signature: <tt>FnBaseLocator* (*LocatorCreateCallback)()</tt>.
typedef FnBaseLocator* (*LocatorCreateCallback)();
/// @typedef LocatorMatchesCallback
/// A callback function pointer type for \c FnBaseLocator::matches() with the
/// following signature: <tt>bool (*LocatorMatchesCallback)(const
/// Foundry::Katana::ViewerAPI::ViewerLocationEvent& event)</tt>.
typedef bool (*LocatorMatchesCallback)(
    const Foundry::Katana::ViewerAPI::ViewerLocationEvent& event);
/// @typedef LocatorGetBoundsCallback
/// A callback function pointer type for \c FnBaseLocator::getBounds() with
/// the following signature: <tt>FnAttribute::DoubleAttribute
/// (*LocatorGetBoundsCallback)(const
/// Foundry::Katana::ViewerAPI::ViewerLocationEvent& event)</tt>.
typedef FnAttribute::DoubleAttribute (*LocatorGetBoundsCallback)(
    const Foundry::Katana::ViewerAPI::ViewerLocationEvent& event);
/// @typedef LocatorComputeExtentCallback
/// A callback function pointer type for \c FnBaseLocator::computeExtent() with
/// the following signature: <tt>FnAttribute::DoubleAttribute
/// (*LocatorComputeExtentCallback)(const
/// Foundry::Katana::ViewerAPI::ViewerLocationEvent& event)</tt>.
typedef FnAttribute::DoubleAttribute (*LocatorComputeExtentCallback)(
    const Foundry::Katana::ViewerAPI::ViewerLocationEvent& event);
/// @typedef OverridesBaseGeometryCallback
/// A callback function pointer type for
/// \c FnBaseLocator::overridesBaseGeometry() with the following signature:
/// <tt>bool (*OverridesBaseGeometryCallback)(const
/// Foundry::Katana::ViewerAPI::ViewerLocationEvent& event)</tt>.
typedef bool (*OverridesBaseGeometryCallback)(
    const Foundry::Katana::ViewerAPI::ViewerLocationEvent& event);


/**
 * A base implementation of a ViewerDelegateComponent which tracks/maintains a
 * tree of SceneNode objects and locator plug-ins associated with each location
 * represented by a SceneNode.
 */
class FnBaseLocatorVDC : public FnKat::ViewerAPI::ViewerDelegateComponent
{
    /// A container for various type of registered locator plug-ins.
    struct LocatorContainer
    {
        std::string name;
        LocatorCreateCallback create;
        LocatorMatchesCallback matches;
        OverridesBaseGeometryCallback overridesBaseGeometry;
        LocatorGetBoundsCallback getBounds;
        LocatorComputeExtentCallback computeExtent;
    };
    typedef std::vector<LocatorContainer*> LocatorContainerVector;

    /// Scene Nodes are used to build and track the scene graph.
    class SceneNode : public std::enable_shared_from_this<SceneNode>
    {
    public:
        enum class InheritableFlag {
            kFalse,
            kTrue,
            kNotSet,
        };

        /// Shared pointer type.
        typedef std::shared_ptr<SceneNode> Ptr;
        /// Weak pointer type.
        typedef std::weak_ptr<SceneNode> WeakPtr;

        explicit SceneNode(const std::string& locationPath);
        virtual ~SceneNode();

        /// Adds the given scene node under the given name as a child to this
        /// scene node.
        void addChild(const std::string& name, SceneNode::Ptr child);

        /// Removes the child scene node with the given name from this scene
        /// node.
        void removeChild(const std::string& name);

        /// Returns the child scene node with the given name.
        SceneNode::WeakPtr getChild(const std::string& name);

        /// Returns all child scene nodes that are stored in this scene node.
        std::map<std::string, SceneNode::Ptr> getChildren() const;

        /// Returns the parent scene node of this scene node.
        SceneNode::WeakPtr getParent() const;

        /// Sets the parent scene node of this scene node to the given scene
        /// node.
        void setParent(SceneNode::WeakPtr parent);

        /// Returns the path of the scene graph location represented by this
        /// scene node.
        std::string getLocationPath() const;

        /// Adds the name of a locator plug-in that matches this scene node.
        void addLocatorName(const std::string& locatorName);

        /// Removes the name of a locator plug-in that matches this scene node.
        void removeLocatorName(const std::string& locatorName);

        /// Clears the set of names of locator plug-ins that have been
        /// associated with this scene node.
        void clearLocatorNames();

        /// Returns true if any locator plug-in has been associated with this
        /// scene node.
        bool hasLocator() const;

        /// Returns the names of all locator plug-ins that have been associated
        /// with this scene node.
        std::set<std::string> getLocatorNames() const;

        /// Returns a list of descendant locations that are associated with
        /// locator plug-ins that have been associated with this scene node.
        void getDescendantLocatorLocations(
            LocatorLocationsMap& locatorLocations);

        /// Returns true if the location represented by this scene node is
        /// currently selected.
        bool isSelected() const;

        /// Sets the selection state of this scene node to the given state.
        void setSelected(bool selected);

        /// Returns true if the location represented by this scene node is
        /// enabled.
        bool isEnabled() const;

        /// Sets the local enabled state of this scene node to the given
        /// state.
        void setEnabled(bool enabled);

        /// Returns true if the location is hidden
        bool isHidden() const;

        /// Sets the hidden state of this node
        void setHidden(InheritableFlag hidden);

        /// Returns true if the location is pickable
        bool isPickable() const;

        /// Sets the pickable state of this node
        void setPickable(InheritableFlag pickable);
    private:
        std::string m_locationPath;
        SceneNode::WeakPtr m_parent;
        std::map<std::string, SceneNode::Ptr> m_children;
        std::set<std::string> m_locatorNames;
        bool m_selected;
        bool m_enabled;
        InheritableFlag m_hidden;
        InheritableFlag m_pickable;
    };

public:
    FnBaseLocatorVDC();
    virtual ~FnBaseLocatorVDC();

    static void flush();

    static FnKat::ViewerAPI::ViewerDelegateComponent* create()
    {
        return new FnBaseLocatorVDC();
    }

    /// Registers a locator class and its callbacks with this VDC.
    static void registerLocator(
        const std::string& name,
        LocatorCreateCallback create,
        LocatorMatchesCallback matches,
        OverridesBaseGeometryCallback overridesBaseGeometry,
        LocatorGetBoundsCallback getBounds,
        LocatorComputeExtentCallback computeExtent);

    /// Initializes this ViewerDelegateComponent.
    virtual void setup() {}

    /// Cleans up this ViewerDelegateComponent.
    virtual void cleanup() {}

    /// Handles viewer location events and maintains the SceneNode tree.
    virtual bool locationEvent(
        const Foundry::Katana::ViewerAPI::ViewerLocationEvent& event,
        bool locationHandled);

    /// Tracks the selection state of the scene graph.
    virtual void locationsSelected(
        const std::vector<std::string>& locationPaths);

    /// Returns private data.
    virtual void* getPrivateData(void* inputData);

    /// Gets the bounds of the given location.
    virtual FnAttribute::DoubleAttribute getBounds(
        const std::string& locationPath);

    /// Computes the extent of the given location.
    virtual FnAttribute::DoubleAttribute computeExtent(
        const std::string& locationPath);

public:
    /// Handles the removal of scene nodes from the selected path
    void removeLocation(const std::string& locationPath);

    /// Creates a new SceneNode instance based on the given viewer location
    /// event. Can be overridden if a custom SceneNode type is desired.
    virtual SceneNode::Ptr createSceneNode(
        const FnKat::ViewerAPI::ViewerLocationEvent& event);

    /// Marks all viewports attached to the ViewerDelegate as dirty.
    void dirtyAllViewports();

    /// Returns the top-level SceneNode.
    SceneNode::Ptr getRootSceneNode() const { return m_rootSceneNode; }

    /// Returns a map of locator plug-ins and the scene graph locations
    /// associated with them.
    std::weak_ptr<LocatorLocationsMap> getLocatorLocations();

    /// Returns the list of registered locator plug-in classes.
    static LocatorContainerVector getRegisteredLocators();

    /// Returns true if the scene graph location with the given path is
    /// currently selected.
    bool isLocationSelected(const std::string& locationPath);

    /// Returns true if the passed location is hidden
    bool isLocationHidden(const std::string& locationPath);

private:
    static LocatorContainerVector m_registeredLocators;
    SceneNode::Ptr m_rootSceneNode;
    std::shared_ptr<LocatorLocationsMap> m_locatorLocations;
    std::unordered_set<std::string> m_selectedLocations;
};


/***** Viewport Layer *****/

/**
 * Class implementing a ViewportLayer that owns instances of locator plug-ins
 * and is responsible for drawing them.
 */
class FnBaseLocatorViewportLayer : public FnKat::ViewerAPI::ViewportLayer
{
public:
    FnBaseLocatorViewportLayer();
    virtual ~FnBaseLocatorViewportLayer();

    static Foundry::Katana::ViewerAPI::ViewportLayer* create()
    {
        return new FnBaseLocatorViewportLayer();
    }

    virtual void freeze() {}

    virtual void thaw() {}

    /// Initializes the GL components of the ViewportLayer.
    virtual void setup();

    /// Cleans up the ViewportLayer resources.
    virtual void cleanup();

    /// Draws the viewport layer's contents.
    virtual void draw();

    /// Draws the viewport layer's contents for a selection picker pass.
    virtual void pickerDraw(
        unsigned int x,
        unsigned int y,
        unsigned int w,
        unsigned int h,
        const Foundry::Katana::ViewerAPI::PickedAttrsMap& ignoreAttrs);

    /// Returns true if the viewport layer picks objects when the pointer is
    /// hovering over them.
    virtual bool usesPickingOnHover() { return true; }

    /// Processes viewport resizing.
    virtual void resize(unsigned int width, unsigned int height);

    /// Sets a generic option.
    virtual void setOption(
        Foundry::Katana::ViewerAPI::OptionIdGenerator::value_type optionId,
        FnAttribute::Attribute attr);

    /// Creates all needed local instances of locator plug-ins.
    virtual void initializeLocators();

    /// Draws locator plug-ins for their associated locations, optionally
    /// ignoring locations with the given scene graph location paths.
    virtual void drawLocators(bool isPicking,
                              std::set<std::string>& ignoreLocations);

private:
    FnKat::ViewerAPI::ViewerDelegateWrapperPtr m_viewerDelegate;
    FnKat::ViewerAPI::ViewportWrapperPtr m_viewport;
    FnBaseLocatorVDC* m_viewerDelegateComponent;
    std::weak_ptr<LocatorLocationsMap>
        m_locatorLocations;  //* Deprecate & use the VDC function?
    std::map<std::string, FnBaseLocator::Ptr> m_locators;

    GLShaderProgram m_shader;
    std::string m_vdcName;
    bool m_initialized;

    std::vector<std::string> m_excludedLocationsPicking;
};

/** @} */

/// @cond FN_INTERNAL_DEV

#define REGISTER_LOCATOR(CLASS_NAME)                                          \
    FnKat::ViewerUtils::FnBaseLocatorVDC::registerLocator(                    \
        #CLASS_NAME, &CLASS_NAME::create, &CLASS_NAME::matches,               \
        &CLASS_NAME::overridesBaseGeometry, &CLASS_NAME::getBounds,           \
        &CLASS_NAME::computeExtent);

/// @endcond

}  // namespace ViewerUtils
}  // namespace Katana
}  // namespace Foundry

#endif
